# Laravel Dusk用 フォーム入力補完パッケージ

```
composer install wizkch/formtest
```

## チュートリアル：パッケージの利用方法

### 1. 既存のフォーム（ローカル / リモート）からインプットフィールドを検知

例）phpコンテナの /formページのform#formのインプットを探索しform_testというファイル名で保存する
※tests/form_seeds/ に保存される
※form_test.csv.sampleで保存されるので.sampleは削る

```
docker exec -it formtest php artisan test:pick_up_form_fields -c php -u form -e form -o form_test
```

オプションの説明は、以下で確認

```
docker exec -it formtest php artisan test:pick_up_form_fields -h
```

### 2. 検知結果のCSVにテストデータを入力する
form_test.csv.sampleには、すでに「正常系」としてデータ入力サンプルが一行登録されている

![CSV1](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAB6AAAACYCAMAAAB9AwdcAAAABGdBTUEAALGPC/xhBQAAAAFzUkdCAK7OHOkAAAMAUExURVxcXD2I/8/Pz87Ozis0NicsNdDQ0ABQvCQsLhkdHWJnaLGyssnJySYtLycuMLm5uqOlpYmLjFZYWJ6goCkwMnJ3eDA4OTg+QCw1N2Npar/AwDI5O7O0tczMzLa3t83NzXZ7fFNZWmdsbj9FR8bGxsjIyKmqq8vLzE1TVC82ODlAQsPDw8XFxS00NkJISsLCwrW2t5qbm21yc1BWWMrKyoSFhcHBwa2ur2FjY0hNT6ChoT1ERjtCRF5jZb29va+wsMTExU1UVaanqLe4uJCTlIGDg7q7ux8jI1BSUkRKTJyenhoeHlpfYVZcXY6Skr6/vzA5Oy0xMVxeXl1iY52fn66vrzQ7PampqpOVlo6QkbKzs6ytrXuAgXh8fSIlJVRaW7S1tUlQUX2CgrO0tIWJiTg/QSsyNFxhYicrK4+QkIuOj35/f2pub8fIyGRqa4iMjby8vbCxsmpvcKusrEVHR5iam5qdnXt8fJWWlqGjpIOHiHF0dEtRU1heX0ZMTm90dbm6un+AgJaYmXl9fjU+QDY9P2BlZmZrbHR5elJXWYeLi1aLxsfHxyUpKSwzNYuNjomLi5iZmZWXmD1DRUhOUDM8PkBHSX+DhMnKyp+iojw/PyMnJ3R4eXB1djg8PGxuboGFhqepqZGSk7K/zTM2Nmlra3d8fTY5OcDAwMDBwTAzM2lub7q6umtwcU9VVi41N4yQkKWnpyouLh0hIUFEREdKSs3OzkxPTzY8PoCEhUVLTWBmZ5KTkzpBQhwgIKOkpD5BQWZoaMfLz6q7zXZ4eIuMjXR1dWRlZRxvwS43OVJVVW9xcUpMTABVvZWZmkJISYeJiT5/xCksLDg/QFlbW2xxcgBSvV5gYDR5wwBjvwBavlyPx2iVyEBCQhJrwZmbnEOBxM/P0DE1NQFlv2uXyK29znScyVJUVJSuzABgvoCjycTK0EuFxQhowMvNzzp8w4upy5ygoMDHz1GJxmOTxyNxwpmxzC53w6K2zCp1w4SlyrnDzr3Fz6e5zW6ZyH2gygsvADsAACAASURBVHja7JxfTBP5FsfJkPsljSR7U54LPoA8lMSQFCU8gNFoDZNIStZoakIm0QeaxgQDpiR9WXxVsrCPkkBkJbjEF6lcXggKAhKQxCVEXXSNqKjsFTTIikbXvb+Z+c2UUjstd1tmas/3QX6202nPmXPO5/dvJutfJBKJRCKRLKcscgGJRCKRSARoEolEIpFIBGgSiUQikdIU0CQSiUQikawmAjSJRCKRSNYEtC2lysrKyThlpdin1lXmWk62k+0k8myyPUCAJkBTSpHtZDvZTp4lQBOgKaXIdrKdbCeRZwnQBGhKKbKdbCfbybMEaAI0AZoKNdlOtpPIswRoAjSlFNlOtpPt5FkCNAGaAE0pRbaT7WQ7eZYATYCmlCLbyXaynQBNgCZAE6Appch2sp1sJ88SoAnQlFJkO9lOtpPIswRoAjSlFNlOtpPt5FkCNAGaAE3lhGwn20nk2UwD9LtH5zMO0O5AwJ6Ci+JOg5TqDAQuJv8Hiu50KCcvA4GOTC2lqbnu3yRGUhQm7m/Us2kXWWkG6Fv4Oekoff38+Z9WBnQ90JbkC/L07rELwtXSKaunlB2oTfKvu9N8xiUcyhsRrW57GeBJwS8cHvP7/b2Zdt3L/WFNf1OATkGYhO7frRKKqsYvf4OATm5kDfoj1ESATj6g518BOJ9RgB4OuqCq2mPtlEp6oe7/iVuOvZ2ZCehp2fjDmQboSwhLIkAbqvUGd1TBQdFmuiwN6DFEaIQAnXRAf/yMjAM0CyvHkemutlzgWCijCnWIFWpnWVdX22ngaoe1y0lqAN1bnJGAvorsYk1VBGjD2TUJKKmeCOSxOAkQoA0VdIVVQIBOPqDvfVnAwocMA3RTNvKV4eNTb0pS0MKF2g8MKVz2NCZ/3SAdAC0eQ1FZ5gFadCCYThgxMUxExudzSre9qRC4Q4BOVG3ATQJ0cgG99BfwafLXDAN0KYTrauumgO8yqVAPX4CX73657IAzAwHdA/T0ZB6gWQqVE6AT0hTgE/Wr4CNAJzrx4MDZVHnAWoCej3rhXuKAntE/PW94HlVvgLfzORYAtHt2cz+2X0wc0B11+qfqNp9nNnoZqRIntCYbQreanFKtm3+yu34L6TSrTdEPe4Y3vRWqjzr6OjCmtdkQus/scjIbinUlE6i84RBx94sG59moviJcE60B6O287rZOYDB9Ab2tYXINDq0YDQ+1pyOgDd31TyIrVlYpGgcupsoD1gH0/IsPo1i59WhGB+7a+1Ws/PX3s0hAP1pc1A45v7i4JG/FXlycv3f+1QpW1xiHZ55/BlbfLilHPF5c/Jjz8e+f8Z/FtUhGvxl9zf41G9DXj3izUdh4W08fz6lduSi81tIaCegbUrN2RLN0QwkK6aytP9AoQJKDrrfbBaHBrw4Rb0tShzjSvAdFjXORX1e396g+r10NdJqZUvfPHgKuVIQn0pq6mS9qqrvEiHTySJK29dYtSX757w4paLt+9wCEG/JswOANAQUny9VPyV5pnajIhTNvU8p4Dh/u32C6x8xy4p6ocEL4oTu8VfZ+6SEI+c0vIyuvXZIG+CsDkiTfb3dRkgZt9iEnin3sALFHAor28w2ksls6WxqzhUO+2a98aTVcvTYLAHqbr7ttMGXVM9WA3u4wuW6FUfP/79m47uKRFe0uo8iKk1WyerPRnjIPWAbQD1b5drjVdfWF5Vv8hZV3EYD+A9AA/Rz4jf15CMy/UI99kTPzXm2NPpaPmATerC+or9yKGFc/VrhvLqDFiWxu4w4eVfZ9/IWGgQhAV6IxPPyrlP9UYIf7hHpslq33itrKV/p5QaA+wM8zFPO7dwN15qVU63Ft++M5tVcxvFPzxbW+jel0GfBr3QteQLJR1luiHFrYa7Pzj3XbuFdCfCNqQU9M03/BPjPLSW8Vt9Qxwbth7Zoz7tZtrLzlYZweVidqmwD7nHro0Tpbi9oS1A0qzC19PA6c0Td91EJeijUd0Nt/3e8D/WkJ6G0PkwlgIH0BHd9dPLKi3WUUWcZZpcgH4WLKPGAVQD9hDnh1/sE7htfRZYXPK8CHtTfv/mB4fRQX0Gt4+/rNo1Hg4Su59ZANoj9wQH8Z/bz25smXFZneUTIX0OMsmsZH7NP5gFfZv3SbeaF92j5RwdJpMB6g84cKTmXd9wlwzTU4ymprW4p5TWOAHsOJoH1ELoZzscbuCJ9y+1MqxPr0NQdra/0lWq+9G8htG5kLeIGS/niFulnaPWYf2c96JPcde8bs5ce13cnMK0fQ3mPv2gUUP43x5UGkZNtQorb3uljZ3Gm/7cvm5cH9HXCopba8zAnsF+NV3hZXVZd9ugFoDuBol72nEbjQz93idZSVz03UsHDaPEXXuhtVw+YD2oTrHoQgine6fuyeHrAARrYAnG0PkzZ50Ss0N15RutMesllAW/FsAu4yAnTMyDLMKvWbBZxLnQcsAuiZz1h5rbQYbW/Jf19pXP6TvbUUD9AqZp8xQkM5zfwnYFkFNN4rI+flUSwsWQvQ1wXUKFu2Qt8DB+WFDieKVKD2FODkcBxAw9XERwjIvcnP5+QAgl8JSRaLB2yxun3IMi+lplnSKOVilsW9XS0pB5QZd7ePm2xUqHFSLkwi6+DgzCw/XzP3iqBMWsnvfX31vv5gNqrdJpaTauCUUv8GBDj61ev1vbKk0XeCr5IZVV4cl3983Q7W8smnEUuBHu6WSsWFrVXRi21tEH63mQ9oE657GWpm89QhUGm/2RjZgrY/TKqRa/McUF3lvWkzX1vxbALuMgJ0zMgyzCpFQ6kbQFsH0I/YIJg3PyhoXWcD6g3bueIBmh/7Qm+dBx6ogB5d0g9etxSgxUv6rQyeQvzCiNoCaOtup9R0MgQ0n8o5o3+KBWm9GppVonZwjHlshvUbomkpNVuIC3yV/aWSCCL7oXx7uehVA96wUDdpCwD8U6FCnOFeKeNf4QrvhwtjqrGxErgSHDaxnAwyRvFmi3L7ZP9pOPlF8jjQ4I5TeXe3avN1ezq0uZAy7ha7fnBL5Jc2CeoxJgPajOv+E05fkRe9G+RJyqdpA2gTwsQLafYkiveeO+MACizwHOwteDYRdxkCOlZkGWWVok4BzSn0gEUA/Qn/1VaIlx48WFIwPKm9uYiFe3EA/YQ/uRPgC9bramuSw11e5NbfswignwLXtPbFqSkWTQ3Yp+G0rwD/jgdoXm269V0wfrUV5H1lNVi/2rsbKIbTxCeJlW/YUn1nalDxRZ5+DvVNo0LdwF+p0ReTG1HIvaLZu1d9JULXlNGB745oYjnxhe8wdU9N3VScEdwwy3gnTuX18Yk1aBNrzC3tqltK+MEd2HTbR8iLyg4LANqM635MfuiG/HzTi2wIVJE2gDYhTC7AuxdtcpzMNmsLDukC6ETcZQTomJFlkFWqzkHoTKEHrAHomQV9vKyPo0f19guVw0aAntRXsh+orUmVvJPaRHlOznK4aQ1AD4bHyzyyBJTq/9mBmjiAdnDMjAP8wVhB9X75YPjGkp6v32PSW4LiARNTKqCPm8K+6NLnoNX9GUaFWivqP+Aob+XhtOoVQZu9bgeiFtKm29qa9xazzraJnZNLcEU5Q59PZEbfjlN5ednpU1dFZBWoYZONY9ppSnBp88yy3WYBQJtx3Rv0qSZ3fuwtGZYDtAlhIj8EWOs/saJyPJ0AnYi7jAAdM7IMskrRTaRwAG0ZQC9H7eBaVVeitenq9TiAfhwb0BqAf7MaoHv0yRNtskSfp7PJjxSRJ6eNAF1kiw1oLTZvfxXQfQ18xc6slGId9Mh7FkY2+iJXMdaoUGv9mKt6OnF3hL0iTyx8faeLhzmsoc40253wxnbG76q1RpW3K6ryZmuVV+/e1USWEk+RVoBMBrQZ190+p2cAc+KP6QJoE8JEAk5qrmt1wmX607i34NlE3GUE6JiRFdtdWp1O4QDaMoBeBx5GDqmxYUj9q7rz6x8C+pnVAO3XFj421I/w7uI2oDdFgJ6VUvXo2ERTahcKDHyRjz1JKNRDgDv2fJhpjzl1R82zbnSGR70x7p9W3v+xd/ZBUdxnHKfHzOOdcGBOxDHk9jRFEIGIL5jSUUk0p9QTHKKQ+HbVqAQzsUkqqERrNQ7TtBERp5PEM8JYcfAtL7yldlKxvgRnECdqU9/S0mIbNcEkTlJtY8a0+757d3veHrd7u3f3PH+4sCxy3+/v9zyfffnt77dPOO1nipW1SReA1rjdW2O1fHMhoNCimwwWjjWZKun6Ey6AlmVXsID2sIuOtvu8yRrRV9B9cE9YCJrBbqQB2usK2iVOkDp6oi81AL1zgee99ZCn1CvMaDYhGgCE9S+zIF/dQk0mbL5m2rMgzn1HJd9+fBdQtvJ2k0Wknok1AHtdrpYobXfTNOa2ZThcQYe+m1DXBKfEt4gtJo0jAGfl2KUGoAeregGtn2fQwlguNu5BH//198wrUwKgr7g/nA5TQBeInr9xhUiYkaaEfmVKADRPlJwgAT3jqGikjkYpRV44/c3TC35+iZ3MUx2hUHMVpkWxQj0R0h1aaU8TPiIT4oF8XYwPQuXl8PVxEJV3L3jEyWht90yfbx3qDtCh7yZU3RBO3HM97+/pG9By7BIA7WFX/wFdoe4FtJ5GcX/NXRZfuHKF/PIuwD+FAWO9ZwRAXxN+ciesAU2WoLk8NcvLHVQPmMI9AyqfQPcGDtBxMIa7TWcLDtAddWquJCd/NC9fCnaW19CDTZ8WFe01QjqVCz9pD6ZQ55as4r/tmEDfTNVG+wrRQKnycifdRsdEBPu9UHktwk9ORgSgNWh318iR3JW46TyoMSejaqO4Q9xNqEe1Am2miq5AwwDQcuzipvr0sqv/gK4CQ5O6DugE0NeEh9Akmm/TqP2cW7YZ4I7oPehvAM6xL2QdD2tAO17iO5VzGsQ7adT+SLjDdFIE6Dp+9miyewUDaOpVfLvmKVU8Acaww7RcVjhKIpMQevpUZsZBbmr7VMhx8Cf1/S/UnQDcnLxUxa/UTDupayb75TG6kRttcIg9L6tZCVk1QuV18ZNuOPKDqLyNFj5qSQhaLE1R1O6OVEhsFfqAvpdY1bSbUAOeuLEK1HQfh7TmcyDOyrGL7VnedvUb0N3chKARD+grfdD3LYfM75h73L3My82XP4XeHhGgSV7fo9e9OHMLwhrQ1Dsm85l3mc8y567F8bCPQXbBSJjtFAGa5PUypsRNDw7QmwDO6iClSD3b6Urd+hJzPtsFsJt+NOroZC9zuEL9FPduTHdsMIW6DWA2u4RV+Ubxw7aQa5/KXUY2xjMDcexkm9C1xFnJ3vFlK6/TBukHOLQEU3n5aNB4JrHQtzs1nHcZe04wAVI1XSQloNCgm5DoSmPGCBQTABfDCdBy7GJ7lrdd/QZ0s8oX0Dqai/s1gN7vL/XcvHUcer+iZ/gkzbt77otvP+kD+J/bcpMnAE78o+f2zT1f3gpvQFOvUmWu627rbAZ4rIm95WJdc73ekmtg84MDNHnpAXUWV0HnFiKoQWLkj9LX8NGlWUo5ZwPkH2sr2LQYgLn1fJQ88Wjodl2cCexpvEWYoC92U8GfLNsMY1cGU6jtlNlXOxwH9k4EWKvdLGrUjKxQ96Qr5sP97Gt1NY+RlDrVVrF3KUBOq6jyUtvMhva2iz+G30UGoDVo95Z9AFXtOx1N66yqnJepBWgtuskKgH1PXnU0HcsC0ZQMYQFoGXZZhBlS3e3qL6DJK6witR3QzWpW33zJPiDr/cRjeSu4dsYN0LfZI3sv3QxzQDs/5CSuZKbBcXTa2B3xTCnh14Nex+7f37g2GEC7P4tcpl1KXX2K+xAEM6a45o/cjo0VJnE6meZyqx3NyAymUFOTEQPYrNS/g5xalpPu2ZzUSmbO0XpuIR6oKjaJK+/5Kez+TfWRAWgN2t3UvpJe44hazynXoTFGAgkNuomDXgTPQP8J7ScSC8xZ/3ZxPcvLrv4COk3tC2hdrQfdc4Mi8vs3vuBve9/9jlo28r//cV8PmkTtnV5y/42vBoQ7oE2mj1elUzTO5d+pbzuaSs0ZPJZteB7QpoJDFJ7XukyRAWhT68nFVCHY18DPKnGRmuMLDp3daXIv1I7TlCU522pMwRVqU1ccI3uaReNycn4FXSLy+TsYrWtySDNs808LT12ZmlKzgpoWeVWnI1IArUW7t2wzMO3erj1GAgktuollOm1V1inNZykJ2Fm/dvE9y9OufgL6uvrrZ+sJ0NSor9uXL7g/mr59Wfq1rJ6eCwP0G4F0rI769mL3ZDjQLn3yWl7RaNJ7BJRSzra2Gg8vKqRHjha3n1fi0znqrzec7mrq0IH28vb6GR5muCQnPutoqnCasN2Di5buY+u6Gh060B5gaNBNWq6ftqiVImo7G2FZpTtAR0rExJiiNKJXOWpH7RjoLAIaAY0phdpRO2pHZxHQCGgENKYUakftqB2dRUAjoDGlUDtqR+0Y6CwCGgGNKYXaUTtqR2cR0AhoTCnUjtpROwY6i4BGQGNKoXbUjtrRWQQ0AhoBjYUataN2DHQWAY2AxpRC7agdtaOzCGgENAIaUwq1o3bUjs4ioBHQmFKoHbWjdgx0FgGNgMaUQu2oHbWjswhoBDSmFGpH7agdA52VDWh14wdRGTEYGBgYGBhBBQIaAY2BgYGBoU9Aq3u7NzoBbYzSiIkxGlE7akftGOisEg4goBHQmFKoHbWjdnQ2WgEdXaYioFE7akftGOgsAhoBjSmF2lE7akdnEdAIaAQ0phRqR+2oHZ1FQCOgMaVQO2pH7egsAjqKAP3OwoWHhe+SNq82I6AxpVA7akft6CwCWu2YNGdEBrUd9dwLpZIHvA2wnPs62Wg+CsMR0JhSqB21o3Z0FgGtdswCeITdZkgeMAwghf1ySPosY+n+2N8ioPWWUrsGDaM2v7Db31Xu471eN28ouSmz25+NsHIy2m5/XGRb9JZSZTuMzrVnLJu7Xr4vVFSbme3P/R3/S/sQBczv1/+CgJYH6M/++q9/Kw3oJbWbX33LI4afVbZajgCopnsLQJLkAZMFcr8+MvWIsRAWKHmTGwGtQCwBiKVa8QGAocp9vCqA58hNCsC8CIPUKIAhItuiF9DKdhida38VIEe+L1TkJTDbNH/HPwSJCpjfr/8FAS0L0Of6AI4rDehnQSpiCxUGdDILaLM/QBsPwm6qbv8sDAH9SN5Hqhyri2L1OIDhiOL1thmgKDSAHp2Xl6ABoDnbENAyIiEvb3notSfn5SUrJXYzQKJZti/zamtrl5vJf2q3RAugA2xiqawNJJMVL7O+AX3mc4qcigO6cH5z1Q6PSARYoqSoQoAyajsLDEa/gDYvgMnGn6yeFIaAToSHVDlWH4V6c8nbyl8QbaiqnBQaQM8THqOEEtCcbQhoGTEU4IHQax8IMEcpsSlFu38t35dsUTmIEkAH2MRSWRtIJiteZn0CumcPwIk+5QEtFTtgzH3OArOHJwUoahZ75TwLbP4BbVwv+yYRAlqDQq3KHcsIBrRGgYDWBNCB+YKAjhhAv/Y+wLUze0IC6AwbvHefexSrYenWwETNAavb9v6ANo77MwIaAY2ARkAjoBHQYQLoT+HrSwMG7IG+EAB6GMAInz9MqoPYDwIUNRCyWA5PkANoxUNScYJHIydUez3YyOASyFy69f6/LNkbzKVmPQI622tMrfBByw67f+Sth83y6q2EVp/umQ9v1SWgs1P8i+Ijudrsy6OEZJ+ATgrZOYJXu5sPe9/4SpHY5ZkF5tHZfu1I8swdcX64dRjvxClLCQGgy1KMvrugJ6C9EiTBOwuMpd6PQatT/Pnr5pMsQJcmeaLVXGr0V4uSs/0D2uuXMrYq4qxkLxF1Ks8mlqik4l1yAJ2SJLvMStQzIZG92l3iaN+AvvPZgFABeiZk+nwIn5IG1oCHeP8FptDbDyDeJ6DVHOnqrriImGtMXjToTcgctIHbN+k3UwwQO/2J5cIh1fYSA0yjXpH54fZ0MEzJLeMOPvJevhXGVD3v/kcWEYQNrARBLGI67jO7t8DItDfo3yosKVnLtvXDJSXjPI8NYaEe/8o0g2Hj38cz320giBHGyXWZkP406b95HAEQnzaKvwZcnQmQnrOZzdxVxFofgHbT6se9jLdyrAATq9iXUQYSxPoQAHoEQaQCJBLkJ3t3KbGDLwWPEs1U0SLbIWnRiwAvjh3vS5Rb9/no5TgDlEp4RCb1H1alw+IX1vOAZm2jjt2+OBb2V+0KyVg193Y37xqUCob8h43GKqKW3Xdw8INgiys6IuoKhUWJYMtZmMTvSTi4OoseZu/TDrLtXp4eS+bOM6U+8kPoMN6Js56yZEvJ87Qlc4lEgFSC+iCK9nm3P+LZBTMI4kGA2eRfzZBIEOqAZY+mg23HO8ZSgvgV2zUWNmeCNe4JbvwR1X9++n/mru4nimSLJz3JuRWgGwyDBAcYP0ZGHXFYQAyRGT9QEgwfDzAsV4dwRRaSySC6BAJExKwRkBAJYpRNJEHeyGZceXMh5D75fL0x++y/cuvjVHdPV88oNzrY2axaVXR3/eqc3+/Uqaqm4x5UErJaWTnriq+K0xcF2vt4+zJoLc9XLWkdWigOQGDbOoSqQlrWHaWdK3rqAN/oqKwswbvox1fCEC62zbPm6Q9phZ/bBJPFYjK0/BCLPdgHsuRT4sIaLFVNH85gVI4hdry9k5ltXktcPFlBuXQsVoXSvROLbTpp1uGr6Y6sjLvi2dkF+r/8/zkR6MPBzBnui4UQ3P8G7z4MY/JgKaNAFyRacyTQRVDufSM2q/trUTU13L0ezJNNhi6JkjxyokH8rRDdaTiAjW8IMzQ+dC7q5qkJDBDP1eC/atgJcL0I4LiIskegpdrRNndEbcSxo1rcwFleV0SU5D8jg1gnNjXpm/Ilj/BD7CQkOES6fF3s1hVx1/S+Zkev8XJ612sBnuZAoLvkU6khFgOckOEKwDT9wwOHvMV4fEFSk7NTJNUyUYq9u1LAKnpcMCJNN9GQ+qVAI2zEGPBg26q9XAv0Yewd7W0hFIuiG1gU6JIZ+a4ZfMNyH5ZEeKNENjhI+0msGbnt7h+mRjgriNEnISlf5dqBV9c3tfn0hzhN0GeeXPG5OAjteDNSxXCpTIns/EuyRa2Iuqn9lLB/U4E+BzDgiq+Kk7tAW271Ek0J/H1SoI1utNJ14o41OTwuuzPhtYNvPKcepou76AlsMop99Ml3hXdMjnwNUPGSV8wAjO8DWRI5iwXROnejSh9i59s7mdnutZJm08qcKD8G6BXkdQH8D9NpVvFVuyMr46569hcE+h8o0DlYgx4A2MlQdeUyhB/s32EG0frKIJBRoBlx7XpzI9CFz2FidrjkAnUxvll8WIPgTyXt8ysaFKxikxX/ct78JK3YjfpHy/IGgwAbQsyp9STK2heo497n/skOqtGg5VQyGYbOZDLJZqDn6gGKU/21VJjXmIN8onEvD9QmQLviaJtDou4FCMWHu16NAHxEBxqszy8ZXqBB9+frQP82G6MM8l4MGjR0JNtTt6h4e10EmtJHTFCYo6/Z0NujLrkdb9udXgOYy6FANyWTVZRfk8kINzaZuFgGaOUE+24CLsQjC0zH+tw71eMHOC96Fy2H5o6+jT9dMPJR8o8O5qU2g8EBh0BThlz7Z2r3L6p0gZ7cCrROw6WlxPGywSjQ/7hAe6n53nndfneTElMbmsJ0wdbA7t2PdKzeYckU+Luvx+eywXFuBOpX5l+kVgDOPnH1D2kwSgW5BlBxLRU5SmdVNRTNSHKBWgf1iqZvafOOhzhN0JtMdtAYhD7V6+IgpJTKauH53VRHvX8BB7SVTr+pscxOUuV5hQJ9NQhVi31HLYFW8FVwyiDQplu9pJDnD0T+6PVgcF8O0Q6YKBle/5kC+dKdi7yVVFE+lpWdD6BUIfhMn9/peJcP0LwxV8vu+1wkBFqY0ZYdH6VTxjFd6HIV+7M6Clu+r0eWJFlc8Lqrj1qJ1u5qVGlDrLy9k5ntXitpNq3MibJBHbGf3WkW2Mco02hW8VW7IyvjrrT+cQS6pwC63WuGXvnhfnr0vzewnvrV5UrNPraddqOoi0QKhDIJtL/rDh2A8EBPLgQaNJ6lNzbFIVzyhk31eL5ETBdYk/pTIqSgjNTIszkaXOYLFWEI8wxT9QRAmUjcAww7Fjx+B42nw/Sj+Ig8gBXC97OfP7g16NvUAbn4NlWCf1U4EIwz+3vGQtteFjzqVKRY9Fq9BkuCBSYBdlWB1ikV5Lv3NQt6cSR/0hqE5hwKtH3lyhvAFydGWPyNxfgbPHKeD8KIz7VTn2ijRewdlIgwW8VoGuAq//kXnZAu0BTqRzxHNNT7Xfqare/U+Jp50sDHXp4LNOXJBCeep/XQbKApbB/G2ISf3GYl919msGcTjkNiADmllbj6BxqMWnFbgyN8NmlM4c2+wxq08hDFBG1r0IqDkHcA3Vx8dlrkgFIM/+LjXxeGiia0n7OYV5QCreDrxCmDQNvdChL8MW818PeI2WdwhrehP/mTOxfRx45xCu3ZEqQkwNcnpT7zOew4T9vWhUB7KBKcMMVH/swlJELa65R41fZ9IHs6BAURXjXrhxp3o7INsfr2KjPb1pstmrXKFJQvrkEL7dynkzBmpNOs6qs2R1bGXW394wj0YqZD0G33KIc7lqBuQ8bLdpMYjGLAFzbLdhwz6Gp9Zoz+VEH8zxwI9CjGIvVwiRl8ZE6uem3BGDbBSRYNGxYkR3GzoBOjCGaTQhBl1mFMjlxzWMMxgGXzaRpfqqKR8wyp7oQ73oMT6CpYw/jqCsBvwoFCIrtE49OwQP6F8LaeuTkcohM4UXDMoNuba4659zULenVzEUMGbdrQwQg0Scgcdx3AOhJssZU+2nAfwIXQ9h7ahhQRBaPSIARwt+PbdIHWY6ZH6DWgteZy3I1HZk6MkiLvqi8AN3EgHgP8Kkyh/owomWamyktG53xXlAAAIABJREFU9jLas4TjkvRpb9ndOlf/QINRKvQxczPqpyUIG99FoNWHKCZoE2jFQS56YAsd9gUOKJ1eTuHN5wA+o/3IrDwKtIqvE6dMM2jpVu3Wt8UGRchcLrPn5PRZaCFukPYswQjmutuEiHPwdRoTJnS5ki2kTPSb2cL7kxDCF/nFD1EGSfUjGCklLzRlrTMrstOmtzMXq3U1KtsQq5aiMLPday2aNctUlNlvdZgmpBvWVh00q/KZzZGVcVdb/zAC3VQBv7uV7/Hlj2pH6e2K0JFOl2srVGET6BGoxQnqlkkanSXpAs14re7f9BGhEu93F2jJj/edi+JXRRKeNmmSTiwbbwCweDPKVprEdV7W6cRhDdcAZCqAOhrfNeI7AtHqRfDvkAMT6D3MD4iYKSAcqNe0wwnkcIAbaXcwNFHlEGjZa7WvWdCzrsei6iAE+oEMIJbB04MEK1/Od5JvaHQZQNnfImsJW8GI0kNcFm2nCXQTQJFtQhvP5bi3gpUTWxekPI8rDGy8xaufMg2Acfu6KPmNZLRnCcdVgLf256r+gQajVFBIfpYlD9vaqr+LQLs8xGmClkCrDjKLIRy7bogBpWInV/nodNfj5fbTQNIFWsXXiVPGTWK6uRolv3gy1NbWKKT1vRUled0gnbdZ1rG2fgRfv2bpM7uL/Da8Ucg/IDUP1u8molNnHj2c0uCGcZ9tl9kHslE4K9vv+dn3IF2MyjbEqqW4MLN9x7bu9GQVZaJPgaeRzjb+zkyzks8sR1bHXW39owj0s3xYcvkU28VEEOp7VYH+qqtJ2mYKOm0ktaoINCGNn6lEdx7/xvtcnQKtyTzAhPPj4EVin3kRVKA1bJoHwI5yFvdqmHsRGdz0De2mNYyJCJcHpZi9Y8x2VbNc4QAE+pjlm6SDf331lOmbZwAGscrvXOLwiPmlU6BJpr5mRs929QE0HpBAk2aRRTTCGIt6LLtE7nMbQLNBhctpI4ERJRpzg+1smkD32w5d+eTiX47GfVhqKr3+IwR6wPbr41rgpuDS6zK5JSYNlnFkg4OqXDhuHcFw8Q9hMGpFvzXhkte3F2iXhzhN0BJo1UEWLckgf4shHLMdRTkkON5jBV8o0Cq+DpzIl3ZxjylHUsst6ejgp15USF8p2U/6kF9o62Xduot1onWU/xKjVwgDu7oAxJbq6wDbfLvM1yM7pJl5KPaYLVejsobYxVJcmDnrMSsVZUKeXIaaIzClZ6FZ5DPLkdVxV1v/IAKtU2CUr2AbxyY08HQ8OfF/CvQf4BF9rsXjVjxQmSQuAk3n5CzR3fxtD105Bdr0sOdSoFfnp7uryjsLQAp0gTV3OG13Z0pwW/l40QD0tbtAh+CkbJQPMiOxTDs2ZhygQFOGKZRv1cClxDqru2cJtGmRD2oTU7cqG4KQVaDVvmZGj6Wz5uLjb/LvBbQDFOg+wat1MgD3WHMB5qAXMwygYj4KRhPWTJNNGWwCXWvm8+i1hvuAcjTur3Gfko1FxwHM/hXAGkk7tn3CFOgSksWeUZFjbO/rpcEZrwwAnP4hDEat+B9zZ/cTRbIF8KQnOakMTg8JAwRncURBxJEPwYUQAfFjSSAiDwJyRaPyZWJEcc01XgNoJMpqDMYL4WKyJOqbMaK+eTFkn3x2k43P+6/cqu766j413QMXZ5wXErpmqurUOedXH6dOe0TyvQBtqMSvggrQ2EBeaDjjc6w0XJK/VOzu22o+nAMay9cnp1BAd2rVCObFtWPIhyaR/o1wRiuhK8X6hAnzTDFo8w9pX1oSobAJOs4iXCZLydbJDWriXJWoNSmVGmKDphg8cyCgsZRdlkDlQ5Ob9fkzVRsed1z6BwH0eX6qoiG7/XWaalbPV0fGWwJ0DxwXYzolR3el1AxoQj43O9sjuQD0A1cNBoci8ug8BNC93oP2J0ZAT3gLneO7MJNq1yovgH7pbdaVYEC3vlJFAwBt6GsAoO3zT1XZvAF6w3L2uL9AZYL3+IEsNsIaZB5ADGi/jAa0XLatHkBT2e/Xxr4+x+N+TJscsIa2eDtY4velv3gAHSQOkpjb6V50efKeGO3DVRj8wCOS7wVoXAlSQQVobCAtkIpp8Cp2ZCGXy+yb74yAxvL1ySkM0J5qEKAzibQF5VLmN41ea7/yk978b94vlbqxrMSJoThcuynJ9qrbEY61LZuUSg2xQVOwZw4GtEHK7tnSiMHNIn+masPjjkv/EICOzQGc9WciGKITsJ8XuYy3Aug6S+ywrcJ9vkxOgSdFhzeTWHJoPaeAtvdRLh86/+1z7+XhMEDPAtyOqs+SeQU9Cbu0QjOEb/ADvIrlEdCrVHO1Zo0HAvovOi3rOLn7WntFa/AWN+5rAKCpKViFL++VzRy7kUdAUxumkIk1ixiYCHyUxa4CfM0wgEh9kIyKtP2xXg+gV7VVLKn0Me47j/s7LRA16fqbfrDKtA4mggEdJA7mNtZGXlWzy6+DRvtwFQY/uIcvPG8/oHElSAUVoLGBPNcyKJ1x5fFU3AFwf/2KEdBYvj45ha6gK5F/QIDGIr1tWkGXv+0AHgTk/Eq97hCusLYq57su1xy054ZIiSDJzioX4rSkJBjQBk3ZLKCNUma34NKDyM1if6Zqw+OOS/8IgGbh7gMoiPrA1ce2lPFWAN0P6fdiR4inmeuAqURmQG/7JwTQ83RakhSeOwTQ74POD6U2HFXRcPKzuAPiWshJHgA9Iw6Y5CcA0HReNsbv7AcDGvc1s/TaLKivk2eX+QP0dbbHvSYXlhF1xMpeUD1hHkCsPkhGf2opBN55AD2jnQPXoG2q7zvuFdrS5rIL6Lso0i0Q0AHiEL7j3pQTU2awD1dh8INenAt1+wGNKsEqqACNDWREWybye9BHtT3iMecQ1wDou6ZIQl1OoYBuQVkXEaCxSF9qMRCy5AyZrYbUmjqDXtCaP+u0dV07l3FR/giot0otbUaytaCFUxU41yqDAG3QlM0C2iTlxDnoSKkwV+lmsT9TteFxx6V/AEAfaQBoCcoVvDVAT6uT2t/duL3lOKRbyY8D6C8qgKkvDNCknl+eDAT0cxVwKZcZ+2BnzT6o3ps/QC9rwYqhgG6ASr7aXww+g8Z9zSy9dmWwd/IJ6NoViqwb0BUTPbZExOJEGnaaB9CgPkhG97SlynMPoPdqOZl6VehMTsZ9Q92sYUfsRb5FdRaADhCHmoBaTpYDbB9cYdCDRXVfiUwkk9/nmhWqBKugAjQ2kOsgI5nsE+43HzhbLO5nwI1qwoDG8vXLKQzQWrA4SSZrTYDGIt2jBWTXJEtkyTKA+nEJ6F7VfKBLp19VzDOLcnQSfTzshKLBg9CQ2IRkSTP0ifLJwzAcAmiDpmwW0CYp/wesyyMq6Eu6WezPVG143HHpvAN6Yu4wwJPAG05bArR9Ft6IXfPdzoskZyZhp+8t2vkF9N8Af8ljpjBAD6nQ7e5o1Ou2Toho4Hl1Oj0Yja7z+fcZ8jUFw7avbO4ctX0fDosXhVVEo3YgoDvFcQQLJQkCNO5rZuk1Ssup2ZlrQF/V7w0MwblYs9zCiwCcVLtdr80DaFAfJKOvlnuhkxm95QG0HQdLzM32oRs333fc2SYmzyJdx698H0lBAVdE+1H0YhigM4sjOTOzLJfZKzGTfXCFQQ9iozLmuPY+rLB13Sd57W+7+o4qwSrIUmDsyWAgC82Q2qtOqIpdZt9RWxPHiRHQSL5ITmGAblf845f3MKCRSMer5ZXm9XInkoeX/BPg6IQA9DD/zkVwsj4spmCKt7WkCiqdM5oXsOMAw/3dTUiWWfsetfMwZwa0GmKsKQZAe6zWb8lYi8l0hPJ/Ig6VpT43i/2Zqg2POy6dZ0DHHnUAVIa8BWMrgLafa4uK3TBKposAuvy5/PIL6Dmx8/ypKxzQ1AY6XPVYegpppvSxuaO7xS9bbs9i5yDizlOTo+6uU125c3lpRE5xZdkcOmpm9O4IXnPVLgDQH8W2wrFyM6Bni3rWzX3NLL01cSM3cTvnQWIXJKVc13pN3tZmgI64ruVYJ5RvGDtF2m89S/rUB8uILspOO47l4YAvk9gpkeOJBXr05Hbcuy2IrDo5HCureENvUJA4mGBJLIrDAJ1ZHMkIHHTF2FbthDVh++AKgx/QQWhwv/uBxybZKbhvb2/f/ZVgFWTJKU5mMBCmn2l2Mzh2wZp05RG7ybOqkSMneCouDGgkXySnTIAWZsWmca6rWFxxAq4MgMYipf7ltrODnRh149dEqs/jgossk5jbsiNxfumZ0vuDI/Xafu4J3zptjp1FF62CJEtKV6DZRXbvDti5YAa0GmLcegOgldUqN6v+h7R4IQ59752m/eFzs9hWtdrQuOPSeQV02xzLYzcQBowtANqmHqtIBkbNQTMLjtuHLlHlF9CzKUgNPRqf3v20siAU0OzaYNVv+9vO/NzJjxVZOKLMDdQx0rTXnZxaNy5uVMw1uAcz9k2oYntMiTiUd/vK5s5Rk1sA8fn1r/OnLaheCgY080xjFeOPr1qjB42AHgYebO/va4D0Fug0sPDd4vp83Dqba0CfAZg83SuyeXWAytREe9zQAXc+1516mRIbcKhTJYd50IymPlhGdDUCw9da14rfwDNfLm7qIvuKm2bnXwCkjuQW0MxvQ2Vh0RtId3OYJJ9SH3ZmcfreH1QPa8MAHSAOirldjd0TTSN9YJ0x2odQGPSAxdR17m6avkB14eBegY+Be2uD29l3XyVYBcmgBVZ/tMk2GAhJfGRRXf2vJqFlhsujif7n39HuixcqBdgNgEbyRXLKAGhpVpctOpEr646erAf4YjyDNoi09g1Awbel3rFdVJDa7IiF/bjs/QkK9kHh/HoFa747Uyg5SCePvy7tb2zgZyFtVXCU/V2mMF3IWrLOSqf8l1N17V8sflnXAGhtiFHrDYBWVqvcrPofkvINviofcqcnmpvFtqpfx/CPOy6dT0A3QnbZQZY3DWg2sTmh0PvEqagM53n47L8dnlNAk2IeT5+e/Uc4oO0R8eorPsOdB3EUcmBFvqapXbyoBfoTrod0F+mPRfJ1rWzuHHWyX7RqpSLkDDomiraUxE2Ajk2BOEPy9TVIetMp/jqesmiuAW0XyHfguM4B3ipAF7V1iPf5JMydarN4DmdNfbCMWH5m97Na6gN0yXPxg137c7xzQsgj57U9LDxKwKRbvJ4K4gdCz6ADxFE7LB5YroYj+5AvVPI/ILXylUFVTXIWtd1vs/JXglRQ3EVKGgyEtprfwRmukfI4lZZ3gxKZAI3ki+SUIRe3MqvHb6TI7QyAxiJtk9XEP3kOpFqrIfLY+ZXCWnGF6AV3ul8LxJc+Oqb9CiKtYmL3JXvJEvtCSkjvLckEaDXEqPUGQCurVW5Ws2SflCvE5tRgJ+xo87hZbKs6oP3jbrDsPALaHoYTjVnk11ze/Dr3W6qr1HNcORA1zQP2aAke8gBo0sve1JHuWSZZAJo64VvM3UUO8SCOiULrXwmxPzUpTus2ehwzHnYS/16OwIAtT15G/GVz56hJ4yizocmxDRICaGI3MtZ0fXhPjIAmn9P1IlGBp6/B0lsuokZZffwUyTmgyadntJWjSperazRAk7YHfdR3ju6RxX2dInerLy377RrJiJDxfoqyw4VlxA9oQq7fZJdspj7UkJwDmpRG7/bMNU1oMJl40uW8GLD4PckC0BnFQSa+xZ33A/5Tptzy2YcKWvA9YEo0wESyckNa/5Wj1f78fP93332V+FWQLpOLpywReOQ1EAcC1z4cWp21NXl8esCuNK8M93otxgNov3yxnMwraM2skkN9zrVzfj/PAGiDSO3zuxj2mlcTvpJl1MNtOIAmC2OjtEx8Tq6UEv89Qf+RuvS7859i6QsScZlvOxvJUkd3nF0yTxfxnUEToPUh9rXe4JmV1WpuVrNkj5RrOqCqVBLFvdIq3SyyVW/GId+4Gyw7G0Bvywdvcc9ejGWj97NZxHKizWvP26/KMlxw/s2y6nIGaOMnOb0cy/4HY3ub/sfe2fW0kZ1xfDW2/qIEDGoS4kC8YPOOU0NtK1ECGIypK3YxQWJJCFhKyELYpuUlQUENNeAmoUlQFqkhkN0qtCRqkWgEyWq1tAmlV92LaCsuusp1PkO+QefVHpsxNmAw9jzPhWfGnpfzP+c5z++cM2fGM7IWjfzRtPJ/Bs4zabOINfDI9LTULtFNT19V2He/AjWblvGyH2O7z5f548pWb3Q7Im/TBbVGs+k33UfS9sc2ab86KV26IvBGhmCAXfeF/Tl7qKiXMeZR5kxEgTqLrzxR2kUR8odbyp1n8rZxyojZkTduq9FtUT+2+kFncU6G1ILM83lx1x52EQUX1E3+LFoFCXkYJ89ZHb3uhudvaD5FmCQWUq3yVixXtxeLOHndj6OOR+Y5w3aZ7h6XznJ1elrSxgauI9vJWa4WrUR9DaSsiCN6ikKtfalYk2W5rJOldjooQgqz0eJZaLlH2DshgI7R5hobj6Ulo+1YcQpYhECtdu13pBtaYT2glNTeOxOYCbsS/npa8vmtTDczMylzmcfxTOXWs7gpmhzUHDi4gE5aI0CT9lA7bxb/zk4VgH4u/tWf8EZoG/l8zJbpxlfi+G1NEQxxHfphAZ3e1taWk8l+tC0RoAnQBGiCFGlnQ+2K1y1/82bKA7qMwVcFZVfTpj11CP3zPPL5KFYA+E1NurS85VzZ86LxAjT/OnCdsCRAE6AJ0AQp0s6NVYZOz0t1QKctc/OGNPwsrxMvyee3YboOfmpvP/fZEd9UHm3nbU4nLC+qLGcJ0ARoAjQBWhHQhsshDxW2PJ9Ice3GbKGbtvRORz6/PUL/RnySp86TRkaAJkAToKlK7a12XV6m+rSvG01WT+ER8vltW2bT2LLJtp6ZRkaAJkAToKlKkXbSTtopZwnQBGiqUqSdtJN2MspZAjQBmqoUaSftpJ1ylgC9JaDVZ1SlSDtpJ+1klLMEaAI0VSnSTtpJO+VsKgJ6b02dgP6IjIyMjIxsV0aAJkCTkZGRkR1MQP9kT+2jj7SqMzVqVrty0k7ayShn450DBGhyLFJO2kk7aaecJUCTY5Fy0k7aSTsZ5SwBmhyLlJN20k7aKWcJ0ARociwK1KSdtJNRzqYwoPX859hUQUx7t3p6ns2SY5Fy0k7aSTvlLAF6bwHde3ipkFseRQm/3dI4HnHXB9bBklIAt1PfsTxPnsxQlSLtpJ20U84mZxzdwbX3G9COJffHVZvsY3/uw+A+gzj9ll2YcJjbsjH9hcHfJsfLjGO1x28sDr4fdpfyf5x6smG0tveAO5Y+0jBB7DYPzFGwIu2knbSrNmcTGUcTc+0tAf3vb+MO6FxEMLtMdgm87GIAI+xn6wksy44f18iOych+9bS5MBkcq2roROchmXV+7q7fyhcKczZ9ZwJ6KVhJlhN1j9a4XGe+YZkCNWnfpfZW/d55qKpyds/iaM3oaHWUtCQmhkcG9Hf/OmQwdD7/U3wBfcn3Zq5QtDVoxDVLt7FZtpMlw9AkAboRHSEJvthw+Parhfum5jUDjieNY/k3t0gy5L+fdmRl+YccuafM9UXF9lIGX78NP+UAcExlgbp8zjjwbs24aYLBpZ4ug/nDYkiNsvRMLQQ2zr9+Zob5wmLOblPgAxjLztv776amLqm1r0PaRWu6OcTYO++E+mLcPFRdObtncVRfhKNR0pKYGB4J0F/8SkyD5hff7tU9aBtWw79qMzvcfr/fjnq//xQy/H434GC/cDuKjGFdqDq8Eto+yeBYtX2eS0aZXfI0D8h/b980oLAJC8eBu8lWpeaczvJYD+x1OkOd19mSIbpg+2IIoycY4XuHbG6CqV9WXdbOibnoMu5SzQp7kvGdlvr6LfboAnWGUtIubS0bBF90y4f64ueh6gL03sXRm8iOkpbExPAIgP6U9bHKnr/97o/sdX6/j4A2wjCkaPnoDt1zEIf41udcxqP15O9HTjx9kf6HtuYxj9Hm9HWnA1PaVAB0I1AR64EmIF22+Yb1dyYru+VRHYteFBUEBwm9wGqJ91oWYC4Tv5p9JG/PjrEAb3g9P+hgj9vtDRDv9fs7LfXmSqgWUqRd0u40oH/RNvZbYDjYVo2nh6oK0HsXR00w7PKWw97E8AiA/gH4+Rfcyn+Lwfxj/wBtw7DyvlXBjkz1iI+b4l3Jt09yulA3m/yOFWq3YahWOaCPF+PhVJNY4FYzcEvqXnvYpukDbqWH7UML2La5gaqHEqB7c1G6xmN7BOhKlPacKQaa0+qEFGkPat8A4+GWF4HAqHfiPDTVAb3zOFrBoDaeKY9XDFcG9KdFOPelsPoX4PsDAeg32oDLG46uMcV870l/G46aVHOscQYtWnUDep7tcMjK9dg1DYbFBu5nYISapN+AUKcKVoFXrW4J0AXAHbFjnQsUJkb7XCfbfDCaVAkp0i7T7keW0JMGAmMxifNQVQF6e3G0S2nnxMdwZUD/NTiw/T+gbj+HuFlATxRX5p4L2qn+iZAetH7RACZ/jF8fRX93yjnWBvItqQnoirA5MTnV5YqAdhrQETrDwsNgQ4iAmkCfwwJ8wy1dOFnLRcIMqe0KaUzlFbAWOMWsQluhdS6YIv1M1PvlOTOxah8ARu5q1Qkp0i7T3ikC+jGwGOxdKXsoWQLjqBeOsG/eVuzXtbcN6O/a238qrWfAHF9A62tqZiMBuo8D9AvUy6ez1+MFB2hbcC9fljhBaB4Gz4F3LH3EexvKT1+wja9rWsXCLUyqKtXscj0E/C7XWWHbemsJ+Z93BB4ErJga0kAztMBNIbjrci0BuS6Xi3PgWTdOh8PytThEuIDgdMs68OMqrguFWhmg33cGpnssAvNipj5yMehvtGhrXa7H7PYEe6Ve7zMDBoR1re1sPuxdo+FlNey6yS06OBGmOgYns62xBWq7VWh0qBFSpD2ofQqwCPEfgVCl6KFkiY2jPgSmOXG1vdfbYEf9N2WJjuFRX1TyZ6AkvoDOkQZ4FABd08dy+F7o6EAL7mm1bffkI9nnR1DcxtYDRsM1P5vvH2jHWkdGfeWpTVZZnwGlu+f6YTysUC5c+22PPnmqVLo0m7GK2+o9LG4Vif3kPnGONorZIq8ITH2s4NtdGXxQq/F+1uBt0npHjVpteZdw9yMbqJFFN24epUmvlQNaZp8AfUJaioWzL42zsdMpxMsz/HMTA/x6hccu7NAZ9iKBSpzmFg2sCK+Ywo3WGErdeUarWkiRdpn2B8DXrH/WlorvRYzkoWQJjqP6XOm+A1fbc0qEyr56NMExPCqgfwB+Hech7iLciARo3pQAHZ4DXkZjrV1lTFxPzHzwIkGoY0U0pcJlqfNCq1y4rA3dn02WKmWxWofZQGW1co+rt2YBlxf7bnQw4phedzHs75edEyMssqu1rVZrB9BitVo5+l0XJkA+4N9qY26q5MvXCIYbYK5CUeBatXIGuDcD+m0xSvkXA6yx5xm+kr54Fqd6AoDOhvnmkwUnv37cbm+0mgbZC5qPRQD0IhxTpvkOA9AQ84CcSbWPGpF2af0i0DM7xcCl1HcKeChZwuNoY2D+UwPcG/jwNL2gDjBYEhvDowH6l5eh+XucAZ0rvmHExnbOeRtur2rZHqC12gED8hkhF3xFeHeAHSvHNm6R3s1SeAgfpFXL+AOFtlS3AZ9olQt3Nf0yW7z9U4VJE6yC96DvsPjlb/eO2ZHFDV9fA4RpBINiMAveg/YBPq7h1Y/SwRsLlY5SfodZhm/XFcEVOL1N/iiDAqCXgUa+OecCrvB5fY/rKIuAxk2Rxdz6kI/vsrMtilFlQLNc5iOpk4V4M0GKtMeqPecC2wQFbine0JQ8lCzxcXQNml4J0GD4WKTvEMsncTE8GqC/B/4T7+egh8R5ETZZO2RD9vsLmK/LzAyTUsq9YKRWirFU03bAq6xkDoWn4+TW2onKigiFi1l9bRebV4Zr1UkSrAKArniIE+KsryvABN9LNv+/vfP7aSO74rg0uF8J4gfUCmeC4hAMCQ1GTgooiDoYDIgKitlRkwABJMIKCwkp5scKmiACUUg2YO0TLK5Qk5XzwiqiiWDbsiJR/gBUiaftY7VSX/qc/6B3Zu6MZ+wZQ5rUHifnvMTGk7HPnXPP5557zz2X97ud2YgZ0GXqpPgUiuS8g+A093YdOGCYhmGqMAD8nAXQzW74N7gbvJEa22qAPuVxpQAdVl9GWVx/zRrQfj6zfR9oJUiR7ifVPco6AdpUj/dkKm5toST596P9SkIK7+27PGpvxEx+ffgxgP4HsPf7jw3oLr7jrAFFS4o0L1aGDZ/X+0NdTPxwy/9Mh+pSeTmbicceUdo8/bbK50U8cv3MbNlIfL20Gu6xgjAscR9ZF8zFXwE2WUg8A3BULlNUE48WRJfSAT0LzfjlSFim5YAhacZlBvS4UuPV9Q0ftc1yb3ckT6wwJqcGp4zWX9kD+to0BDXWvQFoJU2kqzqgl1NDPWxprxfA118yAK173Cae9kOQIt2P1z3mV4rhfa28mUe1tYWSOMCP7mmHIrarU3iyDOFOfn14dkD/JOAvPxT//wC9n+2nxbCe8bcG63WAGj+mo4VgWEFzuawMkUfb1RPZHq7LtfizfNGqpwC6lA7opCF9cRpn1UDafxC0BPScMkKtBV+4uMa93a4828TepErymd6kA7p/BsqZK0zOGvzivA7oSgOg9SWSSiBpDehuw+X3CVKk+4l0Z8ZStLJYx3ebDKhzQ5kWSuIAP7qKNpEDWtASQd8Bnrz68KyA/tGLy98XOwnQymJ9kftq19BwexPqkiNlOy1r3Y9dh024JxaAYcVsoy9FDhhxLkP4Usz2cFnDNcm1fYPO71I6oBkWhzRxKyeXSa1MB2FmqsWTAegBxbu90td6vaq3m5e9mViBU/rtx4wreGmA9uwBU3xI24ghlyFE5oAOG4irB/NV5tWWFKAFvTPt2A6QCVKku1mWlMpjAAAIl0lEQVR39qKOjeYaGuGWze22seCzwUJJnOBHXwNrHNC6K5nLBHRufXg2QP/NjQv/Ks4xoKOdUwsHivTgtvriILlwS2ulpeimpvmocTiauOOoDf92hnWkrbxazo2wOPO5J3EFmJeyPlyX6PNb79pwKqBbzTMe8myH58sXyutvpjbTAH1DiaAT2qx4P/d2w8rkcwhf6LePQF6WtgS0yIYEc9xSJGPi9bYloPUqOJdgrmWnA/qy/qf7xm8lSJHu9rofhuBVKiD2VqCtWT4yadDSQkkc4Ucf8NkzM6ClvPrwLID+rg2Nfy/ONaA3LeewzRdeWpZ0QG+OqynfO88KwLA81RazAppIcmFeBqvH7UDT26wP1+WqPVosJED/FsKOLyXqcEscTV6Uj/pRi7WmAD2pxBmil3NwTfV2pyuUWjXD8Ope7YkxljUDmn1z6rQ3P6/mpN7cCtB6meQNqAvgGYDeFw0j6GcEKdL9BLqzESTfkPJEQHliVC/xmW6hJE7wo1N8pi0roHPsw+0B/e8v4P2xOOeAPgQmehVZQLv6orclbXlxEsOiBugqvxP9gI1hGVc+02WDRZkDSjB5yPru9Cdx3KQO6LjdqY39T0MskDADekUNki/yxOkt1a3Vq71nzjAfPWis8mAC9AJQmpqaMkK91BLQ+q7FBmDcEtDQK5is2NaWIEiR7ibd61MjyDLgwRH2g9YWSuIEP3ofCB4L6Bz7cFtA/3AO3v8U5x7QIiBqkYo2UpHMqXT12I/oEbSrVxDeFIhhBRptd+i4JsqBTm1EPcLG282fEqCXU1nc6dItqFNGKUBfV3f/s3hj+FsxsC604bnLtXRH3W33tR6UyGPZNskS0NvAc0O/YiDXduoFKiwBfdfwP+utAa0nd99D6uAWghTpnkV3nyELLCnPBM7bWCiJE/yox612/myAzrUPtwP0n1tR9MtiRwI6UqFOMWpr0CPwThSEYSWmsW8zqfGIMaSoLPV+R0DbYuEDuk9Lfn69jyH+WC/FfL0uV+3165r1zqBRVB63lkt96Fcj2j55bQM4YHTv2qvAnFKeWwzB369eV2KqKmIA9EuG9n7DzwgDHWo3E+dgCeginqMtXcZ+wBrQXbw6+K9ZKCQSpEj3E+i+ZNy1L2f3TthYKIkj/OgNvDsG0Dn34TaA/t0ehD8UOxLQkTqeuqsnifWhpqQADOtVCHhqebW0zIZeXaaJk1n2dMcKHtAjeuC5CySV5yo+VI71qS1CtUrCgFdN4apMxbFxtCmB9zM/4N8Vxb5GwBvnZrEClCqj1OZqCAkrQL8pSj8jfB44JV/6uAcXLAGNm8o6uPSQx+dv+4bfpAGaj42rhmBZ2Y4gRbpn6M5Gk/qqc2UHs6GLko2FkjjBj9bD7ckK6Nz7cGtA//U3wLk/6vKdcwAtJgVc9JgBLc5DGBEdbliPJwVjiohxYmXlKiDsbqbbCvzNhQ5oFuNe6Lwu53bV3mFcLeluWC0FQrKq71hUW7YoRZLlvKzeNQHCli8iU3DJq+11Sowpz1VqqdS7SXQa2JqQgrOsr9xyWQA64gUGFzRRagoE2LXCuZ5TFegbsQR0KcrPj75angFqlP3a44B70wRo7wCa6sORFTZmOCsSpEj3E+neUMFsK1IrLsXkIvTMdu4dWlsoiRP8aFAtcWAH6Hz4cJvjJs051D99LEBHy8609Pa+wJ+U3K9VCDwLrHeiJbaqNkQ/cGgE9P3e8Lc7aOM3qPqKeWfeYqltVmIfsBd0smGFO93sYdmslXcCraMZfz2AuaKcT92dVFBdShzST7Ma69Cs6YES9m7e1N4L3ENN6qdZyRk1Z2y/YeMsu6xCvnZKtAL0c5PtqpW7o6Xqu13RGtDhA375C3Un5Dr7WVUmQLslfgsMBwlSpPsJdf9njVK5QTkjoWVzRt1ZZWWhJE7wo0PKznRrQOfHh+cW0N3IImo60YbeIj4lu3dc+Uzb3y9XSdVOCo7gnH7jFeGY+qj5NSyxhzn8uWs2F0vtTyxiMnFuy3Qm8kvDOYsF06U2brGo4baq5eS0/CRD5/kwU3rWpWygm9fWaaTtkKAZMGuvwdN2X/Foy6scc7WSNi/De9WAlfsTI+fnBmcDOpnTAe0qG2Y3revhlZHHOmrU5e0UoF3S1BX2+0JxD0GKdD+x7lXrgmyGFbdHmFur7VBmfQYI0A71o0k8sAN0nnz4scdNftQ16MSV9q2eeQvp2Wo/p7rmQHk5j1CWlQrJ4Xjn3b4VbUtZ9GxqtblXaUztzZHo5C7bH+rrft97SOb0wRVBGCtEZyUFdaI9qgybishHw5GAyYAvBTm9+4+AyzH9kaY/W8/a+dnX/9vvs9jwxWEtrRnSw0T+QwyAln9wZe37Tch9zpORpDvvAGMlvjHaUVUQfnTNdodGvnx4bgH9XrJtVWjF4KpjqC6cLntIzur95GUjUDN0NPKkfiV+8+GH3au/u1ufWRqAW7QGtJXUChl5nQQp0p10/0T9qNiW5SCMw/y0gGMB/aoslvXzYGUlddlPV/ON7Sv6LODdD7tVQkArp3IJMod9WQDt44tIBGjSnXT/DFr2Ifac1gKOBTQZ1ueu+WLsfHx98qC++UOXL/aA9ljAJSbGa+ANnxjQq0c1EBoI0GTxpPtn0rIxePsd1gIEaOqyn7rmtXLiN6rl3DIh8yQqW0B3sMvVeuAEaNKddP8MWjba2rpGgCbDIs1z3O+SfnWyvKc588OWwcGE5f86mh/nKebbg3301El30p1algBNhkWaf3yRGkqe+hre0lMn3Ul3alkCNAGauhTpTrqT7iTUsgRoMizSnHQn3Ul3alkCNAGaDIs0J91Jd9KdWpYATYZFmpPupDvpTi1LgCZAk2GR5qQ76U66U8s6BNAkJCQkJCQkThMCNAkJCQkJiTMB/QsSEhISEhISxwkBmoSEhISExIHyX+LhGEDxdIwMAAAAAElFTkSuQmCC)

行をコピーして、テストに必要なデータを作る
例）名前が空白

![CSV1](data:image/png;base64,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)

### 3. 適当なテストを作る

```
docker exec -it formtest php artisan dusk:make FormTest
```

### 4. 3のテスト内で以下の設定をし、2のCSVを3のテストで利用するようにする
cf）tests/Browser/SampleFromTest.php

1. tests/Browser/FormTest.php内で、FillFormトレイトを利用
2. const DATA_FILE_NAME を指定
3. CSV内の参照したいデータキーを指定し、fillForm() 呼び出す（実際の入力データが返ってくる）

```
<?php

namespace Tests\Browser;

use Wizkch\Formtest\Tests\Browser\FillForm;
use Laravel\Dusk\Browser;
use Tests\DuskTestCase;

class FormTest extends DuskTestCase
{
    use FillForm;

    const DATA_FILE_NAME = 'form_test.csv';
```

### 5. 4に適当なテストを書いてみる
今回のフォームには、エラー表示要素のクラスに[data-field='{name}']が含まれる前提
Laravelのエラー表示部はバックエンドが担当だと思うので上記のような属性つけておくようにするとテストで捕捉しやすい

```
<?php

namespace Tests\Browser;

use Wizkch\Formtest\Tests\Browser\FillForm;
use Laravel\Dusk\Browser;
use Tests\DuskTestCase;

class FormTest extends DuskTestCase
{
    use FillForm;

    const DATA_FILE_NAME = 'form_test.csv';

    /**
     * A Dusk test example.
     *
     * @return void
     */
    public function test正常系()
    {
        $this->browse(function (Browser $browser) {
            $browser->visit('/form');

            $this->fillForm($browser, '正常系');

            $browser
                ->screenshot('form_正常系_投稿前')
                ->press('送信')
                ->assertSee('投稿完了')
                ->screenshot('form_正常系_投稿後');
        });
    }

    /**
     * A Dusk test example.
     *
     * @return void
     */
    public function test名前が空白のときエラー表示される()
    {
        $this->browse(function (Browser $browser) {
            $browser->visit('/form');

            $this->fillForm($browser, '名前が空白');

            $browser
                ->screenshot('form_名前が空白_投稿前')
                ->press('送信')
                ->assertDontSee('投稿完了')
                ->assertVisible("[data-field='name']")
                ->assertMissing("[data-field='email']")
                ->assertMissing("[data-field='type']")
                ->assertMissing("[data-field='genre']")
                ->assertMissing("[data-field='categories[]']")
                ->assertMissing("[data-field='content']")
                ->screenshot('form_名前が空白_投稿後');
        });
    }
```

### 6. 5のテストを実行する
```
docker exec -it formtest vendor/bin/phpunit tests/Browser/FormTest.php
```

※実際の動作サンプルはリポジトリformtest-playground 参照

[formtest-playground(bitbucket)](bitbucket.org/w012/formtest-playground/)
