<?php

namespace Uehi\Larapack\Models;

use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Storage;

/**
 * ファイルアップロード処理を含む
 * Trait FileUploads
 * @package Uehi\Larapack\Models
 */
trait FileUploads
{

    /**
     * fillableにcolumn追加したかどうか
     * @var bool
     */
    private $addedUploadColumns = false;

    /*
    |--------------------------------------------------------------------------
    | model上書き用
    |--------------------------------------------------------------------------
    */

    /**
     * アップロード対象のcolumnをget
     * @return array : 以下のようにstorageのpublic/local/s3などそのfilesystemに即して保存するか指定
     *      ['public_image1', 'public_image2'];
     *      ['public_image1', 'public_image2', 'local' => ['local_image1', 'local_image2']];
     *      ['public' => ['public_image1', 'public_image2'], 'local' => ['local_image1', 'local_image2'], 's3' => ['local_image1', 'local_image2']];
     * ※keyはfilesystems.phpのキー(public/local/s3など)に合わせる
     * FIXME: 細かいfilesystemsの設定値変更に対応しているわけではない(default設定のみ検証)
     */
    protected function getUploadColumns()
    {
        return [];
    }

    /**
     * localなfileの表示権限を返す。trueの場合表示可能
     *      /uploader/read で使われる。
     * @param $column
     * @return bool
     */
    public function localAuthorize($column)
    {
        return true;
    }

    /*
    |--------------------------------------------------------------------------
    | アップロードパラメータ保持用のaccessor
    |--------------------------------------------------------------------------
    */

    /**
     * アップロードパラメータ保持用のaccessor名
     */
    private function getUploadSupportColumns()
    {
        $inputSuffix = Config::get('const.uploads.input_suffix');
        $deleteSuffix = Config::get('const.uploads.delete_suffix');
        $columns = [];
        $normalized = $this->normalizedUploadColumns();
        foreach ($normalized as $type => $columnsInType) {
            foreach ($columnsInType as $column) {
                $columns[] = "{$column}{$inputSuffix}";
                $columns[] = "{$column}{$deleteSuffix}";
            }
        }
        return $columns;
    }

    protected $uploadInfo = [];
    protected function setUploadInfo($key, $value)
    {
        $this->uploadInfo[$key] = $value;
    }
    protected function getUploadInfo($key)
    {
        return !empty($this->uploadInfo[$key]) ? $this->uploadInfo[$key] : null;
    }

    /**
     * override アップロードパラメータを保持するためattributes拡張
     * @param array $attributes
     * @return $this
     */
    public function fill(array $attributes)
    {
        parent::fill($attributes);

        // ファイルアップに必要な情報をオブジェクト内に格納しておく
        foreach ($this->getUploadSupportColumns() as $column) {
            if (!empty($attributes[$column])) {
                $this->setUploadInfo($column, $attributes[$column]);
            }
        }


        return $this;
    }

    /*
    |--------------------------------------------------------------------------
    | boot
    |--------------------------------------------------------------------------
    */

    /**
     * boot
     */
    public static function bootFileUploads()
    {
        $self = new self;
        $uploadColumns = $self->normalizedUploadColumns();
        $inputSuffix = Config::get('const.uploads.input_suffix');
        $deleteSuffix = Config::get('const.uploads.delete_suffix');

        // ファイル名が渡ってきたら本来の保存領域に入れる
        static::saved(function ($record) use ($uploadColumns, $inputSuffix, $deleteSuffix) {
            if (!empty($uploadColumns)) {
                $isUpdate = false;
                foreach ($uploadColumns as $type => $columns) {
                    foreach ($columns as $column) {
                        // requestから保存ファイル名、削除フラグある場合取得
                        $inputColumn = "{$column}{$inputSuffix}";
                        $deleteColumn = "{$column}{$deleteSuffix}";
                        $tmpName = !empty($record->getUploadInfo($inputColumn)) ? $record->getUploadInfo($inputColumn) : null;
                        $deleteFlag = !empty($record->getUploadInfo($deleteColumn));

                        // 存在する場合
                        if (!empty($tmpName)) {
                            // 新しいファイル(tmp dirが含まれている時tmpだと判断)ある場合tmpから移動
                            if (static::isTmpPath($tmpName)) {
                                $baseDir = Storage::disk($type)->getDriver()->getAdapter()->getPathPrefix();

                                if ($type === 'public' && Config::get('const.uploads.include_storage_path')) {
                                    // tmpName の storage/tmp/~~の"storage/"を消す(設計ミス)
                                    $tmpPath = $baseDir . str_replace('storage/', '', $tmpName);
                                } else {
                                    $tmpPath = $baseDir . $tmpName;
                                }

                                if ($type !== 's3' && is_file($tmpPath) || $type === 's3' && Storage::disk("s3")->exists($tmpName)) {
                                    $saveDir = $record->getTable() . '/' . floor($record->id / 1000) . '/';
                                    $fullDir = $baseDir . $saveDir;
                                    $fullPath = $fullDir . basename($tmpPath);

                                    // ファイル移動はシームレスにいかなかったのでS3と分岐
                                    if ($type === 's3') {
                                        if (! Storage::disk($type)->exists($fullDir)) {
                                            Storage::disk($type)->makeDirectory($fullDir);
                                        }
                                        Storage::disk($type)->move($tmpPath, $fullPath);
                                    }
                                    else {
                                        if (!file_exists($fullDir)) {
                                            File::makeDirectory($fullDir, 0707, true);
                                        }

                                        if (rename($tmpPath, $fullPath)) {
                                            chmod($fullPath, 0707);
                                            // localの場合sessionを消しておく
                                            static::deletePrivateSession($tmpName);
                                        }
                                    }

                                    // modelにset
                                    $savePath = str_replace($baseDir, "", $fullPath);
                                    if ($type === 'public' && Config::get('const.uploads.include_storage_path')) {
                                        // storage/をつける(設計ミス)
                                        $savePath = "storage/{$savePath}";
                                    }
                                    $record->{$column} = $savePath;
                                    $isUpdate = true;
                                }
                            }
                        }

                        // deleteフラグがある場合null set
                        if ($deleteFlag && !empty($record->{$column})) {
                            $record->{$column} = null;
                            $isUpdate = true;
                        }

                        // 指定fieldがdirtyかどうか
                        // dirtyなら以前のファイルは削除
                        if ($record->isDirty($column)) {
                            $original = $record->getOriginal($column);
                            if (!empty($original) && $original !== $record->{$column}) {
                                if ($type === 's3') {
                                    if (Storage::disk($type)->exists($original)) {
                                        Storage::disk($type)->delete($original);
                                    }
                                }
                                else {
                                    $path = public_path($original);
                                    if (is_file($path)) {
                                        unlink($path);
                                    }
                                }
                            }
                        }

                    }
                }

                // 変更あればsave
                if ($isUpdate) {
                    $record->save();
                }
            }
            return true;
        });

    }

    /*
    |--------------------------------------------------------------------------
    | ファイル格納関連
    |--------------------------------------------------------------------------
    */

    /**
     * form値をtmpディレクトリに移動
     * FIXME: publicディレクトリ固定なので悩ましい
     * @param $column
     * @return bool
     */
    public function uploadToTmp($column)
    {
        $request = app('request');
        $params = $request->all();

        // hasManyの場合, $columnはchilds[1][image]のような形で来るので本来のcolumn名を抽出
        if (preg_match('/^(.+)\[([0-9]+)\]\[(.+)\]$/', $column, $m)) {
            $hasMany = true;
            $childProp = $m[1];
            $childNum = $m[2];
            $childColumn = $m[3];
            $value = !empty($params[$childProp][$childNum][$childColumn]) ? $params[$childProp][$childNum][$childColumn] : null;
        } else {
            $hasMany = false;
            $value = !empty($params[$column]) ? $params[$column] : null;
        }

        if (!empty($value)) {
            $path = Config::get('const.uploads.tmp_dir');

            if ($hasMany) {
                $column = $childColumn;
                $file = $request->file($childProp)[$childNum][$childColumn];
            } else {
                $file = $request->file($column);
            }

            $type = $this->getUploadType($column);

            if (!$type) {
                return false;
            }

            $visibility = $type !== 'local' ? 'public' : $type;
            $filename = Storage::disk($type)->put($path, $file, $visibility);

            return $filename;
        }
        return false;
    }

    /*
    |--------------------------------------------------------------------------
    | 共通処理
    |--------------------------------------------------------------------------
    */

    /**
     * filenameがtmpかどうかを判断
     * tmp dirが含まれている時tmpだと判断
     * @param $filename
     */
    public static function isTmpPath($filename)
    {
        $tmp = Config::get('const.uploads.tmp_dir') . '/';
        return strpos($filename, $tmp) !== false;
    }

    /**
     * 指定columnの保存がpublicかlocalかを返す
     * falseの場合、そのcolumnはupload columnではない
     * @param $column
     * @return bool|int|string
     */
    public function getUploadType($column)
    {
        $normalized = $this->normalizedUploadColumns();
        foreach ($normalized as $type => $upColumns) {
            foreach ($upColumns as $upColumn) {
                if ($upColumn === $column) {
                    return $type;
                }
            }
        }
        return false;
    }

    /**
     * getUploadColumnsにて指定された配列をpublic/local/s3分類した以下の形に整形
     *      ['public' => ['public_image1', 'public_image2'], 'local' => ['local_image1', 'local_image2'], 's2' => ['local_image1', 'local_image2']]
     * ※"local"は昔"private"の名だったので、互換性を保つためprivate->localにしている(filesystemに合わせlocalに変更)
     * @return array
     */
    protected function normalizedUploadColumns()
    {
        $normalized = ['public' => [], 'local' => [], 's3' => []];
        foreach ($this->getUploadColumns() as $key => $val) {
            if (is_array($val)) {
                foreach ($val as $childVal) {
                    if ($key === 'local' || $key === 'private') {
                        // local
                        $normalized['local'][] = $childVal;
                    }
                    // s3
                    elseif ($key === 's3') {
                        $normalized['s3'][] = $childVal;
                    } else {
                        // public
                        $normalized['public'][] = $childVal;
                    }
                }
            } else {
                // public
                $normalized['public'][] = $val;
            }
        }
        return $normalized;
    }

    /**
     * sessionに保存しているfilepathを消す
     * priavteの場合,sessionにfilepathを保存している
     * @param $filename
     */
    private static function deletePrivateSession($filename)
    {
        // session存在時(not api)
        if (app('request')->hasSession()) {
            $key = config('const.uploads.private_session_key');
            $sessionValue = app('request')->session()->get($key);
            if (empty($sessionValue) || !is_array($sessionValue)) {
                return true;
            }
            foreach ($sessionValue as $key => $val) {
                if ($val === $filename) {
                    unset($sessionValue[$key]);
                }
            }
            return app('request')->session()->put($key, $sessionValue);
        }
        return null;
    }

}
