<?php

namespace Uehi\Larapack;


use Illuminate\Http\Request;
use Illuminate\Support\HtmlString;
use Symfony\Component\Debug\Exception\ClassNotFoundException;
use Symfony\Component\Debug\Exception\FatalErrorException;

class Util
{

    /**
     * 各method内のjsをappendしたかを管理する
     * 1methodが複数回呼ばれた場合に何回もjsを吐き出すのを防ぐ
     * @var array
     */
    private $appends = [];

    /*
    |--------------------------------------------------------------------------
    | 共通
    |--------------------------------------------------------------------------
    */

    /**
     * プロジェクト内にファイルを設置している場合そのview名を返し
     * 設置していない場合デフォルトのview名を返す
     * @param $name
     * @return string
     */
    public function getPartialName($name)
    {
        $sharedValues = app('view')->getShared();
        $viewBaseDirName = $sharedValues['viewBaseDirName'];
        if (file_exists(resource_path("views/admin/{$viewBaseDirName}/{$name}.blade.php"))) {
            return "admin.{$viewBaseDirName}.{$name}";
        } else {
            return "larapack::partials.admin.{$name}";
        }
    }

    /**
     * プルダウン用の数値配列を生成
     * @param $start: 開始番号
     * @param $end: 終了番号
     * @param int $interval: 間隔
     * @param null $unit: 末尾に付加する文字列
     * @param bool $zeroPad: 先頭に0付加するか
     * @return array
     */
    public function getSelectNum($start, $end, $interval = 1, $unit = null, $zeroPad = false)
    {
        $ret = [];
        for ($key = $start; $key <= $end; $key = $key + $interval) {
            if ($zeroPad) {
                $key = sprintf("%02d", $key);
            }
            $ret[$key] = "{$key}{$unit}";
        }
        return $ret;
    }

    /**
     * table名から紐づくmodelのnamespaceを求める
     * @param $table
     * @return bool|mixed
     */
    public function getModelByTableName($table)
    {
        $models = $this->getModels();
        foreach ($models as $model) {
            if ($table === (new $model)->getTable()) {
                return $model;
            }
        }
        return false;
    }

    /**
     * 全modelのnamespaceを取得
     * @return array
     */
    public function getModels()
    {
        $path = app_path('Models');
        $out = [];
        $ret = scandir($path);
        foreach ($ret as $val) {
            if ($val === '.' or $val === '..') continue;
            $filename = $path . '/' . $val;
            if (is_dir($filename)) {
                $out = array_merge($out, getModels($filename));
            } else {
                $name = substr(basename($filename),0,-4);
                $out[] = "App\\Models\\{$name}";
            }
        }
        return $out;
    }

    /*
    |--------------------------------------------------------------------------
    | View内出力関連
    |--------------------------------------------------------------------------
    */

    /**
     * 既にappend下かどうか
     * @param $method
     * @return bool
     */
    private function isAppend($method)
    {
        return !empty($this->appends[$method]);
    }

    /**
     * jsをappendする
     */
    private function appendJs($js, $methodName)
    {
        $js = '<script type="text/javascript">' . $js . '</script>';
        app('larapack')->appendJsBlock($js);
        $this->appends[$methodName] = true;
    }

    /**
     * 更新管理者IDから管理者名を取得
     * @param $record モデルオブジェクト
     * @param $record
     * @param array $options
     *      $model 更新者ユーザ格納モデル
     *      $inputField 更新者IDフィールド
     *      $displayField 返すフィールド
     * @return string
     * @throws ClassNotFoundException
     * @throws FatalErrorException
     */
    public function inputUser($record, $options = [])
    {
        $defaults = [
            'model' => class_exists(\App\Models\AdminUser::class) ? \App\Models\AdminUser::class : null,
            'inputField' => 'updated_id',
            'displayField' => 'title',
        ];
        $options = array_merge($defaults, $options);

        // model有無チェック
        if (empty($options['model'])) {
            throw new FatalErrorException('Input User Model should not be null.', null, null, __FILE__, __LINE__);
        } elseif (!class_exists($options['model'])) {
            throw new ClassNotFoundException(sprintf('Class %s not found.', $options['model']), $options['model']);
        }

        if (!empty($record->{$options['inputField']})) {
            $model = $options['model'];
            $user = $model::find($record->{$options['inputField']});
            if (!empty($user->{$options['displayField']})) {
                return $user->{$options['displayField']};
            }
        }
        return '';
    }

    /**
     * 並び順変更のinput text表示
     * @param $record
     * @param array $options
     * @return HtmlString
     */
    public function inputSequence($record, $options = [])
    {
        $defaults = [
            'url' => admin_url('/larapack/changeSequence'),
            'model' => get_class($record),
            'field' => 'sequence',
            'redirect' => null,
        ];
        $options = array_merge($defaults, $options);

        $formId = 'changeSequence_' . uniqid();
        $ret = '';

        $ret .= app('form')->model($record, ['url' => $options['url'], 'method' => 'PATCH', 'id' => $formId]);
        $ret .= '<div class="input-group">';
        $ret .= app('form')->text('value', $record->{$options['field']}, ['class' => "form-control"]);
        $ret .= app('form')->hidden('id', $record->id);
        $ret .= app('form')->hidden('model', $options['model']);
        $ret .= app('form')->hidden('field', $options['field']);
        $ret .= app('form')->hidden('redirect', $options['redirect']);
        $ret .= '<span class="input-group-btn">';
        $ret .= "<a href='javascript:void(0);' class='btn btn-primary' onclick=\"jQuery('#{$formId}').submit();\">変更</a>";
        $ret .= '</span>';
        $ret .= '</div>';
        $ret .= app('form')->close();

        return new HtmlString($ret);
    }

    /**
     * 公開非公開のAjax切り替えプルダウンの表示(管理側用)
     * @param array $record
     * @param array $options
     * @return string
     */
    public function selectPublic($record, $options = [])
    {
        $defaults = [
            'url' => admin_url('/larapack/changePublic'),
            'list' => config('values.public'),
            'model' => get_class($record),
            'callback' => '',
            'field' => 'public',
        ];
        $options = array_merge($defaults, $options);

        $selectClass = 'changePublic_select';
        $msgClass = 'changePublic_msg';
        $ret = '';

        $ret .= app('form')->model($record, ['url' => $options['url'], 'method' => 'PATCH']);
        $ret .= app('form')->select('flag', $options['list'], $record->{$options['field']}, ['class' => "form-control {$selectClass}"]);
        $ret .= app('form')->hidden('id', $record->id);
        $ret .= app('form')->hidden('model', $options['model']);
        $ret .= app('form')->hidden('field', $options['field']);
        $ret .= app('form')->close();

        $ret .= "<div class='{$msgClass}'></div>";

        // js
        if (!$this->isAppend(__METHOD__)) {
            $js = '';
            $js .= "
                jQuery(function($){
                    var selectObj = $('.{$selectClass}');
                    // 色
                    var publicColor = '';
                    var notPublicColor = '#E3E3E3';
                    // ロード時に色変える
                    selectObj.each (function(){
                        $(this).closest('tr').children('td').css('background-color', ($(this).val() == '1' ? publicColor : notPublicColor));
                    });
    
                    // イベント
                    selectObj.change(function(e){
                        var msgObj = $(e.target).closest('td').children('.{$msgClass}');
                        msgObj.text('変更中…');
                        var keyValues = {};
                        $(e.target).closest('form').find(':input').each(function(){
                            keyValues[$(this).attr('name')] = $(this).val();
                        });
                        $.ajax({
                            type: 'POST',
                            url: '{$options['url']}',
                            data: keyValues,
                            success: function(html){
                                html = html.replace(/(^\s+)|(\s+$)/g, '');
                                if(html.length > 0){
                                    msgObj.text('変更失敗！');
                                }else{
                                    // 色の変更
                                    $(e.target).closest('tr').children('td').css('background-color', ($(e.target).val() == '1' ? publicColor : notPublicColor));
                                    msgObj.text('');
                                    // callback
                                    {$options['callback']}
                                }
                            },
                            error: function(){
                                msgObj.text('変更失敗！');
                            }
                        });
                    });
                });
            ";
            $this->appendJs($js, __METHOD__);
        }

        return new HtmlString($ret);
    }

    /**
     * 状態のAjax切り替えプルダウンの表示(管理側用)
     * @param array $record
     * @param array $options
     * @return string
     */
    public function selectStatus($record, $options = [])
    {
        $defaults = [
            'url' => admin_url('/larapack/changePublic'),
            'list' => config('values.selectEmpty') + config('values.private_public_statuses'),
            'model' => get_class($record),
            'callback' => '',
            'field' => 'private_public_status',
            'value' => '',
            'public_text_id' => 'public_text',
            'status_text_id' => 'status_text',
        ];
        $options = array_merge($defaults, $options);

        $selectClass = 'changeStatus_select';
        $msgClass = 'changeStatus_msg';
        $ret = '';

        $ret .= app('form')->model($record, ['url' => $options['url'], 'method' => 'PATCH']);
        $ret .= app('form')->select('flag', $options['list'], $options['value'], ['class' => "form-control {$selectClass}"]);
        $ret .= app('form')->hidden('id', $record->id);
        $ret .= app('form')->hidden('model', $options['model']);
        $ret .= app('form')->hidden('field', $options['field']);
        $ret .= app('form')->close();

        $ret .= "<div class='{$msgClass}'></div>";

        // js
        if (!$this->isAppend(__METHOD__)) {
            $js = '';
            $js .= "
                jQuery(function($){
                    var selectObj = $('.{$selectClass}');
                    // 色
                    var publicColor = '';
                    var notPublicColor = '#E3E3E3';
                    var editPublicColor = '#fcf8e3';
                    // ロード時に色変える
                    selectObj.each (function(){
                        var isPublic = $(this).closest('tr').find('#{$options['public_text_id']}').text() == '公開';
                        var isEdit = $(this).closest('tr').find('#{$options['status_text_id']}').text() == '編集中';
                        $(this).closest('tr').children('td').css('background-color', (isPublic ? (isEdit ? editPublicColor : publicColor) : notPublicColor));
                    });
    
                    // イベント
                    selectObj.change(function(e){
                        var msgObj = $(e.target).closest('td').children('.{$msgClass}');
                        msgObj.text('変更中…');
                        var keyValues = {};
                        $(e.target).closest('form').find(':input').each(function(){
                            keyValues[$(this).attr('name')] = $(this).val();
                        });
                        $.ajax({
                            type: 'POST',
                            url: '{$options['url']}',
                            data: keyValues,
                            success: function(html){
                                html = html.replace(/(^\s+)|(\s+$)/g, '');
                                if(html.length > 0){
                                    msgObj.text('変更失敗！');
                                }else{
                                    var select_val = $(e.target).val();
                                    var select_text = $(e.target).find('option:selected').text();
                                    // テキストの変更
                                    if(select_val != '0') {
                                        $(e.target).closest('tr').find('#{$options['public_text_id']}').text(select_text);
                                        $(e.target).closest('tr').find('#{$options['status_text_id']}').text(select_text);
                                    }
                                    else {
                                        $(e.target).closest('tr').find('#{$options['status_text_id']}').text(select_text);
                                    }
                                    // 色の変更
                                    if(select_val != '0') {
                                        $(e.target).closest('tr').children('td').css('background-color', (select_val == '1' ? publicColor : notPublicColor));
                                    }
                                    else if($(e.target).closest('tr').find('#{$options['public_text_id']}').text() == '公開'){
                                        $(e.target).closest('tr').children('td').css('background-color', editPublicColor);
                                    }
                                    msgObj.text('');
                                    // callback
                                    {$options['callback']}
                                }
                            },
                            error: function(){
                                msgObj.text('変更失敗！');
                            },
                            complete: function(){
                                selectObj.val('');
                            }
                        });
                    });
                });
            ";
            $this->appendJs($js, __METHOD__);
        }

        return new HtmlString($ret);
    }

    /**
     * 管理画面一覧の"詳細","プレビュー","編集","コピー","削除"のパネルを返す
     * @param $record 操作するレコード
     * @param array $buttons 表示するボタン
     *      'show', 'preview', 'edit', 'copy', 'delete'が指定可能
     *      配列順に表示される
     * @return string
     */
    public function operationPanel($record, array $buttons = ['show', 'preview', 'edit', 'copy', 'delete'])
    {
        $sharedValues = app('view')->getShared();
        $controllerNameSnake = $sharedValues['controllerNameSnake'];

        $delForm = '';
        $delId = 'operationPanalDel_' . uniqid();
        if (in_array('delete', $buttons)) {
            $delForm .= app('form')->model($record, ['url' => admin_url("/{$controllerNameSnake}/{$record->id}"), 'method' => 'DELETE', 'id' => $delId]);
            $delForm .= app('form')->close();
        }

        $ret = '';
        foreach ($buttons as $type) {
            if ($type == 'show') {
                $ret .= '<a class="btn btn-default" href="' . admin_url("/{$controllerNameSnake}/{$record->id}") . '">詳細</a>';
            } elseif ($type == 'preview') {
                $ret .= '<a class="btn btn-default" href="' . admin_url("/{$controllerNameSnake}/{$record->id}/preview") . '" target="_blank">プレビュー</a>';
            } elseif ($type == 'edit') {
                $ret .= '<a class="btn btn-primary" href="' . admin_url("/{$controllerNameSnake}/{$record->id}/edit") . '">編集</a>';
            } elseif ($type == 'copy') {
                $ret .= '<a class="btn btn-primary" href="' . admin_url("/{$controllerNameSnake}/{$record->id}/copy") . '">コピー</a>';
            } elseif ($type == 'delete') {
                $onclick = "if (window.confirm('削除してもいいですか?')) jQuery('#{$delId}').submit();";
                $ret .= '<a class="btn btn-danger" href="javascript:void(0);" onclick="' . $onclick . '">削除</a>';
            }
        }

        return new HtmlString("<div class='btn-group'>{$ret}</div>{$delForm}");
    }

    /**
     * paginator
     * @param $records
     * @return mixed
     */
    public function paginator($records)
    {
        return new HtmlString($records->appends(app('request')->except('page', 'back'))->links());
    }

    /**
     * sortのパラメータを付加したリンクを返す
     * @param $field
     * @param $title
     * @return HtmlString
     */
    public function sortLink($field, $title)
    {
        $currentSort = app('request')->get('sort');
        $currentSortDir = app('request')->get('sort_dir');
        if (!empty($currentSortDir)) {
            $currentSortDir = strtolower($currentSortDir);
        }

        $newSort = $field;

        // ソート方向を決める
        if ($currentSort === $field) {
            if ($currentSortDir === 'asc') {
                $newSortDir = 'desc';
            } elseif ($currentSortDir === 'desc') {
                $newSortDir = '';
            } else {
                $newSortDir = 'asc';
            }
        } else {
            $newSortDir = 'asc';
        }

        // icon
        $iconType = '';
        if ($currentSort === $field) {
            if ($currentSortDir === 'asc') {
                $iconType = '<i class="fa fa-sort-asc fa-lg" aria-hidden="true"></i>';
            } elseif ($currentSortDir === 'desc') {
                $iconType = '<i class="fa fa-sort-desc fa-lg" aria-hidden="true"></i>';
            }
        } else {
            $iconType = '<i class="fa fa-sort fa-lg" aria-hidden="true"></i>';
        }

        $newParams = app('request')->except(['sort', 'sort_dir', 'page', 'back']);
        if (!empty($newSort) && !empty($newSortDir)) {
            $newParams['sort'] = $newSort;
            $newParams['sort_dir'] = $newSortDir;
        }
        $newParams = http_build_query($newParams);
        $html = "<a href='?{$newParams}'>{$title} {$iconType}</a>";

        return new HtmlString($html);
    }

    /**
     * 表示件数を変える
     * @return HtmlString
     */
    public function selectLimit()
    {
        $class = 'limitSelect';
        $ret = app('form')->select('limit', config('values.pageLimits'), null, ['class' => "form-control {$class}"]);

        // requestをqueryに変換
        $params = app('request')->except(['page', 'back', 'sort', 'sort_dir', 'limit']);
        $paramStr = http_build_query($params);

        // js
        $js = "
            jQuery(function($){
                var queryStr = '{$paramStr}';
                $('.{$class}').change(function(e){
                    var val = $(this).val();
                    location.href = '?' + queryStr + '&limit=' + val;
                });
            });
        ";
        $this->appendJs($js, __METHOD__);

        return new HtmlString($ret);
    }

    /**
     * 管理画面必須アイコンを返す
     */
    public function getRequiredIcon()
    {
        return new HtmlString(' <span class="ness"><i class="fa fa-asterisk"></i></span>');
    }

    /*
    |--------------------------------------------------------------------------
    | google map url発行関係
    |--------------------------------------------------------------------------
    */

    /**
     * GoogleMaps JS APIのJSファイルのURLを、APIKeyもしくはclient IDを含めて返す
     * APIKey, client IDのいずれかをconstに設定しておく
     * channel id追加したい場合はconst設定
     * @param array $params
     * @param string|null $scheme urlスキーム(http|https)
     * @return string JS APIのJSのURL
     */
    public function gmapJsUrl(array $params = [], $scheme = null)
    {
        if (empty($scheme) || (!is_scalar($scheme))) {
            $scheme = 'https';
        }

        $defaults = [
            'language' => 'ja',
            'region' => 'jp',
        ];
        // version指定あれば
        if ($v = app('config')->get('const.gmap.v')) {
            $defaults['v'] = $v;
        }
        $params = http_build_query(array_merge($defaults, $params));

        $url = "{$scheme}://maps.google.com/maps/api/js?{$params}";

        $keyQuery = $this->getGmapKeys();
        if (!empty($keyQuery)) {
            $url .= "&{$keyQuery}";
        }

        return $url;
    }

    /**
     * Google Static Mapsの画像のURLを、APIKeyもしくはclient IDを含めて返す。
     * APIKey, client IDのいずれかをconstに設定しておく
     * channel id追加したい場合はconst設定
     *
     * ※keyの指定方法にかかわらず、constにgmap.private_keyがあれば
     * URL中にデジタル署名(signature)を挿入します。
     *
     * @param $lat string
     * @param $lon string
     * @param $zoom string
     * @param $width string
     * @param $height string
     * @param string|null $scheme urlスキーム(http|https)
     * @return string StaticMapsのURL
     */
    public function gmapStaticUrl($lat, $lon, $zoom, $width, $height, $scheme = null)
    {
        if (empty($scheme) || (!is_scalar($scheme))) {
            $scheme = 'https';
        }

        $url = "{$scheme}://maps.googleapis.com/maps/api/staticmap";
        $url .= "?center={$lat},{$lon}&zoom={$zoom}&size={$width}x{$height}";
        $url .= "&maptype=roadmap&markers=color:red|{$lat},{$lon}&region=jp";

        $keyQuery = $this->getGmapKeys();
        if (!empty($keyQuery)) {
            $url .= "&{$keyQuery}";
        }

        // GETのみで成り立つAPIはデジタル署名必須
        if ($secret = app('config')->get('const.gmap.private_key')) {
            $url = $this->gmapSignUrl($url, $secret);
        }

        return $url;
    }

    /**
     * 設定からkeyを読み込みhttpクエリ文字列で返す
     * @return string
     */
    private function getGmapKeys()
    {
        $params = [];
        // ApiKeyが優先、ApiKeyなければclientIdを拾う。なければ何もせず返す
        if ($apiKey = app('config')->get('const.gmap.key')) {
            $params['key'] = rawurlencode($apiKey);
        } elseif ($clientId = app('config')->get('const.gmap.client_id')) {
            $params['client'] = rawurlencode($clientId);

            // 集計用にchannel IDを指定していればチャネルIDを付与
            $channelId = app('config')->get('const.gmap.channel_id');

            // 無指定の場合はホスト名+baseUrlから生成したものを利用
            if (!$channelId) {
                $host = app('request')->getHttpHost();
                $base = app('request')->getBaseUrl() ? '_' . preg_replace('{/}', '_', app('request')->getBaseUrl()) : '';
                $channelId = "{$host}{$base}";
            }
            $params['channel'] = rawurlencode($channelId);
        }
        return http_build_query($params);
    }

    //google maps api premium plan

    // Encode a string to URL-safe base64
    protected function encodeBase64UrlSafe($value)
    {
        return str_replace(array('+', '/'), array('-', '_'),
            base64_encode($value));
    }

    // Decode a string from URL-safe base64
    protected function decodeBase64UrlSafe($value)
    {
        return base64_decode(str_replace(array('-', '_'), array('+', '/'),
            $value));
    }

    // Sign a URL with a given crypto key
    // Note that this URL must be properly URL-encoded
    private function gmapSignUrl($myUrlToSign, $privateKey)
    {
        // parse the url
        $url = parse_url($myUrlToSign);

        $urlPartToSign = $url['path'] . "?" . $url['query'];

        // Decode the private key into its binary format
        $decodedKey = $this->decodeBase64UrlSafe($privateKey);

        // Create a signature using the private key and the URL-encoded
        // string using HMAC SHA1. This signature will be binary.
        $signature = hash_hmac("sha1",$urlPartToSign, $decodedKey,  true);

        $encodedSignature = $this->encodeBase64UrlSafe($signature);

        return $myUrlToSign."&signature=".$encodedSignature;
    }

}
