<?php

namespace Uehi\Larapack;


use Illuminate\Support\Collection;
use Illuminate\Support\HtmlString;

class MultiBlock
{

    /**
     * @var string js内でのMultiBlockオブジェクト名
     */
    private $mbObjName = "mb";

    /**
     * 確認表示画面かどうか
     * Larapack::formのisConfirmを見る
     * @return mixed
     */
    private function isConfirm()
    {
        return app('larapack')->form()->isConfirm();
    }

    /*
    |--------------------------------------------------------------------------
    | 処理
    |--------------------------------------------------------------------------
    */

    /**
     * n段を表示
     * @param string $name プロパティ名(メインmodelのfunction名)
     * @param string $template テンプレートファイル
     * @param array $options
     *      array sortKey 順番を決めるfield default id
     *      boolean ck n段内部にck利用する場合はtrueにする
     *      boolean first データが1件もない場合に1つフォームを表示させておくか
     * @return HtmlString
     */
    public function show($name, $template, array $options = [])
    {
        $defaults = [
            'sortKey' => 'id',
            'ck' => false,
            'first' => false
        ];
        $options = array_merge($defaults, $options);

        // 設定値(同一ページで複数のmultiblockができるようにユニークに)
        // model
        $templateName = preg_replace('/(::|\-|\.)/', '_', $template);
        $templateName = str_replace('___', '_', $templateName);
        // adding id
        $addId = "{$templateName}_mainDetail";
        // templateid
        $templateId = "{$templateName}_template";
        // blockIdFormat(jsで___でsplitするため$templateNameに___が入らない前提)
        $blockIdFormat = "{$templateName}___%s";

        // $this->mbObjNameもユニークに
        $this->mbObjName = "{$templateName}_mb";

        // 対象データ
        // 初期値
        $errors = app('view')->shared('errors');
        if (count($errors) > 0) {
            // validationエラー時は前回の入力値から取る
            // 以下FormBuilder->typeをselect or checkboxにするための苦肉の策(そうしないとold()が2次元配列を返してくれない)
            app('form')->select('dummy');
            $data = app('form')->getValueAttribute($name);
        } else {
            // 初回表示時
            $data = app('larapack')->form()->getDisplayInput($name);
        }
        // 型を配列で統一
        if ($data instanceof Collection) {
            $data = $data->toArray();
        }

        // 確認画面から戻った場合も上記のエラー時と同じっぽい挙動になるので同じことで修正
        if(!empty($data)) {
            $hasIntKey = false;
            foreach($data as $key => $val) {
                if(is_int($key)) {
                    $hasIntKey = true;
                    break;
                }
            }
            if(!$hasIntKey) {
                app('form')->select('dummy');
                $data = app('form')->getValueAttribute($name);
            }
        }

        // 出力文字列
        $ret = '';

        // 入力画面のみ
        if (!$this->isConfirm()) {
            // js読み込み
            app('larapack')->appendJs(asset('/larapack/js/multi_block.js'));
            $ckInit = ($options['ck']) ? "jQuery.event.add(window, 'load', function(){ {$this->mbObjName}.initCkEditor() });" : '';
            $js = "
                // 段落追加
                var {$this->mbObjName}  = new MultiBlock('{$addId}', '{$templateId}', '{$options['sortKey']}', " . ($options['ck'] ? 'true' : 'false') . ");
                {$ckInit}
            ";
            app('larapack')->appendJsBlock('<script type="text/javascript">' . $js . '</script>');

            // ブロック何もなくても全削除のため専用パラメータ飛ばす
            $ret .= app('form')->hidden("{$name}__exists", null);
            // 複製用のテンプレート
            $blockId = sprintf($blockIdFormat, 'NNN');
            $block = '';
            $block .= "<div id='{$templateId}' style='display: none;'>";
            $block .= "<div id='{$blockId}'>";
            $block .= app('view')->make($template, [
                'key' => 'NNN',
                'blockId' => $blockId
            ]);
            $block .= '</div>';
            $block .= '</div>';

            $ret .= $block;
        }

        // メイン部分
        $ret .= "<div id='{$addId}'>";
        // フォーム値がなければ一つだけ表示する
        if (empty($data) && $options['first']) {
            $data[0] = true;
        }
        // 値をループして持ち回す
        if (!empty($data)) {
            if (!empty($options['sortKey'])) {
                $data = $this->sort($data, [$options['sortKey'] => 'asc']);
            }
            foreach ($data as $key => $val) {
                if ($key === 'NNN') {
                    continue; //型まで比較
                }
                $blockId = sprintf($blockIdFormat, $key);
                $block = '';
                $block .= "<div id='{$blockId}'>";
                $block .= app('view')->make($template, [
                    'key' => $key,
                    'blockId' => $blockId
                ])->withErrors(app('view')->shared('errors'));
                $block .= '</div>';

                $ret .= $block;
            }
        }
        $ret .= '</div>';

        // ボタン
        if (!$this->isConfirm()) {
            $ret .= "<a href='javascript:void(0);' onclick='{$this->mbObjName}.addForm();return false;' id='addButton' class='btn btn-primary btn-sm'>";
            $ret .= 'ブロックを追加';
            $ret .= '</a>';
        }

        return new HtmlString($ret);
    }

    /**
     * 上に移動
     * @param string $blockId ブロックID
     * @param boolean $scriptOnly trueにするとjsのみ返す
     * @return HtmlString|null
     */
    public function up($blockId, $scriptOnly = false)
    {
        if ($this->isConfirm()) {
            return null;
        }
        $script = "javascript:{$this->mbObjName}.up('{$blockId}');";
        if ($scriptOnly) {
            return new HtmlString($script);
        } else {
            return new HtmlString("<a href=\"{$script}\" class='btn btn-success btn-sm'><i class='fa fa-arrow-up'></i></a>");
        }
    }

    /**
     * 下に移動
     * @param string $blockId ブロックID
     * @param boolean $scriptOnly trueにするとjsのみ返す
     * @return HtmlString|null
     */
    public function down($blockId, $scriptOnly = false)
    {
        if ($this->isConfirm()) {
            return null;
        }
        $script = "javascript:{$this->mbObjName}.down('{$blockId}');";
        if ($scriptOnly) {
            return new HtmlString($script);
        } else {
            return new HtmlString("<a href=\"{$script}\" class='btn btn-success btn-sm'><i class='fa fa-arrow-down'></i></a>");
        }
    }

    /**
     * 削除
     * @param string $blockId ブロックID
     * @param boolean $scriptOnly trueにするとjsのみ返す
     * @return HtmlString|null
     */
    public function delete($blockId, $scriptOnly = false)
    {
        if ($this->isConfirm()) {
            return null;
        }
        $script = "javascript:{$this->mbObjName}.deleteBlock('{$blockId}');";
        if ($scriptOnly) {
            return new HtmlString($script);
        } else {
            return new HtmlString("<a href=\"{$script}\" class='btn btn-danger btn-sm'><i class='fa fa-times'></i></a>");
        }
    }

    /**
     * entityオブジェクトの配列をソートする
     * 配列のキーは変更されないようにする
     * @param array $data
     * @param array $sort
     * @return array|bool
     */
    private function sort(array $data, array $sort)
    {
        // 配列要素が1つ以下の場合ソート必要なし
        if (count($data) <= 1) {
            return $data;
        }

        // $sortからソート対象のpropertyとソート方向を取得
        $field = $order = null;
        foreach ($sort as $key => $val) {
            $field = $key;
            $order = strtolower($val);
            break;
        }
        if (empty($field) || empty($order) || (!empty($order) && !in_array($order, ['asc', 'desc']))) {
            return false;
        }

        // sort対象の要素のみキーそのままで取り出す
        $targets = [];
        foreach ($data as $key => $val) {
            $entity = $data[$key];
            if (!isset($entity[$field])) {
                // 該当のソートができない場合そのまま返す
                return $data;
            }
            $targets[$key] = $entity[$field];
        }

        // sort
        ($order == 'asc') ? asort($targets) : arsort($targets);

        // targetsのキーの順に$dataの要素を取り出す
        $ret = [];
        foreach (array_keys($targets) as $k) {
            $ret[$k] = $data[$k];
        }

        return $ret;
    }

}