<?php

namespace Uehi\Larapack\Providers;

use Illuminate\Database\Connection;
use Illuminate\Foundation\AliasLoader;
use Illuminate\Routing\Router;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\ServiceProvider;
use Illuminate\Support\Facades\Route;
use Illuminate\Support\Facades\View;
use Illuminate\View\Compilers\BladeCompiler;
use Uehi\Larapack\Extensions\Validator\LarapackValidator;
use Uehi\Larapack\Larapack;

class LarapackServiceProvider extends ServiceProvider
{

    /**
     * パッケージのルートディレクトリ
     * @var null|string
     */
    protected $baseDir = null;

    /**
     * パッケージのsrcディレクトリ
     * @var null|string
     */
    protected $srcDir = null;

    /**
     * 登録するcommand
     * @var array
     */
    protected $commands = [
        'InstallCommand',
        'UninstallCommand',
        'MigrateMakeCommand',
        'RequestMakeCommand',
        'ModelMakeCommand',
        'ControllerMakeCommand',
        'ViewMakeCommand',
        'CrudMakeCommand',
        'DockerInstallCommand',
        'JsonInstallCommand',
        'CleanUploadsCommand',
    ];

    /**
     * 登録するconnection
     * custom blueprintのため
     * @var array
     */
    protected $connections = [
        'mysql' => '\Uehi\Larapack\Extensions\Database\LarapackMySqlConnection'
    ];

    /**
     * 登録するrouteMiddleware
     * @var array
     */
    protected $routeMiddleware = [
        'auth.admin' => \Uehi\Larapack\Middleware\AdminAuthMiddleware::class,
        'request.format' => \Uehi\Larapack\Middleware\RequestFormatMiddleware::class,
        'prefix_basic_auth' => \Uehi\Larapack\Middleware\PrefixBasicAuthMiddleware::class,
    ];

    /**
     * 登録するmiddleware group
     * @var array
     */
    protected $middlewareGroups = [
        'web' => ['request.format', 'prefix_basic_auth'],
        'api' => ['request.format', 'prefix_basic_auth'],
    ];

    /**
     * LarapackServiceProvider constructor.
     * @param \Illuminate\Contracts\Foundation\Application $app
     */
    public function __construct($app)
    {
        parent::__construct($app);

        $this->baseDir = dirname(dirname(__DIR__));
        $this->srcDir = $this->baseDir . '/src';
    }

    /**
     * Bootstrap any application services.
     *
     * @return void
     */
    public function boot()
    {
        // merge configs
        $this->mergeConfigs();

        // load routes
        $this->loadRoutesFrom("{$this->baseDir}/routes/web.php");
        // viewを登録( view(larapack::hoge)でアクセス可能 )
        $this->loadViewsFrom("{$this->baseDir}/resources/views", 'larapack');
        // langを登録
        $this->loadTranslationsFrom("{$this->baseDir}/resources/lang/", 'larapack');
        // migrate時に実行されるもの
        $this->loadMigrationsFrom("{$this->baseDir}/database/migrations");
        // helper
        $this->registerHelpers();
        // validator
        Validator::resolver(function ($translator, $data, $rules, $messages, $attributes) {
            return new LarapackValidator($translator, $data, $rules, $messages, $attributes);
        });

        if (app()->runningInConsole()) {
            // vendor:publishで必ず外出ししてほしいものを指定
            $this->publishRequired();

            // vendor:publishで外出しすることでカスタマイズ可能なものを指定
            $this->publishOptional();
        }

        // 全view共通の変数を登録
        $this->registerViewValues();
    }

    /**
     * Register any application services.
     *
     * @return void
     */
    public function register()
    {
        // Larapack classをalias登録
        app()->booting(function () {
            $loader = AliasLoader::getInstance();
            $loader->alias('Larapack', \Uehi\Larapack\Facades\Larapack::class);
        });
        app()->singleton('larapack', function ($app) {
            return new Larapack();
        });

        // custom connectionを追加
        $this->registerConnections();

        // routeMiddlewareを登録
        $this->registerMiddleware();
        // commandを登録
        $this->registerCommands();
        // event providerを登録
        app()->register(LarapackEventServiceProvider::class);
    }

    protected function registerConnections()
    {
        foreach ($this->connections as $driver => $class) {
            app()->bind('db.connection.' . $driver, $class);
            Connection::resolverFor($driver, function ($connection, $database, $prefix, $config) use ($class) {
                return new $class($connection, $database, $prefix, $config);
            });
        }
    }

    /**
     * middlewareを登録
     */
    protected function registerMiddleware()
    {
        // register route middleware.
        foreach ($this->routeMiddleware as $key => $middleware) {
            app('router')->aliasMiddleware($key, $middleware);
        }

        // register middleware.
        foreach ($this->middlewareGroups as $key => $middlewares) {
            foreach ($middlewares as $middleware) {
                app('router')->pushMiddlewareToGroup($key, $middleware);
            }
        }
    }

    /**
     * commandを登録
     */
    protected function registerCommands()
    {
        foreach ($this->commands as $command) {
            $this->commands('Uehi\Larapack\Commands\\' . $command);
        }
    }

    /**
     * helperを登録
     */
    public function registerHelpers()
    {
        $helpers = [
            "{$this->srcDir}/helpers.php"
        ];

        foreach ($helpers as $helper) {
            if (file_exists($helper)) {
                require $helper;
            }
        }
    }

    /**
     * vendor:publishで必ず外出ししてほしいものを指定
     * php artisan vendor:publish --tag=larapack
     * ※追加したものはUninstallCommandで削除するように記述すること
     */
    protected function publishRequired()
    {
        // 設定関連 ----------------------------------------------------------------------------------------------------
        // 必要な設定ファイル
        $this->publishes(["{$this->baseDir}/config/const.php" => config_path('const.php')], 'larapack');
        // css,jsファイル関連
        $this->publishes(["{$this->baseDir}/resources/assets" => public_path('larapack')], 'larapack');
        // バリデーション言語ファイル
        $this->publishes(["{$this->baseDir}/resources/lang/ja/validation.php" => base_path('resources/lang/ja/validation.php')], 'larapack');

        // views -------------------------------------------------------------------------------------------------------
        // 管理画面metaタグ
        $this->publishes(["{$this->baseDir}/resources/views/partials/admin/_meta.blade.php" => resource_path('views/vendor/larapack/partials/admin/_meta.blade.php')], 'larapack');
        // 管理画面メニュー
        $this->publishes(["{$this->baseDir}/resources/views/partials/admin/_menu.blade.php" => resource_path('views/vendor/larapack/partials/admin/_menu.blade.php')], 'larapack');
        // admin_usersのview
        $this->publishes(["{$this->baseDir}/resources/views/admin/admin_users" => resource_path('views/admin/admin_users')], 'larapack');
    }

    /**
     * vendor:publishで外出し可能なもの
     * php artisan vendor:publish --tag=~~~
     * tag:
     *      larapack-auth: larapack-auth.phpを外出し
     *      larapack-form: larapack-form.phpを外出し
     *      larapack-views: viewを全て外出し
     */
    protected function publishOptional()
    {
        // "php artisan vendor:publish"コマンドが叩かれた場合に
        // 該当のgroupを指定されたときだけpublish実行
        $command = \Request::server('argv', null); // ["artisan", "vendor:publish", "--tag=larapack"]
        $isPublish = !empty($command[1]) && $command[1] === 'vendor:publish';
        if (!empty($command[2])) {
            if (preg_match('/^--tag=(.+)$/', $command[2], $m)) {
                if (!empty($m[1])) {
                    $tag = $m[1];
                }
            }
        }

        // publish ------
        if ($isPublish && !empty($tag)) {
            if ($tag === 'larapack-auth') {
                // larapack-auth.php
                $this->publishes(["{$this->baseDir}/config/larapack-auth.php" => config_path('larapack-auth.php')], 'larapack-auth');
            } elseif ($tag === 'larapack-file') {
                // larapack-form.php
                $this->publishes(["{$this->baseDir}/config/larapack-form.php" => config_path('larapack-form.php')], 'larapack-form');
            } elseif ($tag === 'larapack-views') {
                // 共通layoutやpartialを全て出す
                $this->publishes(["{$this->baseDir}/resources/views/layouts" => resource_path('views/vendor/larapack/layouts')], 'larapack-views');
                $this->publishes(["{$this->baseDir}/resources/views/partials" => resource_path('views/vendor/larapack/partials')], 'larapack-views');
            }
        }

    }

    /**
     * configをmerge
     */
    protected function mergeConfigs()
    {
        // larapack-auth.php は authにマージする
        // project側のconfigディレクトリにある場合そちらを読み込むようにする
        if (!file_exists(config_path('larapack-auth.php'))) {
            $this->mergeConfigRecursiveFrom("{$this->baseDir}/config/larapack-auth.php", 'auth');
        } else {
            $this->mergeConfigRecursiveFrom(config_path('larapack-auth.php'), 'auth');
        }

        // larapack-form.php は constにマージする
        // project側のconfigディレクトリにある場合そちらを読み込むようにする
        if (!file_exists(config_path('larapack-form.php'))) {
            $this->mergeConfigRecursiveFrom("{$this->baseDir}/config/larapack-form.php", 'const');
        } else {
            $this->mergeConfigRecursiveFrom(config_path('larapack-form.php'), 'const');
        }

        // values.php
        // project側のconfigディレクトリにある場合、それとmergeする形になる
        $this->mergeConfigFrom("{$this->baseDir}/config/values.php", 'values');
    }

    /**
     * View共通の変数登録
     */
    protected function registerViewValues()
    {
        View::composer('*', function($view){
            // controller名とaction名
            list($controller, $action) = get_current_route_names();
            View::share('controllerName', $controller);
            View::share('actionName', $action);

            // prefix毎の処理
            $curentRoute = Route::getCurrentRoute();
            if (!empty($curentRoute)) {
                $prefix = $curentRoute->getPrefix();

                $prefix = explode('/', $prefix);
                $prefix = $prefix[0];

                if ($prefix === 'admin') {
                    // ログインユーザ
                    View::share('auth', Auth::guard('admin')->user());
                }
            }
        });
    }

    /**
     * mergeConfigFromのrecursive版
     * @param  string  $path
     * @param  string  $key
     * @return void
     */
    protected function mergeConfigRecursiveFrom($path, $key)
    {
        $config = $this->app['config']->get($key, []);
        $this->app['config']->set($key, array_replace_recursive(require $path, $config));
    }

}
