<?php

namespace Uehi\Larapack\Extensions\Validator;

use Illuminate\Validation\Validator;
use Illuminate\Support\Arr;

class LarapackValidator extends Validator
{

    /**
     * override
     *   input type fileにかけると常にアップ存在していないとだめなので
     *   => "{column}{input_suffix}"のinputがあればtrueにする
     * @param  string  $attribute
     * @param  mixed   $value
     * @return bool
     */
    public function validateRequired($attribute, $value)
    {
        // {column}{input_suffix}のinput値がある場合fileとみなして処理
        $params = $this->getData();
        $inputColumn = $attribute . app('config')->get('const.uploads.input_suffix');
        if (!empty($params[$inputColumn])) {
            return true;
        }

        // 通常
        return parent::validateRequired($attribute, $value);
    }

    /**
     * 全角ひらがな
     * 全角ダッシュ「ー」のみ必要と考えられるので追加
     * @param $attribute
     * @param $value
     * @return bool
     */
    public function validateHiragana($attribute, $value)
    {
        return (bool) preg_match("/^(\xe3(\x81[\x81-\xbf]|\x82[\x80-\x93]|\x83\xbc))*$/", $value);
    }

    /**
     * 全角カタカナ
     * @param $attribute
     * @param $value
     * @return bool
     */
    public function validateKatakana($attribute, $value)
    {
        return (bool) preg_match("/^(\xe3(\x81[\x81-\xbf]|\x82[\x80-\x93]|\x83\xbc))*$/", $value);
    }

    /**
     * 全角ひらがな,ー，スペースのみ
     * @return boolean
     */
    public function validateHiraganaPlus($attribute, $value)
    {
        return (bool) preg_match('/^([ぁ-ん]|[　 ]|[ー])*$/u', $value);
    }

    /**
     * 全角カタカナ,ー，スペースのみ
     * @return boolean
     */
    public function validateKatakanaPlus($attribute, $value)
    {
        return (bool) preg_match('/^([ァ-ヶ]|[　 ]|[ー])*$/u', $value);
    }

    /**
     * 電話またはFAX
     * @return boolean
     */
    public function validateTel($attribute, $value)
    {
        $pattern = '/^(0\d{1,4}[\s-]?\d{1,4}[\s-]?\d{1,4}|\+\d{1,3}[\s-]?\d{1,4}[\s-]?\d{1,4}[\s-]?\d{1,4})$/';
        return (bool) preg_match($pattern, $value);
    }

    /**
     * ハイフン必須の電話またはFAX
     * @return boolean
     */
    public function validateTelHyphen($attribute, $value)
    {
        $pattern = '/^(0\d{1,4}[-]\d{1,4}[-]\d{1,4}|\+\d{1,3}[-]\d{1,4}[-]\d{1,4}[-]\d{1,4})$/';
        return (bool) preg_match($pattern, $value);
    }

    /**
     * 郵便番号
     * @return boolean
     */
    public function validateZipCode($attribute, $value)
    {
        $pattern = '/^[0-9]{3}-?[0-9]{4}$/';
        return (bool) preg_match($pattern, $value);
    }

    /**
     * Email(RFC準拠＆docomo,auの非準拠対応)
     * @return boolean
     */
    public function validateEmailJp($attribute, $value)
    {
        $pattern = '/(\\A(?:^([a-z0-9][a-z0-9_\\-\\.\\+\\?]*)@([a-z0-9][a-z0-9\\.\\-]{0,63}\\.(com|org|net|biz|info|name|net|pro|aero|coop|museum|[a-z]{2,4}))$)\\z)|(^[a-z0-9\?\.\/_-]{3,30}@(?:[a-z0-9][-a-z0-9]*\.)*(?:[a-z0-9][-a-z0-9]{0,62})\.(?:(?:[a-z]{2}\.)?[a-z]{2,4}|museum|travel)$)/i';
        return (bool) preg_match($pattern, $value);
    }

    /**
     * 複数のEmail(RFC準拠＆docomo,auの非準拠対応)
     * (カンマ区切り)
     * @return boolean
     */
    public function validateEmailMultiple($attribute, $value)
    {
        $emails = explode(',', $value);
        foreach ($emails as $email) {
            if (!$this->validateEmailJp($attribute, $email)) {
                return false;
            }
        }
        return true;
    }

    /**
     * validateDifferentのnull許容版
     * 用意されているvalidateDifferentは指定fieldがnullの場合failになるので
     * @return boolean
     */
    public function validateDifferentIfNotNull($attribute, $value, $parameters)
    {
        $this->requireParameterCount(1, $parameters, 'different_if_not_null');

        foreach ($parameters as $parameter) {
            $other = Arr::get($this->data, $parameter);

            if (!empty($value) && !empty($other) && $value === $other) {
                return false;
            }
        }

        return true;
    }

    /**
     * different_if_not_nullのvalidation messageのattributesをreplace
     *
     * @param  string  $message
     * @param  string  $attribute
     * @param  string  $rule
     * @param  array   $parameters
     * @return string
     */
    public function replaceDifferentIfNotNull($message, $attribute, $rule, $parameters)
    {
        return $this->replaceSame($message, $attribute, $rule, $parameters);
    }

    /**
     * recaptcha
     * @return boolean
     */
    public function validateReCaptcha($attribute, $value)
    {
        if($last_re_captcha = session()->has('last_re_captcha')) {
            if($value == $last_re_captcha) {
                return true;
            }
        }

        $url = 'https://www.google.com/recaptcha/api/siteverify';

        // POST送信するデータ
        $data = array(
            'secret' => config('const.re_captcha.secret'),
            'response' => $value,
            'remoteip' => $_SERVER["REMOTE_ADDR"]
        );

        // URL エンコード
        $data = http_build_query($data, "", "&");

        $headers = array(
            'Content-Type: application/x-www-form-urlencoded',
        );

        // 送信時のオプション
        $options = array('http' => array(
            'method' => 'POST',
            'content' => $data,
            'header' => implode("\r\n", $headers),
        ));

        // ストリームコンテキストを作成
        $options = stream_context_create($options);

        // file_get_contents
        $contents = file_get_contents($url, false, $options);

        $json_results = json_decode($contents, true);

        if($json_results["success"] === true) {
            session(['last_re_captcha' => $value]);
        }

        return (bool) $json_results["success"];
    }

}