<?php

namespace Uehi\Larapack\Controllers\Admin;

use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Request;
use Illuminate\Support\Facades\Route;
use Illuminate\Support\Facades\View;
use Uehi\Larapack\Facades\Larapack;
use Illuminate\Support\Str;

trait AdminCrud
{

    /**
     * model
     * @var
     */
    private $model;

    /**
     * formRequest
     * @var
     */
    private $formRequest;

    /**
     * 管理画面のsubject
     * 各controllerで上書きして使う
     * @return string
     */
    protected function getSubject()
    {
        return 'サンプル';
    }


    /**
     * 権限あるかを返す
     * 各controllerで上書きして使う
     * falseにするとアクセスできなくなる
     */
    protected function hasRole()
    {
        return true;
    }

    /**
     * formのみの画面かを指定
     * trueにするとcontroller全体として1データを編集するだけの画面になる
     * @return bool
     */
    protected function isSingle()
    {
        return false;
    }

    /**
     * 管理画面Formの設定
     * 各controllerで上書きして使う
     * @return array
     */
    protected function getFormSettings()
    {
        return [
            'inputBack' => true, // Form画面に戻るボタンを用意するか
            'inputPreview' => false, // Form画面にプレビューボタンを用意するか
        ];
    }

    /**
     * bladeを格納するdefaultのディレクトリ名
     * views/admin/{ここのディレクトリ名}
     */
    protected function viewBaseDirName()
    {
        list($controller, $action) = get_current_route_names();
        return Str::snake($controller);
    }

    /*
    |--------------------------------------------------------------------------
    | 共通処理
    |--------------------------------------------------------------------------
    */

    /**
     * get formRequest
     * @return string
     */
    protected function formRequest()
    {
        if (!empty($this->formRequest)) {
            return $this->formRequest;
        }
        // controller名からRequest名を判断:  App\Http\Requests\{Controller名の単数形}Request
        $name = preg_replace('/.*\\\|Controller/', '', get_class());
        return $this->formRequest = get_current_project_info()['request_namespace'] . '\\' . Str::singular($name) . 'Request';
    }

    /**
     * get model
     * @return string
     */
    protected function model()
    {
        if (!empty($this->model)) {
            return $this->model;
        }
        // controller名からModel名を判断: App\Models\{Controller名の単数形}
        $name = preg_replace('/.*\\\|Controller/', '', get_class());
        return $this->model = get_current_project_info()['model_namespace'] . '\\' . Str::singular($name);
    }

    /**
     * controller, action名から判断したviewを返す
     * @param array $data
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    protected function adminView($data = [])
    {
        list($controller, $action) = get_current_route_names();
        $controller = Str::snake($controller);

        // baseとなるviewのディレクトリ
        $viewBaseDirName = $this->viewBaseDirName();

        $currentProjectInfo = get_current_project_info();
        $viewBaseDir = $currentProjectInfo['view_base_dir'];

        View::share('viewBaseDir', $viewBaseDir);
        View::share('viewBaseDirName', $viewBaseDirName);

        // recordはshareにする(Form等で利用)
        if (!empty($data['record'])) {
            View::share('record', $data['record']);
            unset($data['record']);
        }

        // actionと同名のbladeファイルが用意されていたらそれを読み込む
        if (file_exists($viewBaseDir . "/{$action}.blade.php")) {
            return view("admin.{$controller}.{$action}", $data);
        }

        if (in_array($action, ['index'])) {
            // indexのform画面
            if (file_exists($viewBaseDir . '/index.blade.php')) {
                // index.blade.phpを用意していたらそれを読込
                return view("admin.{$controller}.index", $data);
            }
            // 固定layout
            return view('larapack::layouts.index', $data);
        } elseif (in_array($action, ['create', 'edit', 'show'])) {
            // create, edit, showのform画面
            if (file_exists($viewBaseDir . '/create.blade.php')) {
                // create.blade.phpを用意していたらそれを読込
                return view("admin.{$controller}.create", $data);
            }
            // 固定layout
            return view('larapack::layouts.create', $data);
        }
        $viewName = "admin.{$controller}.{$action}";
        return view($viewName, $data);
    }

    /**
     * 一覧にリダイレクト
     * @param bool $back : trueにすると一覧の検索条件を保持
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector
     */
    protected function adminRedirect($back = true)
    {
        if ($this->isSingle()) {
            return $this->singleRedirect();
        }
        list($controller, $action) = get_current_route_names();
        $controller = Str::snake($controller);
        $back = $back ? '?back=1' : '';
        return redirect(admin_url("/{$controller}{$back}"));
    }

    /**
     * isSingleの場合のリダイレクト
     * edit(データ内場合add)にリダイレクト
     */
    protected function singleRedirect()
    {
        list($controller, $action) = get_current_route_names();
        $controller = Str::snake($controller);

        // 最初の1件を取得
        $model = $this->model();
        $record = $model::first();
        if (!empty($record)) {
            return redirect(admin_url("/{$controller}/{$record->id}/edit"));
        } else {
            return redirect(admin_url("/{$controller}/create"));
        }
    }

    /**
     * requestパラメータをsessionに格納
     */
    private function poolQuery()
    {
        $params = Request::all();

        // session名
        $prefix = Route::getCurrentRoute()->getPrefix();
        $sessionName = "{$prefix}_query";

        // backフラグがある場合は除外
        if (!empty($params['back'])) {
            unset($params['back']);
        }

        // 格納
        Request::session()->put($sessionName, $params);
    }

    /**
     * 格納したsessionパラメータをrequestパラメータにする
     */
    private function restoreQuery()
    {
        // session名
        $prefix = Route::getCurrentRoute()->getPrefix();
        $sessionName = "{$prefix}_query";

        if (Request::session()->has($sessionName)) {
            Request::merge(Request::session()->get($sessionName));
        }
    }

    /*
    |--------------------------------------------------------------------------
    | 各画面actionの前処理
    |--------------------------------------------------------------------------
    */

    public function beforeProcess()
    {
        // 権限ない場合Forbidden
        if (!$this->hasRole()) {
            abort(403, 'Access denied');
        }

        // urlに利用
        list($controller, $action) = get_current_route_names();
        View::share('controllerNameSnake', Str::snake($controller));
        // subject
        View::share('subject', $this->getSubject());
        // formSettings
        // singleの場合inputBackは必ずfalse
        $settings = $this->getFormSettings();
        if ($this->isSingle()) {
            $settings['inputBack'] = false;
        }
        View::share('formSettings', $settings);
    }

    /*
    |--------------------------------------------------------------------------
    | 各画面action
    |--------------------------------------------------------------------------
    */

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $this->beforeProcess();
        $model = $this->model();

        // Formの初期値をパラメータから埋めるため
        app('form')->considerRequest(true);

        // isSingleの場合この画面は存在しない
        if ($this->isSingle()) {
            return $this->singleRedirect();
        }

        // 編集画面等から戻ったときにsessionからqueryを復元する
        if (!empty(Request::get('back'))) {
            $this->restoreQuery();
        }

        // params
        $params = Request::all();

        // query
        $query = $model::createSearchQuery($params, true);

        $limit = !empty($params['limit']) ? $params['limit'] : config('values.perPage');
        // pagination
        $records = $query->paginate($limit);

        // 編集画面等から戻るためsessionにquery格納処理
        if (!empty($params['page']) && !empty($records->lastPage()) && $params['page'] > $records->lastPage()) {
            // 結果が存在するときに、パラメータ指定のpageが存在しない場合最後のpageへリダイレクト
            Request::merge(['page' => $records->lastPage()]);
            $this->poolQuery();
            return $this->adminRedirect(true);
        }
        $this->poolQuery();

        return $this->adminView(compact('records'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $this->beforeProcess();
        $model = $this->model();
        $record = new $model;
        return $this->adminView(compact('record'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @return \Illuminate\Http\Response
     */
    public function store()
    {
        // 入力チェック
        $request = app()->make($this->formRequest());
        $this->beforeProcess();

        // save
        $msgId = $this->insertOrUpdate() ? 'larapack::admin.create_ok' : 'larapack::admin.create_ng';
        $msg = trans($msgId, ['subject' => $this->getSubject()]);
        Larapack::setFlash($msg);
        return $this->adminRedirect();
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        $this->beforeProcess();
        $model = $this->model();

        $record = $model::findOrFail($id);
        return $this->adminView(compact('record'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $this->beforeProcess();
        $model = $this->model();

        $record = $model::findOrFail($id);
        return $this->adminView(compact('record'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param $id
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector
     */
    public function update($id)
    {
        // 入力チェック
        $request = app()->make($this->formRequest());
        $this->beforeProcess();

        // save
        $msgId = $this->insertOrUpdate($id) ? 'larapack::admin.update_ok' : 'larapack::admin.update_ng';
        $msg = trans($msgId, ['subject' => $this->getSubject()]);
        Larapack::setFlash($msg);
        return $this->adminRedirect();
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        $this->beforeProcess();
        $model = $this->model();

        // delete
        $record = $model::findOrFail($id);
        $msgId = $record->delete() ? 'larapack::admin.delete_ok' : 'larapack::admin.delete_ng';
        $msg = trans($msgId, ['subject' => $this->getSubject()]);

        // redirect
        Larapack::setFlash($msg);
        return $this->adminRedirect();
    }

    /**
     * idあればupdate,なければinsertする
     * idあればupdate,なければinsertする
     * @param null $id
     * @return bool
     */
    protected function insertOrUpdate($id = null)
    {
        $model = $this->model();
        $params = Request::all();
        if (!empty($id)) {
            $record = $model::findOrFail($id);
            $record->fill($params);
        } else {
            $record = new $model($params);
        }

        // 親モデルinsert後、子モデルinsertするのでtransaction
        DB::beginTransaction();
        $success = true;
        if (!$record->save()) {
            $success = false;
        } else {
            // リレーション存在 && 同名のパラメータ存在時保存
            $relations = $record->getAllRelations();
            foreach ($relations as $prop => $val) {
                // BelongsToMany, HasManyは同時保存
                if ($val['type'] === 'BelongsToMany') {
                    if (array_key_exists($prop, $params)) {
                        $record->{$prop}()->sync($params[$prop]);
                    }
                } elseif ($val['type'] === 'HasMany') {
                    if (array_key_exists("{$prop}__exists", $params)) { // 存在確認パラメータ(by MultiBlock.php)
                        $foreignKey = $record->{$prop}()->getForeignKeyName();
                        $instances = [];
                        $ids = [];
                        if (!empty($params[$prop]) && is_array($params[$prop])) {
                            foreach ($params[$prop] as $param) {
                                $id = !empty($param['id']) ? $param['id'] : null;
                                if (!empty($id)) {
                                    $ids[] = $id;
                                }
                                $child = $record->{$prop}()->findOrNew($id);
                                $child->fill($param);
                                $instances[] = $child;
                            }
                        }

                        // 渡ってきていないIDは全て消す
                        if (!empty($ids)) {
                            $record->{$prop}()->where($foreignKey, $record->id)->whereNotIn('id', $ids)->delete();
                        } else {
                            $record->{$prop}()->where($foreignKey, $record->id)->delete();
                        }

                        // 保存
                        if (!empty($instances)) {
                            $record->{$prop}()->saveMany($instances);
                        }
                    }
                }
            }
        }
        $success ? DB::commit() : DB::rollBack();
        return $success;
    }

}
