<?php

namespace Uehi\Larapack\Commands;

use Illuminate\Database\Console\Migrations\BaseCommand;
use Uehi\Larapack\Extensions\Database\Migrations\MigrationCreator;
use Illuminate\Support\Composer;
use Illuminate\Support\Str;
use Illuminate\Support\Arr;

class JsonInstallCommand extends BaseCommand
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'larapack:jsoninstall';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Install from json';

    /**
     * The migration creator instance.
     *
     * @var \Illuminate\Database\Migrations\MigrationCreator
     */
    protected $creator;

    protected $composer;

    public $ckeditor_path = "/js/admin/";

    /**
     * Create a new command instance.
     *
     * @return void
     */
    public function __construct(Composer $composer)
    {
        parent::__construct();

        $this->creator = new MigrationCreator(app('files'));
        $this->composer = $composer;
    }

    /**
     * Execute the console command.
     *
     * @return mixed
     */
    public function handle()
    {
        $json_path = base_path("larapack.json");
        if(!file_exists($json_path)) {
            $this->error('Not found larapack.json!');
            return false;
        }

        $json_data = json_decode(file_get_contents($json_path), true);
        if(empty($json_data["modules"])) {
            $this->info('Nothing install.');
            return true;
        }

        define("MIGRATION_INDENT", "\n\t\t\t");
        define("MIGRATION_REPLACE", "// MIGRATION_REPLACE");
        define("ADMIN_LIST_TH_INDENT", "\n\t");
        define("ADMIN_LIST_TH_REPLACE", "// ADMIN_LIST_TH_REPLACE");
        define("ADMIN_LIST_TD_INDENT", "\n\t\t");
        define("ADMIN_LIST_TD_REPLACE", "// ADMIN_LIST_TD_REPLACE");
        define("ADMIN_SEARCH_INDENT", "\n\t");
        define("ADMIN_SEARCH_REPLACE", "// ADMIN_SEARCH_REPLACE");
        define("ADMIN_MENU_INDENT", "\n\t\t\t");
        define("ADMIN_MENU_REPLACE", "// ADMIN_MENU_REPLACE");
        define("ADMIN_MENU_AUTH_INDENT", "\n\t\t\t\t");
        define("ADMIN_MENU_AUTH_REPLACE", "// ADMIN_MENU_AUTH_REPLACE");
        define("ADMIN_MODEL_INDENT", "\n\t\t");
        define("ADMIN_MODEL_REPLACE", "// ADMIN_MODEL_REPLACE");
        define("MIDDLEWARE_INDENT", "\n\t\t");
        define("MIDDLEWARE_USE_REPLACE", "// MIDDLEWARE_USE_REPLACE");
        define("MIDDLEWARE_REPLACE", "// MIDDLEWARE_REPLACE");

        foreach($json_data["modules"] as $key => $module) {
            if(!is_array($module)) {
                if($module == "migrateInstall") {
                    exec("php artisan config:cache");
                    $this->call('migrate:install');
                }
                else if($module == "larapackInstall") {
                    $this->call('larapack:install');
                    exec("php artisan config:cache");

                    // const.php調整
                    $const_path = config_path('const.php');
                    $const_text = file_get_contents($const_path);
                    $const_text = str_replace("'key' => '',", "'key' => env('GMAP_KEY', ''),", $const_text);
                    file_put_contents($const_path, $const_text);

                    // app.php調整
                    $app_path = config_path('app.php');
                    $app_text = file_get_contents($app_path);
                    $app_text = str_replace("'locale' => 'en',", "'locale' => 'ja',", $app_text);
                    $app_text = str_replace("'timezone' => 'UTC',", "'timezone' => 'Asia/Tokyo',", $app_text);
                    file_put_contents($app_path, $app_text);
                }
            }
            else {
                foreach($module as $module_name => $value) {
                    if($module_name == "custom") {
                        if(!is_array($value)) {
                            $value = array($value);
                        }
        
                        // メニューの初期調整
                        $view_menu_path = resource_path("views/vendor/larapack/partials/admin/_menu.blade.php");
                        $view_menu_text = $this->createAdminMenuText($view_menu_path);
        
                        foreach($value as $table_data) {
                            if(empty($table_data["table"]) || empty($table_data["fileds"])) {
                                $this->error('No table data!');
                                continue;
                            }
        
                            $table_name = Str::plural(Str::snake($table_data["table"]));

                            // nested
                            if(!empty($table_data["nested"])) {
                                exec("composer require kalnoy/nestedset:^4.3");
                            }
        
                            // migrationファイルの作成
                            $this->createMigrationFile($table_data);

                            // PrivatePublicの場合
                            if(!empty($table_data["isPrivatePublic"])) {
                                $org_table = $table_data["table"];
                                $table_data["table"] = "Public" . $org_table;
                                $this->createMigrationFile($table_data, true);
                                $table_data["table"] = $org_table;

                                // values.php追加
                                $private_public_statuses = [["comment" => "状態", "name" => "private_public_statuses", "value" => ['0' => '編集中', '1' => '公開', '2' => '非公開']]];
                                $this->createValuesFile($private_public_statuses);
                            }

                            // commandファイルの作成
                            $this->createCommandFile($table_data);
        
                            // migrate
                            exec("php artisan migrate");
                            
                            // マスタの初期値設定
                            if(!empty($table_data["isMaster"]) && !empty($table_data["value"])) {
                                // seederファイルの作成
                                $this->createSeederFile($table_data);
            
                                // composer dump-autoload
                                $this->composer->dumpAutoloads();

                                // seed(dump-autoloadがこのプロセスでは反映されてないのでexecで別プロセスでする)
                                exec('php artisan db:seed --class ' . Str::plural($table_data["table"]) . "TableSeeder");
                            }

                            // Modelファイルの作成
                            $this->createModelFile($table_data);
                            
                            // Requestファイルの作成
                            $this->createRequestFile($table_data);
        
                            if(!empty($table_data["slug"])) {
                                // Ruleファイルの作成
                                $this->createRuleFile($table_data, $value);

                                // values.php追加
                                $slug_regex = [["comment" => "slug正規表現", "name" => "slug_regex", "value" => '/^[0-9a-zA-Z-_]+$/']];
                                $this->createValuesFile($slug_regex);
                            }

                            // Controllerファイルの作成
                            $this->createControllerFile($table_data);
        
                            // Viewファイルの作成
                            $this->createViewFile($table_data);

                            // Middlewareファイルの作成
                            $this->createMiddlewareFile($table_data);

                            // Front関連ファイルの作成
                            $this->createFrontFile($table_data);

                            // メニューの追加
                            if(empty($table_data["isMaster"])) {
                                $view_menu_text = str_replace(ADMIN_MENU_REPLACE, '<li class="{{ preg_match(\'/admin\\/' . $table_name . '/\', Request::path()) ? \'active\' : \'\' }}">
                <a href="{{ url(\'/admin/' . $table_name . '/\') }}">' . $table_data["title"] . (empty($table_data["isSingle"]) ? "一覧" : "") . '</a>
            </li>' . ADMIN_MENU_INDENT . ADMIN_MENU_REPLACE, $view_menu_text);
                            }
                            else {
                                $view_menu_text = str_replace(ADMIN_MENU_AUTH_REPLACE, '<li class="{{ preg_match(\'/admin\\/' . $table_name . '/\', Request::path()) ? \'active\' : \'\' }}">
                    <a href="{{ url(\'/admin/' . $table_name . '/\') }}">' . $table_data["title"] . (empty($table_data["isSingle"]) ? "一覧" : "") . '</a>
                </li>' . ADMIN_MENU_AUTH_INDENT . ADMIN_MENU_AUTH_REPLACE, $view_menu_text);
                            }
                        }
        
                        // メニューの最終調整
                        $view_menu_text = str_replace(ADMIN_MENU_INDENT . ADMIN_MENU_REPLACE, "", $view_menu_text);
                        $view_menu_text = str_replace(ADMIN_MENU_AUTH_INDENT . ADMIN_MENU_AUTH_REPLACE, "", $view_menu_text);
                        file_put_contents($view_menu_path, $view_menu_text);
                    }
                    else if($module_name == "values") {
                        $this->createValuesFile($value);
                    }
                    else if($module_name == "ckeditorInstall") {
                        $this->ckeditorInstall($value);
                    }
                }
            }
        }
    }

    /**
     * 管理画面メニュー調整
     */
    private function createAdminMenuText($view_menu_path) {
        $view_menu_text = file_get_contents($view_menu_path);
        $view_menu_text = str_replace("    <h3>コンテンツ</h3>\n", "", $view_menu_text);
        $view_menu_text = str_replace('<i class="fa fa-user"></i> ユーザ', '<i class="fa fa-cog"></i> 管理者向け', $view_menu_text);
        $view_menu_text = '<h3>メニュー</h3>
<ul class="nav sidebar-menu">
    <li>
        <a><i class="fa fa-newspaper-o"></i> コンテンツ <span class="fa fa-chevron-down"></span></a>
        <ul class="nav child-menu">' . ADMIN_MENU_INDENT . ADMIN_MENU_REPLACE . '
        </ul>
    </li>
</ul>
' . $view_menu_text;
        $ul_base = '<ul class="nav child-menu">';
        $ul_pos = strrpos($view_menu_text, $ul_base);
        $view_menu_text = substr_replace($view_menu_text, ADMIN_MENU_AUTH_INDENT . ADMIN_MENU_AUTH_REPLACE, $ul_pos + strlen($ul_base), 0);
        return $view_menu_text;
    }

    /**
     * マイグレーションファイル作成
     */
    private function createMigrationFile($table_data, $isPrivatePublic = false) {
        $file_info = $this->writeMigration($table_data["table"]);
        $file_path = $file_info["dirname"] . "/" . $file_info["basename"];
        $file_text = file_get_contents($file_path);
        $file_text = str_replace("\$table->string('title');", MIGRATION_REPLACE, $file_text);
        
        foreach($table_data["fileds"] as $field => $field_info) {
            if($field_info["type"] != "belongsToMany") {
                $migrate_text = $this->createMigrateText($field, $field_info);
                $file_text = str_replace(MIGRATION_REPLACE, $migrate_text . MIGRATION_INDENT . MIGRATION_REPLACE, $file_text);
            }
            else {
                $kakko_base = "});";
                $kakko_pos = strrpos($file_text, $kakko_base);
                $snake_main = Str::snake($table_data["table"]);
                $snake_relation = Str::snake($field);
                $file_text = substr_replace($file_text, "\n\n\t\t" .  "// " . (!empty($field_info["comment"]) ? $field_info["comment"] : "")  . "中間テーブル
        Schema::create('" . $snake_main . "_" . $snake_relation . "', function (Blueprint \$table) {
            \$table->integer('" . $snake_main . "_id')->unsigned()->index('" . $snake_main . "_id', '" . $snake_main . "_id');
            \$table->foreign('" . $snake_main . "_id', '" . $snake_main . "_id_foreign')->references('id')->on('" . Str::plural($snake_main) . "')->onDelete('cascade');
            \$table->integer('" . $snake_relation . "_id')->unsigned()->index('" . $snake_relation . "_id', '" . $snake_relation . "_id');
            \$table->foreign('" . $snake_relation . "_id', '" . ($isPrivatePublic ? $snake_main . "_" : "") . $snake_relation . "_id_foreign')->references('id')->on('" . Str::plural($snake_relation) . "')->onDelete('cascade');
        });", $kakko_pos + strlen($kakko_base), 0);

                $drop_base = "Schema::dropIfExists('" . Str::plural($snake_main) . "');";
                $drop_pos = strrpos($file_text, $drop_base);
                $set_off = "DB::statement('SET FOREIGN_KEY_CHECKS = 0');";
                if(strpos($file_text, $set_off) === false) {
                    $file_text = substr_replace($file_text, $set_off . "\n\t\t", $drop_pos, 0);
                    // 前に文字を追加したので再検索
                    $drop_pos = strrpos($file_text, $drop_base);
                }
                $set_on = "DB::statement('SET FOREIGN_KEY_CHECKS = 1');";
                if(strpos($file_text, $set_on) === false) {
                    $file_text = substr_replace($file_text, "\n\t\t" . $set_on, $drop_pos + strlen($drop_base), 0);
                }
                $set_on_pos = strpos($file_text, $set_on);
                $file_text = substr_replace($file_text, "Schema::dropIfExists('" . $snake_main . "_" . $snake_relation . "');" . "\n\t\t", $set_on_pos, 0);
            }
        }

        // メール認証トークン用フィールドとrememberTokenの追加
        if(!empty($table_data["front"]) && ($table_data["front"]["type"] == "user_form")) {
            $file_text = str_replace(MIGRATION_REPLACE, "\$table->string('email_verification_token')->nullable()->comment('メール認証トークン');" . MIGRATION_INDENT . MIGRATION_REPLACE, $file_text);
            $file_text = str_replace(MIGRATION_REPLACE, "\$table->rememberToken();" . MIGRATION_INDENT . MIGRATION_REPLACE, $file_text);
        }
        
        // 並び順の追加
        if(!empty($table_data["hasSequence"])) {
            $file_text = str_replace(MIGRATION_REPLACE, "\$table->unsignedInteger('sequence')->nullable()->comment('並び順');" . MIGRATION_INDENT . MIGRATION_REPLACE, $file_text);
        }

        // ステータスの追加
        if(!empty($table_data["isPrivatePublic"])) {
            $file_text = str_replace(MIGRATION_REPLACE, "\$table->unsignedInteger('private_public_status')->default(0)->comment('状態');" . MIGRATION_INDENT . MIGRATION_REPLACE, $file_text);
        }

        // slugの追加
        if(!empty($table_data["slug"]) && ($table_data["slug"] !== "no-form")) {
            $file_text = str_replace(MIGRATION_REPLACE, "\$table->string('slug')->default('')->comment('slug');" . MIGRATION_INDENT . MIGRATION_REPLACE, $file_text);
        }

        // nestedの追加
        if(!empty($table_data["nested"])) {
            $file_text = str_replace("use Illuminate\Database\Migrations\Migration;", "use Illuminate\Database\Migrations\Migration;\nuse Kalnoy\Nestedset\NestedSet;", $file_text);
            $file_text = str_replace(MIGRATION_REPLACE, "NestedSet::columns(\$table);" . MIGRATION_INDENT . MIGRATION_REPLACE, $file_text);
        }

        $file_text = str_replace(MIGRATION_INDENT . MIGRATION_REPLACE, "", $file_text);
        
        if($table_data["table"] == "User") {
            $create_base = "Schema::create(";
            $create_pos = strpos($file_text, $create_base);
            $file_text = substr_replace($file_text, "Schema::dropIfExists('users');\n\t\t", $create_pos, 0);

            $drop_base = "Schema::dropIfExists('users');";
            $drop_pos = strrpos($file_text, $drop_base);
            $file_text = substr_replace($file_text, "\n\t\t" . "Schema::create('users', function (Blueprint \$table) {
            \$table->increments('id');
            \$table->string('name');
            \$table->string('email')->unique();
            \$table->string('password');
            \$table->rememberToken();
            \$table->timestamps();
        });", $drop_pos + strlen($drop_base), 0);
        }

        file_put_contents($file_path, $file_text);
    }

    /**
     * Write the migration file to disk.
     *
     * @param  string  $name
     * @return string
     */
    protected function writeMigration($name)
    {
        // $name(モデル名)からmigration名を決める
        $table = Str::plural(Str::snake($name));
        $filename = "create_{$table}_table";
        if($table == "users") {
            $filename = "re_" . $filename;
        }

        $file = pathinfo($this->creator->create(
            $filename, $this->getMigrationPath(), $table, false
        ));

        //$this->line("<info>Created Migration:</info> {$file}");
        return $file;
    }

    /**
     * マイグレーションテキスト取得
     */
    private function createMigrateText($field, $field_info) {
        // 位置情報
        if($field_info["type"] == "map") {
            $ret = "\$table->double('lat')->nullable()->comment('緯度');
            \$table->double('lon')->nullable()->comment('経度');
            \$table->unsignedSmallInteger('zoom')->nullable()->comment('ズーム');";
            return $ret;
        }

        $ret = "\$table->";
        switch($field_info["type"]) {
            case "string":
            case "password":
            case "imageFile":
            case "documentFile":
            case "email":
            case "katakana":
            case "hiragana":
            case "tel":
            case "zip":
            case "permalink":
                $ret .= "string";
                break;

            case "select":
            case "radio":
            case "checkbox":
                $ret .= "unsignedInteger";
                break;

            case "text":
                $ret .= "text";
                break;

            case "EmailMultiple":
                $ret .= "text";
                $field = Str::snake($field) . "_mails";
                break;

            case "selectDate":
                $ret .= "date";
                break;

            case "textDate":
                if(empty($field_info["time"])) {
                    $ret .= "date";
                }
                else {
                    $ret .= "timestamp";
                }
                break;

            case "nestedSelect":
                $ret .= "unsignedInteger";
                $field = Str::snake($field) . "_id";
                break;

            default:
                $ret .= $field_info["type"];
                break;
        }

        $ret .= "('{$field}')";

        if(!empty($field_info["nullable"])) {
            $ret .= "->nullable()";
        }

        if(isset($field_info["default"])) {
            if(is_integer($field_info["default"])) {
                $ret .= "->default(" . $field_info["default"] . ")";
            }
            else if((in_array($field_info["type"], ["selectDate", "textDate"])) && ($field_info["default"] == "today")) {
                $ret .= "";
            }
            else {
                $ret .= "->default('" . $field_info["default"] . "')";
            }
        }

        if(!empty($field_info["comment"])) {
            $ret .= "->comment('" . $field_info["comment"] . "')";
        }

        $ret .= ";";

        return $ret;
    }

    /**
     * コマンド作成
     */
    private function createCommandFile($table_data) {
        // Commandsフォルダ作成
        if(!file_exists(app_path("Console/Commands"))) {
            $this->laravel['files']->makeDirectory(app_path("Console/Commands"), 0755, true);
        }

        if(!empty($table_data["isPrivatePublic"])) {
            $command_class = "ChangeStatusCommand";
            $command_path = app_path("Console/Commands/{$command_class}.php");
            if(!file_exists($command_path)) {
                $command_text = <<<EOT
<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;

class ChangeStatusCommand extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected \$signature = 'custom:change_status {model} {id} {field} {flag}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected \$description = 'Change private or public';

    /**
     * Create a new command instance.
     *
     * @return void
     */
    public function __construct()
    {
        parent::__construct();
    }

    /**
     * Execute the console command.
     *
     * @return mixed
     */
    public function handle()
    {
        \$model = \$this->argument('model');
        \$id = \$this->argument('id');
        \$field = \$this->argument('field');
        \$flag = \$this->argument('flag');
        
        \$record = \$model::findOrFail(\$id);
        \$record->{\$field} = \$flag;
        if(\$flag > 0) {
            \$record->public = \$flag;
        }
        \$isSuccess = \$record->save();

        \$relations = \$record->getAllRelations();
        \$record = \$model::findOrFail(\$id)->with(array_keys(\$relations))->first();

        if(\$isSuccess && (\$flag > 0)) {
            \$model_strings = explode('\\\', \$model);
            \$model_strings[count(\$model_strings) - 1] = 'Public' . \$model_strings[count(\$model_strings) - 1];
            \$public_model = implode('\\\', \$model_strings);

            \$current_public_data = \$public_model::find(\$id);
            if(!empty(\$current_public_data)) {
                if(!empty(\$current_public_data->getUploadColumns())) {
                    foreach(\$current_public_data->getUploadColumns() as \$column) {
                        if (file_exists(public_path(\$current_public_data->\$column))) {
                            unlink(\$current_public_data->\$column);
                        }
                    }
                }
                \$current_public_data->forceDelete();
            }

            if(\$flag == 1) {
                \$public_data = \$record->toArray();
                \$relation_params = [];
                foreach (\$relations as \$prop => \$val) {
                    if (\$val['type'] === 'BelongsToMany') {
                        \$tmp = [];
                        foreach(\$public_data[\$prop] as \$tmp_val) {
                            \$tmp[] = \$tmp_val['id'];
                        }
                        \$relation_params[\$prop] = \$tmp;
                    }
                    else {
                        \$relation_params[\$prop] = \$public_data[\$prop];
                    }
                    
                    unset(\$public_data[\$prop]);
                }

                \$public_record = new \$public_model();
                if(!empty(\$public_record->getUploadColumns())) {
                    \$public_path = false;
                    foreach(\$public_record->getUploadColumns() as \$column) {
                        if(!empty(\$public_data[\$column])) {
                            if(empty(\$public_path)) {
                                \$public_path = dirname(\$public_data[\$column]) . "/public/";
                                if (!file_exists(public_path('/' . \$public_path))) {
                                    \File::makeDirectory(public_path('/' . \$public_path), 0707, true);
                                }
                            }
                            
                            if (copy(public_path('/' . \$public_data[\$column]), public_path('/' . \$public_path . basename(\$public_data[\$column])))) {
                                chmod(public_path('/' . \$public_path . basename(\$public_data[\$column])), 0707);
                            }

                            \$public_data[\$column] = \$public_path . basename(\$public_data[\$column]);
                        }
                    }
                }

                \$public_model::insert(\$public_data);

                \$public_record = \$public_model::find(\$id);

                foreach (\$relations as \$prop => \$val) {
                    if (\$val['type'] === 'BelongsToMany') {
                        if (array_key_exists(\$prop, \$relation_params)) {
                            \$public_record->{\$prop}()->sync(\$relation_params[\$prop]);
                        }
                    } elseif (\$val['type'] === 'HasMany') {
                        \$foreignKey = \$public_record->{\$prop}()->getForeignKeyName();
                        \$instances = [];
                        \$ids = [];
                        if (!empty(\$relation_params[\$prop]) && is_array(\$relation_params[\$prop])) {
                            foreach (\$relation_params[\$prop] as \$param) {
                                \$id = !empty(\$param['id']) ? \$param['id'] : null;
                                if (!empty(\$id)) {
                                    \$ids[] = \$id;
                                }
                                \$child = \$public_record->{\$prop}()->findOrNew(\$id);
                                \$child->fill(\$param);
                                \$instances[] = \$child;
                            }
                        }

                        // 渡ってきていないIDは全て消す
                        if (!empty(\$ids)) {
                            \$public_record->{\$prop}()->where(\$foreignKey, \$public_record->id)->whereNotIn('id', \$ids)->delete();
                        } else {
                            \$public_record->{\$prop}()->where(\$foreignKey, \$public_record->id)->delete();
                        }

                        // 保存
                        if (!empty(\$instances)) {
                            \$public_record->{\$prop}()->saveMany(\$instances);
                        }
                    }
                }
            }
        }

        if(!\$isSuccess) {
            \$this->line('error');
        }
    }
}
EOT;
                file_put_contents($command_path, $command_text);
            }
        }
    }

    /**
     * シーダー作成
     */
    private function createSeederFile($table_data) {
        $seeder_name = Str::plural($table_data["table"]) . "TableSeeder";
        $this->call('make:seeder', ["name" => $seeder_name]);

        $seeder_path = database_path("seeds/" . $seeder_name . ".php");
        $seeder_text = file_get_contents($seeder_path);
        $replace_text = "";
        foreach($table_data["value"] as $value) {
            $replace_text .= "DB::table('" . Str::plural(Str::snake($table_data["table"])) . "')->insert([
            'title' => '" . $value . "',
            'public' => 1,
        ]);
        ";
        }
        $seeder_text = str_replace("//", $replace_text, $seeder_text);
        file_put_contents($seeder_path, $seeder_text);
    }

    /**
     * モデル作成
     */
    private function createModelFile($table_data) {
        $this->call('larapackmake:model', ['name' => $table_data["table"]]);

        $model_path = app_path("Models/" . $table_data["table"] . ".php");
        $model_text = file_get_contents($model_path);

        $kakko_base = "}";
        $kakko_pos = strrpos($model_text, $kakko_base);
        $model_text = substr_replace($model_text, "\t" . ADMIN_MODEL_REPLACE . "\n\n", $kakko_pos, 0);

        if(strpos($model_text, "createSearchQuery as createSearchQueryDefault;") === false) {
            $model_text = str_replace("use LarapackModel, ", "use LarapackModel {
        createSearchQuery as createSearchQueryDefault;
    }
    use ", $model_text);
        }

        if(strpos($model_text, "public static function createSearchQuery") === false) {
            $model_text = str_replace(ADMIN_MODEL_REPLACE, '/**
     * override
     * @param array $params
     * @param bool $admin
     * @return mixed
     */
    public static function createSearchQuery($params = [], $admin = false)
    {
        // query
        $query = static::createSearchQueryDefault($params, $admin);

        return $query;
    }' . "\n\n\t" . ADMIN_MODEL_REPLACE, $model_text);
        }

        if(strpos($model_text, "public static function defaultOrder") === false) {
            $order = "['id' => '" . (empty($table_data["isMaster"]) ? "desc" : "asc") . "']";
            if(!empty($table_data["hasSequence"])) {
                $order = "'sequence IS NULL ASC, sequence DESC, id ASC'";
            }
            $model_text = str_replace(ADMIN_MODEL_REPLACE, "/**
     * default order
     * @return array
     */
    public static function defaultOrder()
    {
        return " . $order . ";
    }" . "\n\n\t" . ADMIN_MODEL_REPLACE, $model_text);
        }

        $upload_colums = [];
        $email_multiple_colums = [];
        $hidden_colums = [];
        foreach($table_data["fileds"] as $field => $field_info) {
            if($field_info["type"] == "password") {
                $field_base = ", '" . $field . "'";
                $field_pos = strrpos($model_text, $field_base);
                if($field_pos !== false) {
                    $model_text = substr_replace($model_text, "", $field_pos, strlen($field_base));
                }
                else {
                    $field_base = "'" . $field . "', ";
                    $field_pos = strrpos($model_text, $field_base);
                    if($field_pos !== false) {
                        $model_text = substr_replace($model_text, "", $field_pos, strlen($field_base));
                    }
                    else {
                        $field_base = "'" . $field . "'";
                        $field_pos = strrpos($model_text, $field_base);
                        if($field_pos !== false) {
                            $model_text = substr_replace($model_text, "", $field_pos, strlen($field_base));
                        }
                    }
                }

                $hidden_colums[] = $field;
            }

            if(!empty($field_info["dispSearch"])) {
                $condition_text = "";
                if(!empty($field_info["comment"])) {
                    $condition_text .= "// " . $field_info["comment"] . ADMIN_MODEL_INDENT;
                }
                if(in_array($field_info["type"], ["select", "radio"])) {
                    $condition_text .= "if (isset(\$params['" . $field . "']) && is_numeric(\$params['" . $field . "'])) {
            \$query->where('" . $field . "', \$params['" . $field . "']);
        }\n" . ADMIN_MODEL_INDENT;
                }
                else if(in_array($field_info["type"], ["selectDate", "textDate"])) {
                    $condition_text .= "if (!empty(\$params['start_" . $field . "'])) {
            \$query->where('" . $field . "', '>=', \$params['start_" . $field . "']);
        }
        if (!empty(\$params['end_" . $field . "'])) {
            \$query->where('" . $field . "', '<=', \$params['end_" . $field . "']);
        }\n" . ADMIN_MODEL_INDENT;
                }
                else if($field_info["type"]) {
                    $condition_text .= "if (!empty(\$params['" . Str::snake($field) . "_id'])) {
            if(\$admin) {
                \$parent = " . $field . "::find(\$params['" . Str::snake($field) . "_id']);
                \$children = \$parent->descendants()->pluck('id')->toArray();
                \$params['" . Str::snake($field) . "_id'] = array_merge([\$params['" . Str::snake($field) . "_id']], \$children);
            }
            if (is_array(\$params['" . Str::snake($field) . "_id'])) {
                \$query->whereIn('" . Str::snake($field) . "_id', \$params['" . Str::snake($field) . "_id']);
            } else {
                \$query->where('" . Str::snake($field) . "_id', \$params['" . Str::snake($field) . "_id']);
            }
        }\n" . ADMIN_MODEL_INDENT;
                }
                $model_text = str_replace('return $query;', $condition_text . 'return $query;', $model_text);
            }

            if(!empty($field_info["hasOne"])) {
                $model_text = str_replace(ADMIN_MODEL_REPLACE, '/**
     * ' . $field_info["comment"] . '
     */
    public function ' . Str::snake($field_info["hasOne"]) . '()
    {
        return $this->hasOne(\'App\Models\\' . $field_info["hasOne"] . '\');
    }' . "\n\n\t" . ADMIN_MODEL_REPLACE, $model_text);
            }

            if($field_info["type"] == "belongsToMany") {
                $snake_main = Str::snake($table_data["table"]);
                $snake_relation = Str::snake($field);
                $model_text = str_replace(ADMIN_MODEL_REPLACE, '/**
     * ' . $field_info["comment"] . '
     */
    public function ' . Str::plural(Str::snake($field)) . '()
    {
        return $this->belongsToMany(\'App\Models\\' . $field . '\', \'' . $snake_main . '_' . $snake_relation . '\');
    }' . "\n\n\t" . ADMIN_MODEL_REPLACE, $model_text);
            }

            if(!empty($field_info["nestedSelect"])) {
                $model_text = str_replace(ADMIN_MODEL_REPLACE, '/**
     * ' . $field_info["comment"] . '
     */
    public function ' . Str::snake($field) . '()
    {
        return $this->belongsTo(\'App\Models\\' . $field . '\', \'' . Str::snake($field) . '_id\');
    }' . "\n\n\t" . ADMIN_MODEL_REPLACE, $model_text);
            }

            if(in_array($field_info["type"], ["imageFile", "documentFile"])) {
                $upload_colums[] = $field;
            }

            if(in_array($field_info["type"], ["EmailMultiple"])) {
                $field_name = Str::snake($field) . "_mails";
                $email_multiple_colums[] = $field_name;

                $model_text = str_replace(ADMIN_MODEL_REPLACE, '/**
     * ' . $field_info["comment"] . '
     */
    public static function get' . $field . 'Mails()
    {
        if (!$record = static::first()) {
            return [];
        }
        return explode(\',\', $record->' . $field_name . ');
    }' . "\n\n\t" . ADMIN_MODEL_REPLACE, $model_text);
            }
        }

        if(!empty($upload_colums)) {
            $model_text = str_replace(ADMIN_MODEL_REPLACE, '/**
     * アップロード対象のcolumnをget
     * @return array
     */
    public function getUploadColumns()
    {
        return [\'' . implode("', '", $upload_colums) . '\'];
    }' . "\n\n\t" . ADMIN_MODEL_REPLACE, $model_text);

            $model_text = str_replace("use Uehi\Larapack\Models\LarapackModel;", "use Uehi\Larapack\Models\LarapackModel;\nuse Uehi\Larapack\Models\FileUploads;", $model_text);
            $model_text = str_replace("use SoftDeletes", "use SoftDeletes, FileUploads", $model_text);
        }

        if(!empty($email_multiple_colums)) {
            $model_text = str_replace(ADMIN_MODEL_REPLACE, '/**
     * boot
     */
    protected static function boot()
    {
        parent::boot();

        // 保存時
        self::saving(function($record) {
            // 改行スペース削除
            foreach ([\'' . implode("', '", $email_multiple_colums) . '\'] as $column) {
                if (!empty($record->{$column})) {
                    $record->{$column} = preg_replace(array(\'/\\r\\n/\',\'/\\r/\',\'/\\n/\'), \'\', $record->{$column});
                    $record->{$column} = preg_replace(\'/(\\t|\\s|　)/\', \'\', $record->{$column});
                }
            }
            return true;
        });
    }' . "\n\n\t" . ADMIN_MODEL_REPLACE, $model_text);
        }
        else if(!empty($hidden_colums)) {
            $replace_text = '/**
     * boot
     */
    protected static function boot()
    {
        parent::boot();

        // 保存時パスワードのhash化
        static::saving(function ($record) {
            ';
            foreach($hidden_colums as $hidden_colum) {
                $replace_text .= 'if (!empty($record->' . $hidden_colum . ')) {
                // ' . $hidden_colum . 'がsetされており現状と異なる場合パスワード変更: hash化する
                if ($record->isDirty(\'' . $hidden_colum . '\')) {
                    $record->' . $hidden_colum . ' = Hash::make($record->' . $hidden_colum . ');
                }
            } elseif ($record->isDirty(\'' . $hidden_colum . '\')) {
                // ' . $hidden_colum . 'にnull変更設定はありえないので変更なしにする
                $record->' . $hidden_colum . ' = $record->getOriginal(\'' . $hidden_colum . '\');
            }
';
            }
            $replace_text .= '            
            return true;
        });
    }';

            $model_text = str_replace(ADMIN_MODEL_REPLACE, $replace_text . "\n\n\t" . ADMIN_MODEL_REPLACE, $model_text);

            $model_text = str_replace("use Uehi\Larapack\Models\LarapackModel;", "use Uehi\Larapack\Models\LarapackModel;\nuse Illuminate\Support\Facades\Hash;", $model_text);
        }

        if(!empty($hidden_colums)) {
            $kakko_first_base = "];";
            $kakko_first_pos = strpos($model_text, $kakko_first_base);
            $model_text = substr_replace($model_text, "\n\n\t" . '/**
     * The attributes that should be hidden for arrays.
     *
     * @var array
     */
    protected $hidden = [
        \'' . implode("', '", $hidden_colums) . '\',
    ];', $kakko_first_pos + strlen($kakko_first_base), 0);
        }

        if(!empty($table_data["nested"])) {
            $model_text = str_replace(ADMIN_MODEL_REPLACE, '/**
     * 配列から全保存
     * @param $params
     * @return mixed
     */
    public static function saveAll($params)
    {
        // 新規のものはid="new連番"となっているので削除
        foreach ($params as $parentKey => $parent) {
            if (!is_numeric($parent[\'id\'])) {
                unset($params[$parentKey][\'id\']);
            }
            if (!empty($parent[\'children\'])) {
                foreach ($parent[\'children\'] as $childKey => $child) {
                    if (!is_numeric($child[\'id\'])) {
                        unset($params[$parentKey][\'children\'][$childKey][\'id\']);
                    }
                }
            }
        }

        return self::rebuildTree($params, true);
    }

    /**
     * ツリー構造で全データ取得
     * @param bool $admin
     * @return mixed
     */
    public static function getPublicTree($admin = false)
    {
        $query = static::with(\'ancestors\')->defaultOrder();
        if (!$admin) {
            $query->where(\'public\', 1);
        }
        $nodes = $query->get()->toTree()->toArray();
' . (!empty($table_data["slug"]) ? '
        // 親slugもつなげたpathを各要素に格納
        $traverse = function (&$categories, $parentPath = \'\') use (&$traverse) {
            foreach ($categories as &$category) {
                $path = \'\';
                if (!empty($parentPath)) $path .= "{$parentPath}/";
                $path .= $category[\'slug\'];

                $category[\'path\'] = $path;

                $traverse($category[\'children\'], $path);
            }
        };
        $traverse($nodes);
' : '') . '
        return $nodes;
    }

    /**
     * 全データのある項目を配列で求める
     * @param $column : slugとかtitleとか
     * @param bool $admin
     * @return array : 以下$column=slugの場合
     *      [\'親id1\' => [\'親id1\' => \'親slug1\'], \'子id1\' => [\'親id1\' => \'親slug1\', \'子id1\', => \'子slug1\'], ...]
     *      全てのID毎に、親から順のslug配列(自分も含む)を返す
     */
    public static function getPublicColumnArray($column, $admin = false)
    {
        $nodes = static::getPublicTree($admin);

        $ret = [];
        $traverse = function ($categories, $parentColumnArr = []) use (&$ret, $column, &$traverse) {
            foreach ($categories as $category) {
                $columnArr = [];
                if (!empty($parentColumnArr)) {
                    $columnArr = $parentColumnArr;
                }
                $columnArr[$category[\'id\']] = $category[$column];
                $ret[$category[\'id\']] = $columnArr;

                $traverse($category[\'children\'], $columnArr);
            }
        };
        $traverse($nodes);

        return $ret;
    }' . "\n\n\t" . ADMIN_MODEL_REPLACE, $model_text);

            $model_text = str_replace("use Uehi\Larapack\Models\LarapackModel;", "use Uehi\Larapack\Models\LarapackModel;\nuse Kalnoy\Nestedset\NodeTrait;", $model_text);
            $model_text = str_replace("use SoftDeletes", "use SoftDeletes, NodeTrait", $model_text);
            $model_text = str_replace(", '_lft', '_rgt', 'parent_id'", ", '_lft', '_rgt', 'parent_id', 'public'", $model_text);
        }

        if(!empty($table_data["front"]) && ($table_data["front"]["type"] == "user_form")) {
            $model_text = str_replace(", 'email_verification_token'", "", $model_text);
            $model_text = str_replace("'email_verification_token', ", "", $model_text);
            $model_text = str_replace("'email_verification_token'", "", $model_text);
            $model_text = str_replace(", 'remember_token'", "", $model_text);
            $model_text = str_replace("'remember_token', ", "", $model_text);
            $model_text = str_replace("'remember_token'", "", $model_text);
        }

        if(!empty($table_data["hasSequence"])) {
            $model_text = str_replace(", 'sequence'", "", $model_text);
            $model_text = str_replace("'sequence', ", "", $model_text);
            $model_text = str_replace("'sequence'", "", $model_text);
        }

        if(!empty($table_data["isPrivatePublic"])) {
            $model_text = str_replace(", 'private_public_status'", "", $model_text);
            $model_text = str_replace("'private_public_status', ", "", $model_text);
            $model_text = str_replace("'private_public_status'", "", $model_text);
        }

        $model_text = str_replace("\n\n\t" . ADMIN_MODEL_REPLACE, "", $model_text);
        file_put_contents($model_path, $model_text);

        if(!empty($table_data["isPrivatePublic"])) {
            $public_model_path = app_path("Models/Public" . $table_data["table"] . ".php");
            $public_model_text = $model_text;
            $public_model_text = str_replace("class " . $table_data["table"] . " extends Model", "class Public" . $table_data["table"] . " extends Model", $public_model_text);
            foreach($table_data["fileds"] as $field => $field_info) {
                if($field_info["type"] == "belongsToMany") {
                    $snake_main = Str::snake($table_data["table"]);
                    $snake_relation = Str::snake($field);
                    $public_model_text = str_replace('\'' . $snake_main . '_' . $snake_relation . '\'', '\'public_' . $snake_main . '_' . $snake_relation . '\'', $public_model_text);
                }
            }
            file_put_contents($public_model_path, $public_model_text);
        }
    }

    /**
     * リクエスト作成
     */
    private function createRequestFile($table_data) {
        // nestedの場合はいらない
        if(!empty($table_data["nested"])) {
            return;
        }

        $this->call('larapackmake:request', ['name' => $table_data["table"]]);

        $request_path = app_path("Http/Requests/" . $table_data["table"] . "Request.php");
        $request_text = file_get_contents($request_path);
        $email_multiple_colums = [];
        foreach($table_data["fileds"] as $field => $field_info) {
            $skip_validation_text = false;
            if($field_info["type"] == "belongsToMany") {
                continue;
            }
            else if(($field == "slug") && !empty($table_data["slug"]) && ($table_data["slug"] === "no-form")) {
                $skip_validation_text = true;
            }

            if(in_array($field_info["type"], ["EmailMultiple"])) {
                $field = Str::snake($field) . "_mails";
                $email_multiple_colums[] = $field;
            }
            else if($field_info["type"] == "nestedSelect") {
                $field = Str::snake($field) . "_id";
            }

            if(!$skip_validation_text) {
                $validation_base = "'{$field}' => ''";
                if($field_info["type"] == "map") {
                    $validation_base = "'lat' => '',
            'lon' => '',
            'zoom' => ''";
                }
                $validation_pos = strpos($request_text, $validation_base);
                $request_text = substr_replace($request_text, $this->createValidationText($field, $field_info, $table_data), $validation_pos, strlen($validation_base));
            }

            if(!empty($field_info["comment"])) {
                $attribute_base = "'{$field}' => ''";
                if($field_info["type"] == "map") {
                    $attribute_base = "'lat' => '',
            'lon' => '',
            'zoom' => ''";
                }
                $attribute_pos = strrpos($request_text, $attribute_base);
                $replace_text = "'{$field}' => '" . $field_info["comment"] . "'";
                if($field_info["type"] == "map") {
                    $replace_text = "'lat' => '緯度',
            'lon' => '経度',
            'zoom' => 'ズーム'";
                }
                $request_text = substr_replace($request_text, $replace_text, $attribute_pos, strlen($attribute_base));
            }
        }

        if(!empty($email_multiple_colums)) {
            $kakko_base = "}";
            $kakko_pos = strrpos($request_text, $kakko_base);
            $request_text = substr_replace($request_text, '/**
     * requestをvalidation前に加工
     * @return array
     */
    public function validationData()
    {
        $data = parent::all();
        // 改行スペース削除
        foreach ([\'' . implode("', '", $email_multiple_colums) . '\'] as $column) {
            if (!empty($data[$column])) {
                $data[$column] = preg_replace(array(\'/\r\n/\',\'/\r/\',\'/\n/\'), \'\', $data[$column]);
                $data[$column] = preg_replace(\'/(\t|\s|　)/\', \'\', $data[$column]);
            }
        }
        return $data;
    }', $kakko_pos, 0);
        }

        $messages = [];
        if(!empty($table_data["front"]["options"]["privacy"])) {
            $kakko_first_base = "];";
            $kakko_first_pos = strpos($request_text, $kakko_first_base);
            $request_text = substr_replace($request_text, "\t'agree' => 'sometimes|accepted',\n\t\t", $kakko_first_pos, 0);

            $messages[] = "'agree.accepted' => '同意してください。',";
        }

        if(!empty($table_data["front"]["options"]["reCaptcha"])) {
            $kakko_first_base = "];";
            $kakko_first_pos = strpos($request_text, $kakko_first_base);
            $request_text = substr_replace($request_text, "\t'g-recaptcha-response' => 'required|sometimes|ReCaptcha',\n\t\t", $kakko_first_pos, 0);

            $messages[] = "'g-recaptcha-response.required' => 'チェックを入れてください。',";
            $messages[] = "'g-recaptcha-response.re_captcha' => 'チェック結果が確認できませんでした。',";
        }

        // slug有だが、nestedでは無い場合
        if(!empty($table_data["slug"]) && empty($table_data["nested"])) {
            $replace_text = "'slug' => [
                'required',
                'regex:' . config('values.slug_regex'),
                new SlugUnique('" . Str::plural(Str::snake($table_data["table"])) . "', \$id)
            ]";
            $kakko_first_base = "'slug' => ''";
            $kakko_first_pos = strpos($request_text, $kakko_first_base);
            $request_text = substr_replace($request_text, $replace_text, $kakko_first_pos, strlen($kakko_first_base));

            $request_text = str_replace("use Illuminate\Support\Facades\Route;", "use Illuminate\Support\Facades\Route;\nuse App\Rules\SlugUnique;", $request_text);
        }

        if(!empty($messages)) {
            $kakko_last_base = "}";
            $kakko_last_pos = strrpos($request_text, $kakko_last_base);
            $request_text = substr_replace($request_text, "\t/**
     * @return array
     */
    public function messages() {
        return [
            " . implode("\n", $messages) . "
        ];
    }\n\n", $kakko_last_pos, 0);
        }

        file_put_contents($request_path, $request_text);
    }

    /**
     * バリデーションテキスト取得
     */
    private function createValidationText($field, $field_info, $table_data) {
        // 位置情報
        if($field_info["type"] == "map") {
            $ret = "'lat' => 'sometimes|nullable|numeric',
            'lon' => 'sometimes|nullable|numeric',
            'zoom' => 'sometimes|nullable|integer'";
            return $ret;
        }

        $validations = [];
        if((empty($field_info["validate"]) || (!in_array("create_required", $field_info["validate"]))) &&
            (isset($field_info["default"]) || (!isset($field_info["default"]) && empty($field_info["nullable"])))) {
            $validations[] = "required";
        }

        if(!empty($field_info["nullable"])) {
            $validations[] = "sometimes";
            $validations[] = "nullable";
        }
        
        if(in_array($field_info["type"], ["selectDate"])) {
            $validations[] = "date";
        }
        else if(in_array($field_info["type"], ["textDate"])) {
            if(empty($field_info["time"])) {
                $validations[] = "date";
            }
            else {
                $validations[] = "date_format:Y-m-d H:i:s";
            }
        }

        if(in_array($field_info["type"], ["imageFile"])) {
            $validations[] = "mimes:jpeg,gif,png";
        }

        if(in_array($field_info["type"], ["email"])) {
            $validations[] = "email";
        }

        if(in_array($field_info["type"], ["katakana"])) {
            $validations[] = "KatakanaPlus";
        }

        if(in_array($field_info["type"], ["hiragana"])) {
            $validations[] = "HiraganaPlus";
        }

        if(in_array($field_info["type"], ["tel"])) {
            $validations[] = "tel";
        }

        if(in_array($field_info["type"], ["zip"])) {
            $validations[] = "ZipCode";
        }

        if(in_array($field_info["type"], ["EmailMultiple"])) {
            $validations[] = "EmailMultiple";
        }

        if(in_array($field_info["type"], ["integer", "unsignedInteger"])) {
            $validations[] = "integer";
        }

        if(!empty($table_data["front"]["options"]["hideForm"])) {
            if(in_array($field, $table_data["front"]["options"]["hideForm"])) {
                $validations[] = "sometimes";
            }
        }

        $other_validate = "";
        if(!empty($field_info["validate"])) {
            foreach($field_info["validate"] as $val) {
                if($val == "unique") {
                    $validations[] = "unique:" . Str::plural(Str::snake($table_data["table"])) . "," . $field . ',{$id}';
                }
                else if($val == "create_required") {
                    $other_validate = "(empty(\$id) ? 'required|' : 'sometimes|nullable|') . ";
                }
                else {
                    $validations[] = $val;
                }
            }
        }

        $ret = "'{$field}' => " . $other_validate . "\"" . implode("|", $validations) . "\"";
        
        return $ret;
    }

    /**
     * ルール作成
     */
    private function createRuleFile($table_data, $all_tables) {
        // Rulesフォルダ作成
        if(!file_exists(app_path("Rules"))) {
            $this->laravel['files']->makeDirectory(app_path("Rules"), 0755, true);
        }

        if(!empty($table_data["nested"]) && !file_exists(app_path("Rules/ArrayUnique.php"))) {
            $rule_text = <<<EOT
<?php

namespace App\Rules;

use Illuminate\Contracts\Validation\Rule;
use Illuminate\Support\Facades\DB;

/**
 * 複数レコード保存時に、同一カラムで同じ値が存在しないことをチェック
 * Class ArrayUnique
 * @package App\Rules
 */
class ArrayUnique implements Rule
{
    /**
     * チェックする文字列
     * @var null
     */
    private \$checkElement = null;

    /**
     * 入力した全てのもの
     * @var array
     */
    private \$inputArray = [];

    /**
     * messageに利用
     * @var string
     */
    private \$attributeName = ':attribute';

    /**
     * ArrayUnique constructor.
     * @param null \$checkElement
     * @param array \$inputArray
     * @param string \$attributeName
     */
    public function __construct(\$checkElement = null, \$inputArray = [], \$attributeName = ':attribute')
    {
        \$this->checkElement = \$checkElement;
        \$this->inputArray = \$inputArray;
        \$this->attributeName = \$attributeName;
    }

    /**
     * Determine if the validation rule passes.
     *
     * @param  string  \$attribute
     * @param  mixed  \$value
     * @return bool
     */
    public function passes(\$attribute, \$value)
    {
        if (empty(\$this->checkElement) || empty(\$this->inputArray)) {
            return true;
        }
        \$counts = array_count_values(\$this->inputArray);
        return \$counts[\$this->checkElement] <= 1;
    }

    /**
     * Get the validation error message.
     *
     * @return string
     */
    public function message()
    {
        return "同じ{\$this->attributeName}が別の箇所で入力されています";
    }
}
EOT;
            file_put_contents(app_path("Rules/ArrayUnique.php"), $rule_text);
        }

        if(!file_exists(app_path("Rules/SlugUnique.php"))) {
            $target_tables = [];
            foreach($all_tables as $table) {
                if(!empty($table["slug"])) {
                    $target_tables[] = "            ['table' => '" . Str::plural(Str::snake($table["table"])) . "', 'attribute' => 'slug'],";
                }
            }
            $target_tables_text = implode("\n", $target_tables);
            $rule_text = <<<EOT
<?php

namespace App\Rules;

use Illuminate\Contracts\Validation\Rule;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Route;

/**
 * slugが全体を通してユニークであることのチェック
 * Class SlugUnique
 * @package App\Rules
 */
class SlugUnique implements Rule
{
    /**
     * table
     * @var null
     */
    private \$table = null;

    /**
     * update時のid
     * @var null
     */
    private \$id = null;

    /**
     * 既にroutingで使われている文字列
     * slugとして使えないためfalseにする
     * @var array
     */
    private \$reserved = [];

    /**
     * Create a new rule instance.
     * @param null \$table
     * @param null \$id
     */
    public function __construct(\$table = null, \$id = null)
    {
        \$this->table = \$table;
        \$this->id = \$id;

        // 既にrouterで使われている文字列を取得
        \$routeCollection = Route::getRoutes();
        foreach (\$routeCollection as \$value) {
            // admin以外
            \$prefix = \$value->getPrefix();
            if (!empty(\$prefix)) \$prefix = trim(\$prefix, '/');

            if (\$prefix !== 'admin') {
                \$paths = explode('/', \$value->uri);
                \$paths = array_filter(\$paths, 'strlen'); // 空要素除去
                if (!empty(\$paths[0]) && !preg_match('/^\{.+\}\$/', \$paths[0])) {
                    \$this->reserved[] = \$paths[0];
                }
            }
        }
        // 重複削除
        \$this->reserved = array_values(array_unique(\$this->reserved));
    }

    /**
     * Determine if the validation rule passes.
     *
     * @param  string  \$attribute
     * @param  mixed  \$value
     * @return bool
     */
    public function passes(\$attribute, \$value)
    {
        if (empty(\$this->table)) {
            return true;
        }

        if (empty(\$value)) {
            // 存在しない場合ここではチェックしない
            return true;
        }

        foreach (\$this->targetTables() as \$targetTable) {
            \$table = \$targetTable['table'];
            \$column = \$targetTable['attribute'];

            // 予約語の場合false
            if (in_array(\$value, \$this->reserved)) {
                return false;
            }

            \$query = DB::table(\$table)
                ->select('id')
                ->where(\$column, \$value)
            ;
            // 当該テーブルでidあるとき除外
            if (\$this->table === \$table && !empty(\$this->id)) {
                \$query->where('id', '<>', \$this->id);
            }

            if (\$query->count() !== 0) {
                return false;
            }
        }

        return true;
    }

    /**
     * Get the validation error message.
     *
     * @return string
     */
    public function message()
    {
        return '入力したスラッグは既に登録されています。';
    }

    /**
     * ユニークチェック対象となるテーブル,column
     * @return array
     */
    private function targetTables()
    {
        return [
{$target_tables_text}
        ];
    }

}
EOT;
            file_put_contents(app_path("Rules/SlugUnique.php"), $rule_text);
        }
    }

    /**
     * コントローラー作成
     */
    private function createControllerFile($table_data) {
        $options = ['name' => $table_data["table"]];
        $options['--single'] = (!empty($table_data["isSingle"])) ? true : false;
        $this->call('larapackmake:controller', $options);

        $controller_path = app_path("Http/Controllers/Admin/" . Str::plural($table_data["table"]) . "Controller.php");
        $controller_text = file_get_contents($controller_path);

        // タイトル
        if(!empty($table_data["title"])) {
            $controller_text = str_replace("return 'タイトル'", "return '" . $table_data["title"] . "'", $controller_text);
        }

        // private_public
        if(!empty($table_data["isPrivatePublic"])) {
            $kakko_last_base = "}";
            $kakko_last_pos = strrpos($controller_text, $kakko_last_base);
            $controller_text = substr_replace($controller_text, "\t" . 'public function update($id)
    {
        if($id == "changeStatus") {
            $model = app(\'request\')[\'model\'];
            $id = app(\'request\')[\'id\'];
            $field = app(\'request\')[\'field\'];
            $flag = app(\'request\')[\'flag\'];

            \Artisan::call(\'custom:change_status\', ["model" => $model, "id" => $id, "field" => $field, "flag" => $flag]);
            $call_result = \Artisan::output();

            return empty($call_result) ? \'\' : \'error\';
        }

        // 入力チェック
        $request = app()->make($this->formRequest());
        $this->beforeProcess();

        // save
        $msgId = $this->insertOrUpdate($id) ? \'larapack::admin.update_ok\' : \'larapack::admin.update_ng\';
        
        // 状態を編集中にする
        $record = $this->model()::findOrFail($id);
        $record->private_public_status = 0;
        $record->save();

        $msg = trans($msgId, [\'subject\' => $this->getSubject()]);
        \Larapack::setFlash($msg);
        return $this->adminRedirect();
    }' . "\n\n", $kakko_last_pos, 0);
        }

        // nested
        if(!empty($table_data["nested"])) {
            $controller_text = str_replace("use Uehi\Larapack\Controllers\Admin\AdminCrud;", "use Uehi\Larapack\Controllers\Admin\AdminCrud;
use App\Models\\" . $table_data["table"] . ";
use App\Rules\ArrayUnique;
use App\Rules\SlugUnique;
use Validator;", $controller_text);

            $kakko_last_base = "}";
            $kakko_last_pos = strrpos($controller_text, $kakko_last_base);
            $controller_text = substr_replace($controller_text, "\t" . '/**
     * 登録編集画面
     */
    public function index()
    {
        $this->beforeProcess();

        // 全データ取得
        $records = ' . $table_data["table"] . '::createSearchQuery([], true)->defaultOrder()->get()->toTree()->toArray();

        return $this->adminView(compact(\'records\'));
    }

    /**
     * 保存処理
     */
    public function store()
    {
        $json = app(\'request\')->input(\'json\');
        if (empty($json)) {
            abort(404);
        }
        $json = json_decode($json, true);

        // validation rule
        $rules = $this->makeValidationRules($json);

        // validation
        $validator = Validator::make($json, $rules, [
            \'*.title.required\' => \'タイトルは必須です\',
' . (!empty($table_data["slug"]) ? '            \'*.slug.required\' => \'スラッグは必須です\',
            \'*.slug.unique\' => \'そのスラッグは他のカテゴリか記事スラッグですでに使われています\',
' : '') . '            \'*.children.*.title.required\' => \'タイトルは必須です\',
' . (!empty($table_data["slug"]) ? '            \'*.children.*.slug.required\' => \'スラッグは必須です\',
            \'*.children.*.slug.unique\' => \'そのスラッグは他のカテゴリか記事スラッグですでに使われています\',
' : '') . '        ]);

        $res = [\'status\' => \'success\', \'errors\' => []];
        if ($validator->fails()) {
            $ret = [];
            foreach ($validator->errors()->messages() as $name => $errorArr) {
                $names = explode(\'.\', $name);
                $namesCount = count($names);

                if ($namesCount === 2) {
                    // 親要素エラー
                    $parentNum = $names[0];

                    // 配列ない場合つくる
                    if (empty($ret[$parentNum])) {
                        $ret[$parentNum] = [\'children\' => [], \'errors\' => []];
                    }

                    // エラー突っ込む
                    foreach ($errorArr as $msg) {
                        $ret[$parentNum][\'errors\'][] = $msg;
                    }
                } elseif ($namesCount === 4) {
                    // 子要素エラー
                    $parentNum = $names[0];
                    $childNum = $names[2];

                    // 配列ない場合つくる
                    if (empty($ret[$parentNum])) {
                        $ret[$parentNum] = [\'children\' => [], \'errors\' => []];
                    }
                    if (empty($ret[$parentNum][\'children\'][$childNum])) {
                        $ret[$parentNum][\'children\'][$childNum] = [\'errors\' => []];
                    }

                    // エラー突っ込む
                    foreach ($errorArr as $msg) {
                        $ret[$parentNum][\'children\'][$childNum][\'errors\'][] = $msg;
                    }
                }
            }
            $res[\'status\'] = \'error\';
            $res[\'errors\'] = $ret;
        } else {
            // 保存
            ' . $table_data["table"] . '::saveAll($json);
            Larapack::setFlash(\'更新しました。\');
        }

        echo json_encode($res);
    }

    /**
     * validationルールを作る
     * @param $json
     * @return array
     */
    private function makeValidationRules($json)
    {
' . (!empty($table_data["slug"]) ? '        // uniqueチェックのため入力されたslugを全て配列に格納
        $slugs = Arr::pluck($json, \'slug\');
        $slugs[] = Arr::pluck($json, \'children.*.slug\');
        $slugs = array_flatten($slugs);
        if (!empty($slugs)) {
            // 空要素削除
            $slugs = array_values(array_filter($slugs, "strlen"));
        }

' : '') . '        // validationルールを作る
        // unique除外IDを*利用時にどう指定するかなかったので自力でrule作りという無念
        $rules = [];
        foreach ($json as $parentKey => $parent) {
            $parentId = !empty($parent[\'id\']) ? $parent[\'id\'] : null;
' . (!empty($table_data["slug"]) ? '            $parentSlug = !empty($parent[\'slug\']) ? $parent[\'slug\'] : null;
' : '') . '
            $rules += [
                "{$parentKey}.title" => \'required\',
' . (!empty($table_data["slug"]) ? '                "{$parentKey}.slug" => $this->slugRule($parentSlug, $slugs, $parentId),
' : '') . '            ];

            if (!empty($parent[\'children\'])) {
                foreach ($parent[\'children\'] as $childKey => $child) {
                    $childId = !empty($child[\'id\']) ? $child[\'id\'] : null;
' . (!empty($table_data["slug"]) ? '                    $childSlug = !empty($child[\'slug\']) ? $child[\'slug\'] : null;
' : '') . '
                    $rules += [
                        "{$parentKey}.children.{$childKey}.title" => \'required\',
' . (!empty($table_data["slug"]) ? '                        "{$parentKey}.children.{$childKey}.slug" => $this->slugRule($childSlug, $slugs, $childId)
' : '') . '                    ];
                }
            }
        }
        // ない場合念のためデフォルトルール
        if (empty($rules)) {
            $rules = [
                \'*.title\' => \'required\',
' . (!empty($table_data["slug"]) ? '                \'*.slug\' => \'required\',
' : '') . '                \'*.children.*.title\' => \'required\',
' . (!empty($table_data["slug"]) ? '                \'*.children.*.slug\' => \'required\',
' : '') . '            ];
        }
        return $rules;
    }
' . (!empty($table_data["slug"]) ? '
    /**
     * slugのvalidationルールを返す
     * @param $input : 入力値
     * @param $allInputs : 複数レコード分全ての入力値
     * @param null $uniqueIgnoreId : ユニークチェック除外ID
     * @return array
     */
    private function slugRule($input, $allInputs, $uniqueIgnoreId = null)
    {
        $rule =[
            \'required\',
            \'regex:\' . config(\'values.slug_regex\'),
            new SlugUnique(\'' . Str::plural(Str::snake($table_data["table"])) . '\', $uniqueIgnoreId),
        ];
        if ($input) {
            $rule[] = new ArrayUnique($input, $allInputs, \'スラッグ\');
        }
        return $rule;
    }' : '') . "\n\n", $kakko_last_pos, 0);
        }

        file_put_contents($controller_path, $controller_text);
    }

    /**
     * 各種ビュー作成
     */
    private function createViewFile($table_data) {
        $table_name = Str::plural(Str::snake($table_data["table"]));

        // nested以外
        if(empty($table_data["nested"])) {
            $options = ['name' => $table_data["table"]];
            $options['--full'] = (!empty($table_data["isFull"])) ? true : false;
            $options['--single'] = (!empty($table_data["isSingle"])) ? true : false;
            $this->call('larapackmake:view', $options);

            // _form
            $view_form_path = resource_path("views/admin/" . $table_name . "/_form.blade.php");
            $view_form_text = "";

            // ajaxzip3がある場合
            if(!empty($table_data["ajaxzip3"])) {
                $view_form_text = "{{ Larapack::appendJs('https://ajaxzip3.github.io/ajaxzip3.js') }}" . "\n\n";
            }

            foreach($table_data["fileds"] as $field => $field_info) {
                $view_form_text .= $this->createAdminFormText($field, $field_info, $table_data);
            }

            if(!empty($table_data["slug"]) && ($table_data["slug"] !== "no-form")) {
                $view_form_text .= $this->createAdminFormText("slug", ["type" => "string", "default" => "", "comment" => "スラッグ"], $table_data);
            }

            file_put_contents($view_form_path, $view_form_text);

            // _list
            $view_list_path = resource_path("views/admin/" . $table_name . "/_list.blade.php");
            if(file_exists($view_list_path)) {
                $view_list_text = file_get_contents($view_list_path);

                $view_list_text = str_replace(["        {{-- 公開・非公開必要な場合記述 --}}\n", "    {{-- 公開・非公開必要な場合記述 --}}\n"], ["", ""], $view_list_text);
                $view_list_text = str_replace(["{{--<", ">--}}", "{{--{", "}--}}"], ["<", ">", "{", "}"], $view_list_text);
                $view_list_text = str_replace("<th>{{ Larapack::util()->sortLink('title', '表示名') }}</th>", ADMIN_LIST_TH_REPLACE, $view_list_text);
                $view_list_text = str_replace('            <a href="{{ admin_url("/{$controllerNameSnake}/{$record->id}/edit") }}">{{ $record->title }}</a>', ADMIN_LIST_TD_REPLACE, $view_list_text);
                $view_list_text = str_replace("<td>\n" . ADMIN_LIST_TD_REPLACE . "\n        </td>", ADMIN_LIST_TD_REPLACE, $view_list_text);
                
                $has_nested_select = false;
                foreach($table_data["fileds"] as $field => $field_info) {
                    $view_list_text = $this->createAdminListText($view_list_text, $field, $field_info);
                    if($field_info["type"] == "nestedSelect") {
                        $has_nested_select = true;
                    }
                }

                if($has_nested_select) {
                    // css追加
                    $view_list_text = '@section(\'cssBlock\')
    @parent
    <style>
        .categoryList {
            margin: 0;
            padding: 0;
        }
        .categoryList li {
            display: inline;
            list-style-type: none;
        }
        .categoryList li:before {
            content: " > ";
        }
        .categoryList li:first-child:before {
            content: none;
        }
    </style>
@endsection

' . $view_list_text;
                }

                // slug
                if(!empty($table_data["slug"]) && ($table_data["slug"] !== "no-form")) {
                    $th_text = "<th>{{ Larapack::util()->sortLink('slug', 'スラッグ') }}</th>";
                    $view_list_text = str_replace(ADMIN_LIST_TH_REPLACE, $th_text . ADMIN_LIST_TH_INDENT . ADMIN_LIST_TH_REPLACE, $view_list_text);
                    $td_text = "<td>{{ \$record->slug }}</td>";
                    $view_list_text = str_replace(ADMIN_LIST_TD_REPLACE, $td_text . ADMIN_LIST_TD_INDENT . ADMIN_LIST_TD_REPLACE, $view_list_text);
                }

                // 並び順
                if(!empty($table_data["hasSequence"])) {
                    $th_text = "<th>{{ Larapack::util()->sortLink('sequence', '並び順') }}</th>";
                    $view_list_text = str_replace(ADMIN_LIST_TH_REPLACE, $th_text . ADMIN_LIST_TH_INDENT . ADMIN_LIST_TH_REPLACE, $view_list_text);
                    $td_text = "<td align=\"center\">{{ Larapack::util()->inputSequence(\$record) }}</td>";
                    $view_list_text = str_replace(ADMIN_LIST_TD_REPLACE, $td_text . ADMIN_LIST_TD_INDENT . ADMIN_LIST_TD_REPLACE, $view_list_text);
                }

                // private_public
                if(!empty($table_data["isPrivatePublic"])) {
                    $view_list_text = str_replace("<th class=\"w100\">{{ Larapack::util()->sortLink('public', '公開/非公開') }}</th>", "<th class=\"w100\">{{ Larapack::util()->sortLink('public', '公開/非公開') }}</th>
    <th class=\"w100\">{{ Larapack::util()->sortLink('private_public_status', '状態') }}</th>
    <th class=\"w100\">状態変更</th>", $view_list_text);
                    $view_list_text = str_replace("<td align=\"center\">
            {{ Larapack::util()->selectPublic(\$record) }}
        </td>", "<td align=\"center\" id=\"public_text\">{{ !empty(\$record->public) ? config('values.public.' . \$record->public) : '' }}</td>
        <td align=\"center\" id=\"status_text\">{{ is_numeric(\$record->private_public_status) ? config('values.private_public_statuses.' . \$record->private_public_status) : '' }}</td>
        <td align=\"center\">
            {{ Larapack::util()->selectStatus(\$record, ['url' => admin_url('/" . $table_name . "/changeStatus'), 'current_public' => \$record->public]) }}
        </td>", $view_list_text);
                }

                $view_list_text = str_replace(ADMIN_LIST_TH_INDENT . ADMIN_LIST_TH_REPLACE, "", $view_list_text);
                $view_list_text = str_replace(ADMIN_LIST_TD_INDENT . ADMIN_LIST_TD_REPLACE, "", $view_list_text);
                file_put_contents($view_list_path, $view_list_text);
            }

            // _search
            $view_search_path = resource_path("views/admin/" . $table_name . "/_search.blade.php");
            if(file_exists($view_search_path)) {
                $view_search_text = file_get_contents($view_search_path);
                $view_search_text = str_replace("    {{-- 公開・非公開必要な場合記述 --}}\n", "", $view_search_text);
                $view_search_text = str_replace(["{{--", "--}}"], ["", ""], $view_search_text);
                $view_search_text = str_replace("<label>
        キーワード&nbsp;&nbsp;
        {{ Form::text('keywords', null, ['class' => 'form-control']) }}
    </label>", "<label>
        キーワード&nbsp;&nbsp;
        {{ Form::text('keywords', null, ['class' => 'form-control']) }}
    </label>" . ADMIN_SEARCH_REPLACE, $view_search_text);

                $has_nested_select = false;
                foreach($table_data["fileds"] as $field => $field_info) {
                    $view_search_text = str_replace(ADMIN_SEARCH_REPLACE, $this->createAdminSearchText($field, $field_info), $view_search_text);
                    if($field_info["type"] == "nestedSelect") {
                        $has_nested_select = true;
                    }
                }

                if($has_nested_select) {
                    // js追加
                    $view_search_text = '@section(\'jsBlock\')
    @parent
    <script>
    jQuery(function($){
        // 検索フォームのクリアボタン機能追加
        $(\'.adminIndexClearForm\').bind(\'click\', function() {
            $(this).closest(\'form\').find(\':hidden\').val(\'\');
        });
    });
    </script>
@endsection

' . $view_search_text;
                }

                $view_search_text = str_replace(ADMIN_SEARCH_REPLACE, "", $view_search_text);
                file_put_contents($view_search_path, $view_search_text);
            }
        }
        else {
            // viewsフォルダ作成
            if(!file_exists(resource_path("views/admin/" . $table_name))) {
                $this->laravel['files']->makeDirectory(resource_path("views/admin/" . $table_name), 0755, true);
            }

            // idnexのみ
            $view_index_path = resource_path("views/admin/" . $table_name . "/index.blade.php");
            $view_index_text = <<<EOT
@extends('larapack::layouts.admin')

@section('cssBlock')
    @parent
    <link rel="stylesheet" href="//ajax.googleapis.com/ajax/libs/jqueryui/1.11.4/themes/smoothness/jquery-ui.css">
@endsection
@section('cssBlock')
    @parent
    <style>
        .master-box {
            margin: 10px 0;
            padding: 10px 0;
        }
        .master-box label {
            margin-bottom: 0 !important;
        }
        .master-box .btnList {
            float: right;
        }
        .master-box ul.master-list {
            margin: 10px 0 0 0;
            padding: 0;
            list-style: none;
        }
        .master-box ul.master-list li {
            display: block;
            margin-bottom: 5px;
            border: 1px solid #999;
            background: #fff;
            border-radius: 3px;
            -webkit-box-shadow: 0 1px 1px rgba(0,0,0,.05);
            box-shadow: 0 1px 1px rgba(0,0,0,.05);
        }
        .master-box ul.master-list li {
            background: #fff;
            display: block;
            text-decoration: none;
            padding: 5px;
        }
        .master-box ul.master-list li:hover {
            cursor: move;
        }
        .master-box ul.master-list ul {
            margin-left: 20px;
            margin-top: 5px;
        }
        .master-box li.master-highlight {
            border: 2px dashed #d9534f !important;
            background: #f5f5f5;
        }
        .toggleArea {
            display: inline-block;
            cursor: pointer;
            font-size: 1.5em;
            color: #d9534f;
            padding: 0 5px;
        }
        .error-message .ico {
            display: inline-block;
            float: left;
            padding: 0 10px 0 0;
            cursor: pointer;
        }
        .error-message .msg {
            margin-left: 30px;
        }
    </style>
@endsection

@section('js')
    @parent
    <script src="//ajax.googleapis.com/ajax/libs/jqueryui/1.11.4/jquery-ui.min.js"></script>
    <script src="{{ url('/larapack/js/jquery.mjs.nestedSortable_custom.js') }}"></script>
@endsection

@section('jsBlock')
    @parent
    <script>
        // Vueで作ろうとしたがいいライブラリが存在せずUIクオリティが
        jQuery(function($) {
            // 初期データ
            var masters = {!! json_encode(\$records) !!};
            var templateHtml = $('#listTemplate').html();
            var sortableObj = $('.sortable');
            var num = 1;

            // 初期データをhtmlにセット
            masters.forEach(function(master) {
                createBlock(master, null);
            });

            // 子項目toggleボタン -----------------------------------------------------------------
            $(document).on('click', '.toggleArea', function(){
                toggleChild($(this).closest('.itemList'));
            });
            // 親追加ボタン -----------------------------------------------------------------
            $(document).on('click', '.addParentBtn', function(){
                createBlock(null);
            });
            // 子追加ボタン -----------------------------------------------------------------
            $(document).on('click', '.addChildBtn', function(){
                var parentObj = $(this).closest('.itemList');
                createBlock(null, parentObj);
                toggleChild(parentObj, true);
            });
            // 削除ボタン -----------------------------------------------------------------
            $(document).on('click', '.delBtn', function(){
                $(this).closest('li').remove()
            });
            // エラーメッセージ削除ボタン -----------------------------------------------------------------
            $(document).on('click', '.error-message > .ico', function(){
                $(this).closest('.error-message').remove()
            });
            // public 背景色変更 -----------------------------------------------------------------
            $(document).on('change', '.publicSelect', function(){
                toggleBg($(this).closest('.itemList'));
            });

            // ブロック追加処理 -----------------------------------------------------------------
            function createBlock(master, parentObj) {
                // masterない場合新規(idは"new連番"にする)
                if (!master) {
                    master = {'id': 'new' + num};
                    num = num + 1;
                }

                // keyをidに置換
                var html = templateHtml.replace(/NNN/g, master.id);

                // まずappendしないとinput要素が切り替わらない
                if (parentObj) {
                    // 親ある場合
                    var childUl = parentObj.find('ul').first();
                    // 子が空の状態の親要素を移動したときに子ulがなくなる現象があるため
                    if (!childUl.html()) {
                        parentObj.append('<ul>');
                        childUl = parentObj.find('ul').first();
                    }
                    childUl.append(html);
                } else {
                    sortableObj.append(html);
                }

                var htmlObj = $('#item_' + master.id);

                // input要素を埋める
                if (master.title) htmlObj.find(":input[name='title_" + master.id + "']").val(master.title);
                if (master.slug) htmlObj.find(":input[name='slug_" + master.id + "']").val(master.slug);
                if (master.public) htmlObj.find(":input[name='public_" + master.id + "']").val(master.public);

                // 親か子かで出し分け
                toggleElements(htmlObj);
                // publicによる背景色
                toggleBg(htmlObj);

                // errorsあれば埋める
                if (master.errors && master.errors.length > 0) {
                    var errorHtml = master.errors.join('<br>');
                    htmlObj.find('.errorArea').first().html('<div class="error-message"><div class="ico"><i class="fa fa-times"></i></div><div class="msg">' + errorHtml + '</div></div>');
                }

                // 子要素あるならそれも
                if (master.children && master.children.length) {
                    master.children.forEach(function(child) {
                        createBlock(child, htmlObj);
                    });
                }
            }

            // 子項目開閉処理 -----------------------------------------------------------------
            function toggleChild(parentObj, forceOpen) {
                var childObj = parentObj.find('ul').first();
                if (forceOpen || !childObj.is(":visible")) {
                    childObj.show();
                    parentObj.find('.toggleArea').first().html('<i class="fa fa-chevron-down"></i>');
                } else {
                    childObj.hide();
                    parentObj.find('.toggleArea').first().html('<i class="fa fa-chevron-up"></i>');
                }
            }
            // 親か子かでtoggleする要素 -----------------------------------------------------------------
            function toggleElements(listObj) {
                listObj.find('.parentOnly').each(function(){
                    if ($(this).closest('ul').hasClass('master-list')) {
                        $(this).show();
                    } else {
                        $(this).hide();
                    }
                });
            }
            // 公開非公開で背景色変更する要素 -----------------------------------------------------------------
            function toggleBg(listObj) {
                var selectObj = listObj.find('.publicSelect').first();
                listObj.css('background-color', selectObj.val() === '1' ? '#FFFFFF' : '#E3E3E3');
            }

            // sortable -----------------------------------------------------------------
            sortableObj.nestedSortable({
                forcePlaceholderSize: true,
                listType: 'ul',
                items: 'li',
                placeholder: 'master-highlight',
                maxLevels: 2,
                opacity: .6,
                relocate: function(event, obj) {
                    // 移動完了したとき
                    // 親要素かどうかで出し分け
                    toggleElements(obj.item);
                }
            });

            // 送信ボタン -----------------------------------------------------------------
            $('.sendBtn').click(function(){
                // 入力要素を渡すために"data-*"要素を追加
                $('.itemList').each(function(){
                    var id = $(this).attr('id');
                    if (id === 'item_NNN') {
                        // 処理しない
                        return;
                    }
                    var idNum = id.split('\_')[1];

                    var titleVal = $(this).find(":input[name='title_" + idNum + "']").val();
                    var slugVal = $(this).find(":input[name='slug_" + idNum + "']").val();
                    var publicVal = $(this).find(":input[name='public_" + idNum + "']").val();
                    $(this).data('title', titleVal);
                    $(this).data('slug', slugVal);
                    $(this).data('public', publicVal);
                });

                var hie = sortableObj.nestedSortable('toHierarchy');
                // var jsonStr = JSON.stringify(hie);
                // console.log(jsonStr);
                // $('#form').submit();
                sendAjax(hie);
            });

            // 送信Ajax処理 -----------------------------------------------------------------
            function sendAjax(jsonObj) {
                if (!jsonObj || (jsonObj && jsonObj.length <= 0)) {
                    alert('カテゴリが何もありません。');
                    return;
                }
                var jsonStr = JSON.stringify(jsonObj);
                console.log(jsonObj);

                // CSRF
                $.ajaxSetup({
                    headers: {
                        'X-CSRF-TOKEN': '{{ csrf_token() }}'
                    }
                });

                // 送信
                var url = '{{ admin_url('/{$table_name}') }}';
                $.ajax({
                    url : url,
                    type : 'post',
                    cache : false,
                    data: {
                        'json': jsonStr
                    },
                    dataType: 'json',
                    success : function(json){
                        if (json['status'] === 'error') {
                            // エラー
                            var errors = json['errors'];

                            jsonObj.forEach(function (parent, i) {
                                if (errors[i] && errors[i].errors) {
                                    parent.errors = errors[i].errors;
                                }

                                if (parent.children) {
                                    parent.children.forEach(function (child, j) {
                                        if (errors[i] && errors[i].children && errors[i].children[j] && errors[i].children[j].errors) {
                                            child.errors = errors[i].children[j].errors;
                                        }
                                    });
                                }
                            });

                            // エラーを含めてリストを再生成
                            sortableObj.empty();
                            jsonObj.forEach(function(master) {
                                createBlock(master, null);
                            });
                        } else {
                            // 成功
                            location.href = url;
                        }
                    },
                    error : function(){
                        location.href = url;
                    }
                });
            }

        });
    </script>
@endsection

@section('content')

    <h1>{{ \$subject }}設定</h1>
    <h2>
        登録されている{{ \$subject }}の一覧です。追加・削除・ドラッグ&ドロップで階層変更や並び替えができます。
    </h2>

    {{ Form::open(['url' => null, 'method' => 'POST', 'name' => 'form', 'id' => 'form']) }}
        <input type="hidden" name="json" id="json">
        <a href="javascript:void(0);" class="btn btn-primary sendBtn">以下の内容で登録</a>
    {{ Form::close() }}

    <div class="master-box">
        <div align="right">
            <a href="javascript:void(0);" class="btn btn-primary addParentBtn">親カテゴリ追加</a>
        </div>
        <ul class="master-list sortable">
        </ul>
    </div>

    {{-- blockテンプレ --}}
    <div id="listTemplate" style="display:none;">
        <li id="item_NNN" class="itemList">
            <div class="form-inline">
                <span class="toggleArea parentOnly">
                    <i class="fa fa-chevron-down"></i>
                </span>

                <label>
                    タイトル&nbsp;&nbsp;
                    <input type="text" name="title_NNN" value="" class="form-control">
                </label>
                <label>
                    スラッグ&nbsp;&nbsp;
                    <input type="text" name="slug_NNN" value="" class="form-control">
                </label>
                <label>
                    {{ Form::select('public_NNN', Config::get('values.public'), 2, ['class' => 'form-control publicSelect']) }}
                </label>
                <div class="btnList">
                    <a href="javascript:void(0);" class="btn btn-primary addChildBtn parentOnly">子カテゴリ追加</a>
                    <a href="javascript:void(0);" class="btn btn-danger delBtn">このカテゴリを削除</a>
                </div>
                <div class="errorArea">
                </div>
            </div>

            <ul>
            </ul>
        </li>
    </div>

@endsection
EOT;

            file_put_contents($view_index_path, $view_index_text);
        }
    }

    /**
     * 管理側入力フォーム
     */
    private function createAdminFormText($field, $field_info, $table_data) {
        $req = (isset($field_info["default"]) || (!isset($field_info["default"]) && empty($field_info["nullable"]))) ? "{{ req() }}" : "";

        // 位置情報
        if($field_info["type"] == "map") {
            $ret = "<tr>
    <th>" . $field_info["comment"] . $req . "</th>
    <td>
        {{ Larapack::form()->map('lat', 'lon', 'zoom', [], true) }}
        {{ Larapack::form()->error('lat') }}
        {{ Larapack::form()->error('lon') }}
        {{ Larapack::form()->error('zoom') }}
    </td>
</tr>
";
            return $ret;
        }

        $method = "text";
        $value = ", null";
        $option1 = ", ['class' => 'form-control']";
        $option2 = "";
        switch($field_info["type"]) {
            case "text":
                $method = "textarea";
                break;

            case "password":
                $method = "password";
                $value = "";
                break;

            case "select":
                $method = "select";
                $value = ", ";
                if(!empty($field_info["empty"])) {
                    $value .= "config('values.selectEmpty') + ";
                }
                if(!empty($field_info["values"])) {
                    $value .= "config('values." . $field_info["values"] . "')";
                }
                else if(!empty($field_info["hasOne"])) {
                    $value .= '$' . Str::plural(Str::snake($field_info["hasOne"]));
                }
                $value .= ", null";
                break;

            case "radio":
                $method = "radio";
                $value = "";
                if(!empty($field_info["values"])) {
                    $value .= "config('values." . $field_info["values"] . "')";
                }
                else if(!empty($field_info["hasOne"])) {
                    $value .= '$' . Str::plural(Str::snake($field_info["hasOne"]));
                }
                break;

            case "selectDate":
                $method = "selectDate";
                $value = "";
                if(!empty($field_info["options"])) {
                    $option1 = ", " . var_export($field_info["options"], true);
                }
                else {
                    $option1 = "";
                }
                break;

            case "textDate":
                $method = "textDate";
                if(!empty($field_info["default"]) && ($field_info["default"] == "today")) {
                    $value = ", !empty(\$record->" . $field . ") ? \$record->" . $field . " : " . (empty($field_info["time"]) ? "date('Y-m-d')" : "date('Y-m-d H:i:s')");
                }
                if(!empty($field_info["time"])) {
                    $option1 = ", ['class' => 'form-control', 'time' => true]";
                }
                break;

            case "checkbox":
                $method = "checkbox";
                $value = ", !empty(\$record->" . $field . ")";
                $option2 = ", ['" . implode("', '", $field_info["text"]) . "']";
                break;

            case "imageFile":
                $method = "imageFile";
                $value = ", []";
                $option1 = ", ['width' => '160px']";
                break;

            case "documentFile":
                $method = "documentFile";
                $value = ", []";
                $option1 = "";
                break;

            case "belongsToMany":
                $method = "multiCheckbox";
                $field = Str::plural(Str::snake($field));
                $value = ", \$" . $field . ", null";
                break;

            case "EmailMultiple":
                $method = "textarea";
                $field = Str::snake($field) . "_mails";
                break;

            case "zip":
                if(!empty($table_data["ajaxzip3"]) && empty($table_data["ajaxzip3"]["button"])) {
                    $option1 = ", ['class' => 'form-control', 'onKeyUp' => 'AjaxZip3.zip2addr(\'";
                    $option1 .= implode("\', \'", $table_data["ajaxzip3"]["field"]);
                    $option1 .= "\');']";
                }
                break;

            case "string":
            case "integer":
            case "unsignedInteger":
            default:
                break;
        }
        $ret = "<tr>
    <th>" . $field_info["comment"] . $req . "</th>
    <td>
        {{ Larapack::form()->" . $method . "('" . $field . "'" . $value . $option1 . $option2 . ") }}
        {{ Larapack::form()->error('" . $field . "') }}
    </td>
</tr>
";
        // ラジオボタンはoldとの比較がうまくいかないためhtml直書き
        if($field_info["type"] == "radio") {
            $ret = '<tr>
    <th>' . $field_info["comment"] . $req . '</th>
    <td>
    @if (!Larapack::form()->isConfirm())
        @foreach (' . $value . ' as $key => $val)
            @php $checked = old(\'' . $field . '\', !empty($record->' . $field . ') ? $record->' . $field . ' : null); @endphp
        <label><input class="form-control radio-inline" name="' . $field . '" type="radio" value="{{ $key }}" @if ($checked == $key) checked="checked" @endif >{{ $val }}</label>
        @endforeach
        {{ Larapack::form()->error(\'' . $field . '\') }}
    @else
        {{ Larapack::form()->getDisplaySelect(\'' . $field . '\', ' . $value . ') }}
    @endif
    </td>
</tr>
';
        }
        else if(($field_info["type"] == "zip") && !empty($table_data["ajaxzip3"]) && !empty($table_data["ajaxzip3"]["button"])) {
            $ret = '<tr>
    <th>' . $field_info["comment"] . $req . '</th>
    <td>
        <div class="form-inline">
            <div class="input-group">
                {{ Larapack::form()->text(\'' . $field . '\', null, [\'class\' => \'form-control\']) }}
                @if (!Larapack::form()->isConfirm()) {{-- 入力時 --}}
                    <span class="input-group-btn">
                        <button class="btn btn-success" type="button" onclick="AjaxZip3.zip2addr(\'' . implode("', '", $table_data["ajaxzip3"]["field"]) . '\');">住所検索</button>
                    </span>
                @endif
            </div>
        </div>
        {{ Larapack::form()->error(\'' . $field . '\') }}
    </td>
</tr>
';
        }
        else if(!empty($field_info["validate"]) && in_array("confirmed", $field_info["validate"])) {
            $ret = '<tr>
    <th>' . $field_info["comment"] . $req . '</th>
    <td>
    @if (!Larapack::form()->isConfirm())
        ' . ((in_array("create_required", $field_info["validate"])) ? '@if (!empty($record->id))
        <div class="alert alert-info" role="alert">
            変更時のみ入力
        </div>
        @endif
        ' : '') . '{{ Larapack::form()->' . $method . '(\'' . $field . '\'' . $value . $option1 . $option2 . ') }}
        {{ Larapack::form()->error(\'' . $field . '\') }}

        <br>確認のため再入力<br>
        {{ Larapack::form()->' . $method . '(\'' . $field . '_confirmation\'' . $value . $option1 . $option2 . ') }}
        {{ Larapack::form()->error(\'' . $field . '_confirmation\') }}
    @else
        {{ Larapack::form()->' . $method . '(\'' . $field . '\'' . $value . $option1 . $option2 . ') }}
    @endif
    </td>
</tr>
';
        }
        else if($field_info["type"] == "nestedSelect") {
            $changeCallback = "''";
            if(!empty($field_info["childUrl"])) {
                $changeCallback = '"// urlを対象へ追加
                var url = \'\';
                url += parentVal && select_values[parentVal] ? select_values[parentVal] + \'/\' : \'\';
                url += childVal && select_values[childVal] ? select_values[childVal] + \'/\' : \'\';

                $(\'#' . $field_info["childUrl"] . 'Area\').text(url);"';
            }
            $ret = '<tr>
    <th>' . $field_info["comment"] . $req . '</th>
    <td>
    @if (!Larapack::form()->isConfirm())
        <div class="form-inline">
            {{ Larapack::form()->nestedSelect($' . Str::plural(Str::snake($field)) . '_tree, \'' . $field_info["selectField"] . '\', \'' . Str::snake($field) . '_id\', ' . $changeCallback . ') }}
        </div>
        {{ Larapack::form()->error(\'' . Str::snake($field) . '_id\') }}
    @else
        // TODO
    @endif
    </td>
</tr>
';
        }
        else if($field_info["type"] == "permalink") {
            $ret = '<tr>
    <th>' . $field_info["comment"] . $req . '</th>
    <td>
    @if (!Larapack::form()->isConfirm())
        @php Larapack::appendCssBlock(\'<style>#' . $field . 'Area { font-weight: bold; font-size: 1.3em; }</style>\'); @endphp
        <div class="form-inline">
            {{ config(\'app.url\') }}/' . (!empty($field_info["parentUrl"]) ? '<span id="' . $field . 'Area"></span>&nbsp;&nbsp;' : '') . '{{ Larapack::form()->text(\'' . $field . '\', null, [\'class\' => \'form-control\', \'style\' => \'width: 350px;\']) }}
        </div>
        {{ Larapack::form()->error(\'' . $field . '\') }}
    @else
        {{ config(\'app.url\') }}/{{ Larapack::form()->getDisplayInput(\'' . $field . '\') }}
    @endif
    </td>
</tr>
';
        }
        else if(($field_info["type"] == "text") && !empty($field_info["ckeditor"])) {
            $ret = '<tr>
    <th>' . $field_info["comment"] . $req . '</th>
    <td>
        {{ Larapack::form()->textarea(\'' . $field . '\', null, [\'id\' => \'ckeditor' . studly_case($field) . '\'], [\'src\' => asset(\'' . $this->ckeditor_path . 'ckeditor/ckeditor.js\')]) }}
        {{ Larapack::form()->error(\'' . $field . '\') }}

        <script>
            // ckeditorのタグより下に書く
            var editor = CKEDITOR.replace(\'ckeditor' . studly_case($field) . '\', {
                //customConfig: \'{{ asset(\'/js/admin/ckeditor_biz_articles/config.js\') }}\',
                width: \'100%\',
                height: \'750\'
            });
        </script>
    </td>
</tr>
';
        }

        return $ret;
    }

    /**
     * 管理側一覧画面
     */
    private function createAdminListText($view_list_text, $field, $field_info) {
        if(!empty($field_info["dispList"])) {
            $th_text = "<th>{{ Larapack::util()->sortLink('{$field}', '" . $field_info["comment"] . "') }}</th>";
            if($field_info["type"] == "nestedSelect") {
                $th_text = "<th>" . $field_info["comment"] . "</th>";
            }
            $view_list_text = str_replace(ADMIN_LIST_TH_REPLACE, $th_text . ADMIN_LIST_TH_INDENT . ADMIN_LIST_TH_REPLACE, $view_list_text);

            $td_text = "<td>";
            if($field_info["dispList"] == "link") {
                $td_text .= '<a href="{{ admin_url("/{$controllerNameSnake}/{$record->id}/edit") }}">';
            }

            if(in_array($field_info["type"], ["select", "radio", "checkbox"])) {
                $td_text .= '{{ !empty($record->' . $field . ') ? ';
                if(!empty($field_info["values"])) {
                    $td_text .= 'config(\'values.' . $field_info["values"] . '.\' . $record->' . $field . ') : \'\' }}';
                }
                else if(!empty($field_info["hasOne"])) {
                    $td_text .= '$' . Str::plural(Str::snake($field_info["hasOne"])) . '[$record->' . $field . '] : \'\' }}';
                }
                else if($field_info["type"] == "checkbox") {
                    $td_text .= "'" . $field_info["text"][1] . "' : '" . $field_info["text"][2] . "' }}";
                }
            }
            else if($field_info["type"] == "nestedSelect") {
                $field_snake = Str::snake($field);
                $td_text .= "\n\t\t\t" . '<ul class="categoryList">
                @foreach ($' . Str::plural($field_snake) . '_titles[$record->' . $field_snake . '_id] as $categoryId => $categoryTitle)
                    <li>
                        <a href="{{ admin_url("/{$controllerNameSnake}/?' . $field_snake . '_id={$categoryId}") }}">{{ $categoryTitle }}</a>
                    </li>
                @endforeach
            </ul>' . "\n\t\t";
            }
            else {
                $td_text .= '{{ $record->' . $field . ' }}';
            }

            if($field_info["dispList"] == "link") {
                $td_text .= "</a>";
            }
            $td_text .= "</td>";
            $view_list_text = str_replace(ADMIN_LIST_TD_REPLACE, $td_text . ADMIN_LIST_TD_INDENT . ADMIN_LIST_TD_REPLACE, $view_list_text);
        }
        return $view_list_text;
    }

    /**
     * 管理側検索画面
     */
    private function createAdminSearchText($field, $field_info) {
        $ret = ADMIN_SEARCH_REPLACE;
        if(!empty($field_info["dispSearch"])) {
            if(in_array($field_info["type"], ["select", "radio", "checkbox"])) {
                $values = "";
                if(!empty($field_info["values"])) {
                    $values = "config('values." . $field_info["values"] . "')";
                }
                else if(!empty($field_info["hasOne"])) {
                    $values = '$' . Str::plural(Str::snake($field_info["hasOne"]));
                }
                else if($field_info["type"] == "checkbox") {
                    $values = "[0 => '" . $field_info["text"][1] . "', 1 => '" . $field_info["text"][2] . "']";
                }
                $ret = ADMIN_SEARCH_INDENT . "<label>
        " . $field_info["comment"] . "&nbsp;&nbsp;
        {{ Form::select('" . $field . "', config('values.selectEmpty') + " . $values . ", null, ['class' => 'form-control']) }}
    </label>";
                $ret .= ADMIN_SEARCH_REPLACE;
            }
            else if(in_array($field_info["type"], ["selectDate", "textDate"])) {
                $time_text = "";
                if($field_info["type"] == "textDate" && !empty($field_info["time"])) {
                    $time_text = ", 'time' => true";
                }
                $ret = ADMIN_SEARCH_INDENT . "<label>
        " . $field_info["comment"] . "&nbsp;&nbsp;
        @php
            Larapack::form()->setIsConfirm(false);
        @endphp
        {{ Larapack::form()->textDate('start_" . $field . "', null, ['class' => 'form-control'" . $time_text . "]) }}
        〜
        {{ Larapack::form()->textDate('end_" . $field . "', null, ['class' => 'form-control'" . $time_text . "]) }}
    </label>";
                $ret .= ADMIN_SEARCH_REPLACE;
            }
            else if($field_info["type"] == "nestedSelect") {
                $ret = ADMIN_SEARCH_INDENT . '<label>
        <div class="form-inline">
            ' . $field_info["comment"] . '&nbsp;&nbsp;
            {{ Larapack::form()->nestedSelect($' . Str::plural(Str::snake($field)) . '_tree, \'' . $field_info["selectField"] . '\', \'' . Str::snake($field) . '_id\') }}
        </div>
    </label>';
                $ret .= ADMIN_SEARCH_REPLACE;
            }
        }
        return $ret;
    }

    /**
     * ミドルウェア作成
     */
    private function createMiddlewareFile($table_data) {
        $middleware_init = true;
        $middleware_text = "";
        $middleware_name = "Get" . $table_data["table"] . "Masters";
        $middleware_path = app_path("Http/Middleware/" . $middleware_name . ".php");
        foreach($table_data["fileds"] as $field => $field_info) {
            if(!empty($field_info["hasOne"]) || ($field_info["type"] == "belongsToMany") || ($field_info["type"] == "nestedSelect")) {
                if($middleware_init) {
                    if(file_exists($middleware_path)) {
                        $this->error('Middleware Already exists!:' . $middleware_name . ".php");
                        return false;
                    }

                    $middleware_text = '<?php

namespace App\Http\Middleware;

' . MIDDLEWARE_USE_REPLACE . '
use Closure;
use Illuminate\Support\Facades\Route;
use Illuminate\Support\Facades\View;

class ' . $middleware_name . '
{
    /**
     * Handle an incoming request.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \Closure  $next
     * @return mixed
     */
    public function handle($request, Closure $next)
    {
        // マスタ関係を取得しviewにset
        $prefix = Route::getCurrentRoute()->getPrefix();
        $admin = $prefix === \'admin\';

        ' . MIDDLEWARE_REPLACE . '

        return $next($request);
    }
}
';
                    $middleware_init = false;
                }

                $class_name = !empty($field_info["hasOne"]) ? $field_info["hasOne"] : $field;
                $middleware_text = str_replace(MIDDLEWARE_USE_REPLACE, "use App\Models\\" . $class_name . ";\n" . MIDDLEWARE_USE_REPLACE, $middleware_text);
                $middleware_text = str_replace(MIDDLEWARE_REPLACE, '// ' . $field_info["comment"] . '
        View::share(\'' . Str::plural(Str::snake($class_name)) . '\', ' . $class_name . '::getPublic(true, $admin));' . MIDDLEWARE_INDENT . MIDDLEWARE_REPLACE, $middleware_text);
                if($field_info["type"] == "nestedSelect") {
                    $middleware_text = str_replace(MIDDLEWARE_REPLACE, '// ' . $field_info["comment"] . 'tree
        View::share(\'' . Str::plural(Str::snake($class_name)) . '_tree\', ' . $class_name . '::getPublicTree($admin));' . MIDDLEWARE_INDENT . MIDDLEWARE_REPLACE, $middleware_text);
                    $middleware_text = str_replace(MIDDLEWARE_REPLACE, 'if($admin) {
            // ' . $field_info["comment"] . 'title配列
            View::share(\'' . Str::plural(Str::snake($class_name)) . '_titles\', ' . $class_name . '::getPublicColumnArray(\'title\', $admin));
        }' . MIDDLEWARE_INDENT . MIDDLEWARE_REPLACE, $middleware_text);
                }
            }
        }

        if(!$middleware_init) {
            $middleware_text = str_replace("\n" . MIDDLEWARE_USE_REPLACE, "", $middleware_text);
            $middleware_text = str_replace(MIDDLEWARE_INDENT . MIDDLEWARE_REPLACE, "", $middleware_text);
            file_put_contents($middleware_path, $middleware_text);

            $kernel_path = app_path("Http/Kernel.php");
            $kernel_text = file_get_contents($kernel_path);
            $kakko_base = "];";
            $kakko_pos = strrpos($kernel_text, $kakko_base);
            $kernel_text = substr_replace($kernel_text, "\t" . '\'' . Str::snake($middleware_name) . '\' => \App\Http\Middleware\\' . $middleware_name . '::class,' . "\n\t", $kakko_pos, 0);
            file_put_contents($kernel_path, $kernel_text);

            $controller_path = app_path("Http/Controllers/Admin/" . Str::plural($table_data["table"]) . "Controller.php");
            $controller_text = file_get_contents($controller_path);
            if(strpos($controller_text, "public function __construct()") === false) {
                $use_base = "use AdminCrud;";
                $use_pos = strrpos($controller_text, $use_base);
                $controller_text = substr_replace($controller_text, "\n\n\t" . '/**
     * ' . Str::plural($table_data["table"]) . 'Controller constructor.
     */
    public function __construct()
    {
        // master取得
        $this->middleware(\'' . Str::snake($middleware_name) . '\')->only([\'index\', \'show\', \'create\', \'edit\']);
    }' . "\n", $use_pos + strlen($use_base), 0);
                file_put_contents($controller_path, $controller_text);
            }
        }
    }

    /**
     * 表側ファイルの作成
     */
    private function createFrontFile($table_data) {
        // 管理画面のみはすぐリターン
        if(empty($table_data["front"])) {
            return;
        }

        // layoutsフォルダ作成
        if(!file_exists(resource_path("views/layouts"))) {
            $this->laravel['files']->makeDirectory(resource_path("views/layouts"), 0755, true);
        }

        // layoutsファイル作成
        $layout_name = "views/layouts/app.blade.php";
        if(!file_exists(resource_path($layout_name))) {
            $layout_text = <<<EOT
<!doctype html>
<html>
<head>
    <meta charset="utf-8">
    <link rel="stylesheet" href="https://maxcdn.bootstrapcdn.com/bootstrap/3.3.5/css/bootstrap.min.css">
</head>
<body>

    <div class="container">
        @yield('content')
    </div>

<script src="https://ajax.googleapis.com/ajax/libs/jquery/1.11.3/jquery.min.js"></script>
<script src="https://maxcdn.bootstrapcdn.com/bootstrap/3.3.5/js/bootstrap.min.js"></script>
@section('js')
@show
</body>
</html>
EOT;
            file_put_contents(resource_path($layout_name), $layout_text);
        }

        $view = Str::plural(Str::snake($table_data["table"]));

        // 各viewフォルダの作成
        if(!file_exists(resource_path("views/" . $view))) {
            $this->laravel['files']->makeDirectory(resource_path("views/" . $view), 0755, true);
        }

        // routeの追加
        $this->addRoute($table_data);

        // controller作成
        $this->createFrontControllerFile($table_data);

        // view作成
        $this->createFrontViewFile($table_data);

        if(in_array($table_data["front"]["type"], ["contact_form", "user_form"])) {
            $upper_table_name = strtoupper($table_data["table"]);
            $snake_case_name = Str::snake($table_data["table"]);
            if (file_exists(base_path(".env"))) {
                $env_text = file_get_contents(base_path(".env"));
                if(strpos($env_text, $upper_table_name . "_FROM_MAIL=") === false) {
                    $env_text .= "\n" . $upper_table_name . "_FROM_MAIL=no-reply@example.com";
                }

                if(!empty($table_data["front"]["options"]["reCaptcha"])) {
                    if(strpos($env_text, "RECAPTCHA_SITEKEY=") === false) {
                        $env_text .= "\n" . "RECAPTCHA_SITEKEY=";
                    }
                }

                if(!empty($table_data["front"]["options"]["reCaptcha"])) {
                    if(strpos($env_text, "RECAPTCHA_SECRET=") === false) {
                        $env_text .= "\n" . "RECAPTCHA_SECRET=";
                    }
                }

                file_put_contents(base_path(".env"), $env_text);
                $this->line("<info>change .env!</info>");
            }

            // const.php調整
            $const_path = config_path('const.php');
            $const_text = file_get_contents($const_path);
            if(strpos($const_text, "'" . $snake_case_name . "_from_mail' =>") === false) {
                $const_text = str_replace("];", "\t'" . $snake_case_name . "_from_mail' => [
        'address' => env('" . $upper_table_name . "_FROM_MAIL', 'no-reply@example.com'),
        'name' => env('APP_NAME', ''),
    ],\n\n];", $const_text);
            }

            if(strpos($const_text, "'re_captcha' =>") === false) {
                $const_text = str_replace("];", "\t're_captcha' => [
        'sitekey' => env('RECAPTCHA_SITEKEY', ''),
        'secret' => env('RECAPTCHA_SECRET', ''),
    ],\n\n];", $const_text);
            }

            file_put_contents($const_path, $const_text);

            // auth.php調整
            if($table_data["front"]["type"] == "user_form") {
                $auth_path = config_path('auth.php');
                $auth_text = file_get_contents($auth_path);
                $guard_base = "'guards' => [
        'web' => [
            'driver' => 'session',
            'provider' => 'users',
        ],

        'api' => [
            'driver' => 'token',
            'provider' => 'users',
        ],";
                $guard_pos = strpos($auth_text, $guard_base);
                if($guard_pos !== false) {
                    $kakko_base = "\n    ],";
                    $kakko_pos = strpos($auth_text, $kakko_base, $guard_pos + strlen($guard_base));
                    $auth_text = substr_replace($auth_text, "\n\n\t\t" .  "'" . $snake_case_name . "' => [
            'driver' => 'session',
            'provider' => '" . $view . "',
        ],\n", $kakko_pos, 0);
                }

                $provider_base = "'providers' => [
        'users' => [
            'driver' => 'eloquent',
            'model' => App\User::class,
        ],";
                $provider_pos = strpos($auth_text, $provider_base);
                if($guard_pos !== false) {
                    $kakko_base = "\n    ],";
                    $kakko_pos = strpos($auth_text, $kakko_base, $provider_pos + strlen($provider_base));
                    $auth_text = substr_replace($auth_text, "\n\n\t\t" .  "'" . $view . "' => [
            'driver' => 'eloquent',
            'model' => App\Models\\" . $table_data["table"] . "::class,
        ],\n", $kakko_pos, 0);
                }

                file_put_contents($auth_path, $auth_text);
            }

            exec("php artisan config:cache");

            if(!file_exists(app_path("Mail"))) {
                $this->laravel['files']->makeDirectory(app_path("Mail"), 0755, true);
            }

            $admin_mail = "Mail/" . $table_data["table"] . "Admin.php";
            if(!file_exists(app_path($admin_mail))) {
                $title_admin = ($table_data["front"]["type"] == "contact_form") ? "お問い合わせがありました。" : "ユーザ登録がありました。";
                $admin_mail_text = <<<EOT
<?php

namespace App\Mail;

use Illuminate\Bus\Queueable;
use Illuminate\Mail\Mailable;
use Illuminate\Queue\SerializesModels;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Support\Facades\View;

class {$table_data["table"]}Admin extends Mailable
{
    use Queueable, SerializesModels;

    /**
     * 文面で利用するデータ
     * @var
     */
    public \$data;

    /**
     * Create a new message instance.
     * ReminderSend constructor.
     * @param \$data
     */
    public function __construct(\$data)
    {
        \$this->data = \$data;
    }

    /**
     * Build the message.
     *
     * @return \$this
     */
    public function build()
    {
        \$from = config('const.{$snake_case_name}_from_mail');
        \$appName = config('app.name');
        \$prefix = !empty(\$appName) ? "【{\$appName}】" : '';
        return \$this->from(\$from['address'], \$from['name'])
            ->subject("{\$prefix}{$title_admin}")
            ->text('layouts.mail', ['content' => 'mails.{$view}.admin']);
    }
}
EOT;
                file_put_contents(app_path($admin_mail), $admin_mail_text);
            }
            else {
                $this->error('Already exists file!:' . app_path($admin_mail));
            }

            // ユーザメールを送るかどうか確認
            $send_user = false;
            foreach($table_data["fileds"] as $field => $field_info) {
                if($field_info["type"] == "email") {
                    $send_user = true;
                    break;
                }
            }

            if($send_user) {
                $user_mail = "Mail/" . $table_data["table"] . "User.php";
                if(!file_exists(app_path($user_mail))) {
                    $title_user = ($table_data["front"]["type"] == "contact_form") ? "お問い合わせありがとうございます。" : "ユーザ登録ありがとうございます。";
                    $user_mail_text = <<<EOT
<?php

namespace App\Mail;

use Illuminate\Bus\Queueable;
use Illuminate\Mail\Mailable;
use Illuminate\Queue\SerializesModels;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Support\Facades\View;

class {$table_data["table"]}User extends Mailable
{
    use Queueable, SerializesModels;

    /**
     * 文面で利用するデータ
     * @var
     */
    public \$data;

    /**
     * Create a new message instance.
     * ReminderSend constructor.
     * @param \$data
     */
    public function __construct(\$data)
    {
        \$this->data = \$data;
    }

    /**
     * Build the message.
     *
     * @return \$this
     */
    public function build()
    {
        \$from = config('const.{$snake_case_name}_from_mail');
        \$appName = config('app.name');
        \$prefix = !empty(\$appName) ? "【{\$appName}】" : '';
        return \$this->from(\$from['address'], \$from['name'])
            ->subject("{\$prefix}{$title_user}")
            ->text('layouts.mail', ['content' => 'mails.{$view}.user']);
    }
}
EOT;
                    file_put_contents(app_path($user_mail), $user_mail_text);
                }
                else {
                    $this->error('Already exists file!:' . app_path($user_mail));
                }
            }

            $mail_layout = "views/layouts/mail.blade.php";
            if(!file_exists(resource_path($mail_layout))) {
                $mail_layout_text = <<<EOT
@include(\$content)

※本メールに返信されても発信元にメールは届きませんのでご注意ください。

-------------------------
{{ config('app.name') }}
URL： {{ secure_url('/') }}
Copyright (c) Wiz Co., Ltd. All Rights Reserved.
EOT;
                file_put_contents(resource_path($mail_layout), $mail_layout_text);
            }
            else {
                $this->error('Already exists file!:' . resource_path($mail_layout));
            }

            if(!file_exists(resource_path("views/mails"))) {
                $this->laravel['files']->makeDirectory(resource_path("views/mails"), 0755, true);
            }
            if(!file_exists(resource_path("views/mails/" . $view))) {
                $this->laravel['files']->makeDirectory(resource_path("views/mails/" . $view), 0755, true);
            }

            $mail_admin = "views/mails/" . $view . "/admin.blade.php";
            if(!file_exists(resource_path($mail_admin))) {
                $body_admin = ($table_data["front"]["type"] == "contact_form") ? "以下のお問合わせがありました。" : "以下のユーザが登録されました。";
                $mail_admin_text = <<<EOT
{$body_admin}

@include('mails.{$view}._content')
EOT;
                file_put_contents(resource_path($mail_admin), $mail_admin_text);
            }
            else {
                $this->error('Already exists file!:' . resource_path($mail_admin));
            }

            if($send_user) {
                $mail_user = "views/mails/" . $view . "/user.blade.php";
                if(!file_exists(resource_path($mail_user))) {
                    $body_user = ($table_data["front"]["type"] == "contact_form") ? "お問い合わせいただきまして、ありがとうございます。

3営業日以内に返信が無い場合、お問い合わせフォームにご入力いただきましたメールアドレスを
お間違いになられているか、システムの不具合によりメールをお送りできていない可能性があります。

その際は誠に恐れ入りますが、再度、お問い合わせフォームよりお問い合わせ願います。

以下のお問い合わせ内容で受け付けました。" : "ユーザ登録いただきまして、ありがとうございます。


以下の内容で登録されました。";
                $append_body = ($table_data["front"]["type"] == "contact_form") ? "" : "また、メールアドレス確認のため、下記URLへアクセスしてください。
{{ secure_url('{$view}/email_verification/' . \$data['email_verification_token']) }}
";
                $mail_user_text = <<<EOT
{$body_user}
@include('mails.{$view}._content')

{$append_body}

※このメールに覚えのない方は、恐れ入りますがメールの破棄をお願い致します。
EOT;
                    file_put_contents(resource_path($mail_user), $mail_user_text);
                }
                else {
                    $this->error('Already exists file!:' . resource_path($mail_user));
                }
            }

            $mail_content = "views/mails/" . $view . "/_content.blade.php";
            if(!file_exists(resource_path($mail_content))) {
                $mail_content_text = "------------------------------------------------\n";
                foreach($table_data["fileds"] as $field => $field_info) {
                    if(!empty($table_data["front"]["options"]["hideForm"])) {
                        if(in_array($field, $table_data["front"]["options"]["hideForm"])) {
                            continue;
                        }
                    }
                    
                    if(!empty($field_info["comment"])) {
                        $mail_content_text .= '■' . $field_info["comment"] . "\n";
                    }

                    if(!in_array($field_info["type"], ["select", "radio", "checkbox"])) {
                        $mail_content_text .= '{{ $data[\'' . $field . '\'] }}';
                    }
                    else {
                        $mail_content_text .= '{{ !empty($data[\'' . $field . '\']) ? ';
                        if(!empty($field_info["values"])) {
                            $mail_content_text .= 'config(\'values.' . $field_info["values"] . '.\' . $data[\'' . $field . '\']) : \'\' }}';
                        }
                        else if(!empty($field_info["hasOne"])) {
                            $mail_content_text .= '$' . Str::plural(Str::snake($field_info["hasOne"])) . '[$data[\'' . $field . '\']] : \'\' }}';
                        }
                        else if($field_info["type"] == "checkbox") {
                            $mail_content_text .= "'" . $field_info["text"][1] . "' : '" . $field_info["text"][2] . "' }}";
                        }
                    }
                    $mail_content_text .= "\n";
                }
                $mail_content_text .= "------------------------------------------------\n";
                file_put_contents(resource_path($mail_content), $mail_content_text);
            }
            else {
                $this->error('Already exists file!:' . resource_path($mail_content));
            }
        }
    }

    /**
     * routesの追加
     */
    private function addRoute($table_data) {
        $routes_path = base_path('routes/web.php');
        $routes_text = file_get_contents($routes_path);

        $index_route = "Route::get('%s','%sController@index');";
        $show_route = "Route::get('%s/show/{id}','%sController@show');";
        $confirm_route = "Route::post('%s/confirm','%sController@confirm')->name('%s.confirm');";
        $resource_route = "Route::resource('%s', '%sController', ['only' => ['index', 'store']]);";
        $user_group = "Route::prefix('%s')->namespace('%s')->name('%s.')->group(function () {";

        $url = Str::plural(Str::snake($table_data["table"]));
        $controller = Str::plural($table_data["table"]);

        $routes_text .= "\n";
        $routes_text .= "// " . $table_data["title"] . "\n";
        if($table_data["front"]["type"] == "basic") {
            $routes_text .= sprintf($index_route, $url, $controller) . "\n";
            $routes_text .= sprintf($show_route, $url, $controller) . "\n";
        }
        else if($table_data["front"]["type"] == "contact_form") {
            if(!empty($table_data["front"]["options"]["confirm"])) {
                $routes_text .= sprintf($confirm_route, $url, $controller, $url) . "\n";
            }
            $routes_text .= sprintf($resource_route, $url, $controller) . "\n";
        }
        else if($table_data["front"]["type"] == "user_form") {
            $routes_text .= sprintf($user_group, $url, $table_data["table"], $url) . "\n";
            $routes_text .= "\t" . "Route::get('register', 'RegisterController@index')->name('register');
    Route::post('register_confirm','RegisterController@confirm')->name('register_confirm');
    Route::post('register', 'RegisterController@store')->name('register_store');
    Route::get('email_verification/{token}', 'EmailVerificationController@index');

    Route::get('login', 'LoginController@showLoginForm')->name('login');
    Route::post('login', 'LoginController@login');
    Route::post('logout', 'LoginController@logout')->name('logout');" . "\n";
            $routes_text .= "});\n";
        }

        file_put_contents($routes_path, $routes_text);
    }

    /**
     * 表側コントローラー作成
     */
    private function createFrontControllerFile($table_data) {
        if($table_data["front"]["type"] != "user_form") {
            $controller = Str::plural($table_data["table"]);
            $controller_path = app_path('Http/Controllers/' . $controller . 'Controller.php');
            if(file_exists(app_path($controller_path))) {
                $this->error('Already exists file!:' . $controller_path);
                return;
            }

            if($table_data["front"]["type"] == "basic") {
                $basic_text = <<<EOT
<?php

namespace App\Http\Controllers;

use Illuminate\Support\Facades\Request;
use App\Models\MODEL_NAME;

class CONTROLLER_NAMEController extends Controller
{
	/**
     * CONTROLLER_NAMEController constructor.
     */
    public function __construct()
    {
        MIDDLEWARE_TEXT
    }

    public function index()
    {
        \$params = Request::all();
        \$records = MODEL_NAME::createSearchQuery(\$params)->paginate(1);
        return view('VIEW_NAME.index', compact('records'));
    }

    public function show(\$id)
    {
        \$record = MODEL_NAME::createSearchQuery()->where('id', \$id)->firstOrFail();
        return view('VIEW_NAME.show', compact('record'));
    }
}
EOT;

                $basic_text = str_replace("CONTROLLER_NAME", $controller, $basic_text);
                $basic_text = str_replace("MODEL_NAME", (empty($table_data["isPrivatePublic"]) ? $table_data["table"] : "Public" . $table_data["table"]), $basic_text);
                $basic_text = str_replace("VIEW_NAME", Str::plural(Str::snake($table_data["table"])), $basic_text);

                foreach($table_data["fileds"] as $field => $field_info) {
                    if(!empty($field_info["hasOne"]) || ($field_info["type"] == "belongsToMany")) {
                        $middleware_name = Str::snake("Get" . $table_data["table"] . "Masters");
                        $basic_text = str_replace("MIDDLEWARE_TEXT", "// master取得
        \$this->middleware('" . $middleware_name . "')->only(['index', 'show']);", $basic_text);
                        break;
                    }
                }
                $basic_text = str_replace("MIDDLEWARE_TEXT", "", $basic_text);
            }
            else if($table_data["front"]["type"] == "contact_form") {
                $confirm_action = <<<EOT

    public function confirm(MODEL_NAMERequest \$request)
    {
        Larapack::form()->setIsConfirm(true);
        return view('VIEW_NAME.index');
    }

EOT;
                if(empty($table_data["front"]["options"]["confirm"])) {
                    $confirm_action = "";
                }

                $use_user_text = "";
                $email_text = "";
                foreach($table_data["fileds"] as $field => $field_info) {
                    if($field_info["type"] == "email") {
                        $use_user_text = "use App\Mail\MODEL_NAMEUser;\n";
                        $email_text = <<<EOT
        if (!empty(\$params['{$field}'])) {
            // user
            Mail::to(\$params['{$field}'])->send(new MODEL_NAMEUser(\$params));
        }
EOT;
                        break;
                    }
                }

                $basic_text = <<<EOT
<?php

namespace App\Http\Controllers;

use App\Http\Requests\MODEL_NAMERequest;
use App\Mail\MODEL_NAMEAdmin;
{$use_user_text}use App\Models\MODEL_NAME;
use App\Models\ADMIN_EMAIL;
use Illuminate\Support\Facades\Mail;
use Request;
use Uehi\Larapack\Facades\Larapack;

class CONTROLLER_NAMEController extends Controller
{

    public function index()
    {
        Larapack::form()->setIsConfirm(false);
        return view('VIEW_NAME.index');
    }
{$confirm_action}
    public function store(MODEL_NAMERequest \$request)
    {
        \$params = \$request->all();

        // 確認画面で戻る押された: 入力画面表示
        if (!empty(\$params['back'])) {
            return redirect()
                ->route('VIEW_NAME.index')
                ->withInput(\$request->except(['back']));
        }

        // 保存
        \$record = new MODEL_NAME(\$params);
        if (!\$record->save()) {
            abort(500);
        }

        // メール送信
        \$adminEmails = ADMIN_EMAIL::getMODEL_NAMEMails();
        if (!empty(\$adminEmails)) {
            // admin
            Mail::to(\$adminEmails)->send(new MODEL_NAMEAdmin(\$params));
        }
{$email_text}
        // ブラウザリロード等での二重送信防止
        \$request->session()->regenerateToken();

        // 完了画面を表示
        return view('VIEW_NAME.complete');
    }

}
EOT;
                $basic_text = str_replace("CONTROLLER_NAME", $controller, $basic_text);
                $basic_text = str_replace("MODEL_NAME", $table_data["table"], $basic_text);
                $basic_text = str_replace("VIEW_NAME", Str::plural(Str::snake($table_data["table"])), $basic_text);
                $basic_text = str_replace("ADMIN_EMAIL", $table_data["front"]["options"]["adminEmail"], $basic_text);
            }

            file_put_contents($controller_path, $basic_text);
        }
        else {
            $controller_path = app_path("Http/Controllers/" . $table_data["table"]);
            if(!file_exists($controller_path)) {
                $this->laravel['files']->makeDirectory($controller_path, 0755, true);
            }

            $register_text = <<<EOT
<?php

namespace App\Http\Controllers\MODEL_NAME;

use App\Http\Controllers\Controller;
use App\Http\Requests\MODEL_NAMERequest;
use App\Mail\MODEL_NAMEAdmin;
use App\Mail\MODEL_NAMEUser;
use App\Models\MODEL_NAME;
use App\Models\ADMIN_EMAIL;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Str;
use Request;
use Uehi\Larapack\Facades\Larapack;

class CONTROLLER_NAMEController extends Controller
{

    public function index()
    {
        Larapack::form()->setIsConfirm(false);
        return view('VIEW_NAME.index');
    }

    public function confirm(MODEL_NAMERequest \$request)
    {
        Larapack::form()->setIsConfirm(true);
        return view('VIEW_NAME.index');
    }

    public function store(MODEL_NAMERequest \$request)
    {
        \$params = \$request->all();

        // 確認画面で戻る押された: 入力画面表示
        if (!empty(\$params['back'])) {
            return redirect()
                ->route('VIEW_NAME.register')
                ->withInput(\$request->except(['back']));
        }

        // 保存
        \$record = new MODEL_NAME(\$params);
        if (!\$record->save()) {
            abort(500);
        }

        // メール認証トークン
        \$key = config('app.key');
        if (Str::startsWith(\$key, 'base64:')) {
            \$key = base64_decode(substr(\$key, 7));
        }
        \$token = hash_hmac('sha256', \$params['email'] . date('YmdHis') . Str::random(40), \$key);
        \$record->email_verification_token = \$token;
        if (!\$record->save()) {
            abort(500);
        }
        \$params['email_verification_token'] = \$token;

        // メール送信
        \$adminEmails = ADMIN_EMAIL::getMODEL_NAMEMails();
        if (!empty(\$adminEmails)) {
            // admin
            Mail::to(\$adminEmails)->send(new MODEL_NAMEAdmin(\$params));
        }
        if (!empty(\$params['EMAIL_FIELD'])) {
            // user
            Mail::to(\$params['EMAIL_FIELD'])->send(new MODEL_NAMEUser(\$params));
        }

        // ブラウザリロード等での二重送信防止
        \$request->session()->regenerateToken();

        // 完了画面を表示
        return view('VIEW_NAME.complete');
    }

}
EOT;
            $register_text = str_replace("CONTROLLER_NAME", "Register", $register_text);
            $register_text = str_replace("MODEL_NAME", $table_data["table"], $register_text);
            $register_text = str_replace("VIEW_NAME", Str::plural(Str::snake($table_data["table"])), $register_text);
            $register_text = str_replace("ADMIN_EMAIL", $table_data["front"]["options"]["adminEmail"], $register_text);
            foreach($table_data["fileds"] as $field => $field_info) {
                if($field_info["type"] == "email") {
                    $register_text = str_replace("EMAIL_FIELD", $field, $register_text);
                    break;
                }
            }
            file_put_contents($controller_path . "/RegisterController.php", $register_text);

            $email_verification_text = <<<EOT
<?php

namespace App\Http\Controllers\MODEL_NAME;

use App\Http\Controllers\Controller;
use App\Models\MODEL_NAME;

class EmailVerificationController extends Controller
{

    public function index(\$token)
    {
        \$record = MODEL_NAME::where('email_verification_token', \$token)->where('deleted_at', null)->first();
        if(!empty(\$record)) {
            \$record->mail_verified = 1;
            \$record->save();
        }

        return view('VIEW_NAME.email_verification')->with("isError", empty(\$record));
    }

}
EOT;
            $email_verification_text = str_replace("MODEL_NAME", $table_data["table"], $email_verification_text);
            $email_verification_text = str_replace("VIEW_NAME", Str::plural(Str::snake($table_data["table"])), $email_verification_text);
            file_put_contents($controller_path . "/EmailVerificationController.php", $email_verification_text);
        }
    }

    /**
     * 表側各種ビュー作成
     */
    private function createFrontViewFile($table_data) {
        $view = Str::plural(Str::snake($table_data["table"]));

        if($table_data["front"]["type"] == "basic") {
            $search_name = "views/" . $view . "/_search.blade.php";
            if(!file_exists(resource_path($search_name))) {
                $admin_search_name = "views/admin/" . $view . "/_search.blade.php";
                if(file_exists(resource_path($admin_search_name))) {
                    $search_text = file_get_contents(resource_path($admin_search_name));
                    $search_text = str_replace("<label>
        公開状態&nbsp;&nbsp;
        {{ Form::select('public', config('values.selectEmpty') + config('values.public'), null, ['class' => 'form-control']) }}
    </label>
", "", $search_text);
                    file_put_contents(resource_path($search_name), $search_text);
                }
            }
            else {
                $this->error('Already exists file!:' . $search_name);
            }

            if(!file_exists(resource_path("views/vendor/pagination"))) {
                $this->laravel['files']->makeDirectory(resource_path("views/vendor/pagination"), 0755, true);
            }
    
            $pagination_name = "views/vendor/pagination/front.blade.php";
            if(!file_exists(resource_path($pagination_name))) {
                $pagination_text = '
{{$paginator->total()}} 件中 {{$paginator->firstItem()}} 〜 {{$paginator->lastItem()}} 件を表示

@if ($paginator->hasPages())
    <ul class="pagination">
        {{-- Previous Page Link --}}
        @if ($paginator->onFirstPage())
            <li class="disabled"><span>&laquo;</span></li>
        @else
            <li><a href="{{ $paginator->previousPageUrl() }}" rel="prev">&laquo;</a></li>
        @endif

        {{-- Pagination Elements --}}
        @foreach ($elements as $element)
            {{-- "Three Dots" Separator --}}
            @if (is_string($element))
                <li class="disabled"><span>{{ $element }}</span></li>
            @endif

            {{-- Array Of Links --}}
            @if (is_array($element))
                @foreach ($element as $page => $url)
                    @if ($page == $paginator->currentPage())
                        <li class="active"><span>{{ $page }}</span></li>
                    @else
                        <li><a href="{{ $url }}">{{ $page }}</a></li>
                    @endif
                @endforeach
            @endif
        @endforeach

        {{-- Next Page Link --}}
        @if ($paginator->hasMorePages())
            <li><a href="{{ $paginator->nextPageUrl() }}" rel="next">&raquo;</a></li>
        @else
            <li class="disabled"><span>&raquo;</span></li>
        @endif
    </ul>
@endif
';
                file_put_contents(resource_path($pagination_name), $pagination_text);
            }

            $index_name = "views/" . $view . "/index.blade.php";
            if(!file_exists(resource_path($index_name))) {
                $index_text = <<<EOT
@extends('layouts.app')

@section('content')

SEARCH_TEXT

<pre>
一覧表示

@if (\$records->total() === 0)

条件に合うデータがありません

@else

@foreach(\$records as \$record)
RECORD_TEXT
-----
@endforeach
</pre>

{{ \$records->appends(request()->input())->links('vendor.pagination.front') }}

@endif

@endsection
EOT;
                if(file_exists(resource_path($search_name))) {
                    $index_text = str_replace("SEARCH_TEXT", "{{ Form::open(['method' => 'GET']) }}
    @include('{$view}._search')

    <a href=\"javascript:void(0);\" onclick=\"$(this).closest('form').find('textarea, :text, select').val('').end().find(':checked').prop('checked', false);\">クリア</a>
    {{ Form::submit('検索', array('class' => 'btn btn-primary')) }}
{{ Form::close() }}", $index_text);
                }
                else {
                    $index_text = str_replace("SEARCH_TEXT", "", $index_text);
                }

                $record_text = "";
                foreach($table_data["fileds"] as $field => $field_info) {
                    if(!empty($field_info["dispList"])) {
                        $record_text .= $field_info["comment"] . ":";
                        if($field_info["dispList"] == "link") {
                            $record_text .= '<a href="{{ url("/' . $view . '/show/{$record->id}") }}">';
                        }
            
                        if(!in_array($field_info["type"], ["select", "radio", "checkbox"])) {
                            $record_text .= '{{ $record->' . $field . ' }}';
                        }
                        else {
                            $record_text .= '{{ !empty($record->' . $field . ') ? ';
                            if(!empty($field_info["values"])) {
                                $record_text .= 'config(\'values.' . $field_info["values"] . '.\' . $record->' . $field . ') : \'\' }}';
                            }
                            else if(!empty($field_info["hasOne"])) {
                                $record_text .= '$' . Str::plural(Str::snake($field_info["hasOne"])) . '[$record->' . $field . '] : \'\' }}';
                            }
                            else if($field_info["type"] == "checkbox") {
                                $record_text .= "'" . $field_info["text"][1] . "' : '" . $field_info["text"][2] . "' }}";
                            }
                        }
            
                        if($field_info["dispList"] == "link") {
                            $record_text .= "</a>";
                        }

                        $record_text .= "\n";
                    }
                }

                $index_text = str_replace("RECORD_TEXT", $record_text, $index_text);

                file_put_contents(resource_path($index_name), $index_text);
            }
            else {
                $this->error('Already exists file!:' . $index_name);
            }

            $show_name = "views/" . $view . "/show.blade.php";
            if(!file_exists(resource_path($show_name))) {
                $show_text = <<<EOT
@extends('layouts.app')

@section('content')

<pre>
詳細表示

RECORD_TEXT
-----
</pre>

<a href="javascript:history.back();">一覧に戻る</a>

@endsection
EOT;
                $record_text = "";
                foreach($table_data["fileds"] as $field => $field_info) {
                    if(in_array($field_info["type"], ["password"])) {
                        continue;
                    }

                    $record_text .= $field_info["comment"] . ":";
                    if(in_array($field_info["type"], ["select", "radio", "checkbox"])) {
                        $record_text .= '{{ !empty($record->' . $field . ') ? ';
                        if(!empty($field_info["values"])) {
                            $record_text .= 'config(\'values.' . $field_info["values"] . '.\' . $record->' . $field . ') : \'\' }}';
                        }
                        else if(!empty($field_info["hasOne"])) {
                            $record_text .= '$' . Str::plural(Str::snake($field_info["hasOne"])) . '[$record->' . $field . '] : \'\' }}';
                        }
                        else if($field_info["type"] == "checkbox") {
                            $record_text .= "'" . $field_info["text"][1] . "' : '" . $field_info["text"][2] . "' }}";
                        }
                    }
                    else if(in_array($field_info["type"], ["imageFile"])) {
                        $record_text .= '{{ Html::image($record->' . $field . ') }}';
                    }
                    else if(in_array($field_info["type"], ["documentFile"])) {
                        $record_text .= '{{ Html::link($record->' . $field . ') }}';
                    }
                    else if(in_array($field_info["type"], ["map"])) {
                        $record_text .= '<div id="map" style="width:300px; height:300px;"></div>
<script>
function initMap() {
    var position = {lat: {{ $record->lat }}, lng: {{ $record->lon }}};
    var map = new google.maps.Map(document.getElementById(\'map\'), {zoom: {{ $record->zoom }}, center: position});
    var marker = new google.maps.Marker({position: position, map: map});
}
</script>
<script src="{{ Larapack::util()->gmapJsUrl() }}&callback=initMap" async defer></script>';
                    }
                    else if(in_array($field_info["type"], ["belongsToMany"])) {
                        $record_text .= '@foreach ($record->' . Str::plural(Str::snake($field)) . ' as $' . Str::snake($field) . ')
{{ $' . Str::snake($field) . '->title }}
@endforeach';
                        
                    }
                    else {
                        $record_text .= '{{ $record->' . $field . ' }}';
                    }
        
                    $record_text .= "\n";
                }

                $show_text = str_replace("RECORD_TEXT", $record_text, $show_text);

                file_put_contents(resource_path($show_name), $show_text);
            }
            else {
                $this->error('Already exists file!:' . $show_name);
            }
        }
        else if(in_array($table_data["front"]["type"], ["contact_form", "user_form"])) {
            $form_name = "views/" . $view . "/_form.blade.php";
            if(!file_exists(resource_path($form_name))) {
                $admin_form_name = "views/admin/" . $view . "/_form.blade.php";
                if(file_exists(resource_path($admin_form_name))) {
                    $form_text = file_get_contents(resource_path($admin_form_name));
                    if(!empty($table_data["front"]["options"]["privacy"])) {
                        $replace_base = "</tr>";
                        $replace_pos = strrpos($form_text, $replace_base);
                        $form_text = substr_replace($form_text, "
@if (!Larapack::form()->isConfirm())
    <div>
        {{ Larapack::form()->checkbox('agree', null, [], [
            '株式会社Wizの<a target=\"_blank\" href=\"http://012grp.co.jp/policy/\">個人情報保護方針</a>に同意する。',
            '同意する',
            '同意しない',
        ]) }}
    </div>
    {{ Larapack::form()->error('agree') }}
@endif", $replace_pos + strlen($replace_base), 0);
                    }

                    if(!empty($table_data["front"]["options"]["reCaptcha"])) {
                        $replace_base = "</tr>";
                        $replace_pos = strrpos($form_text, $replace_base);
                        $form_text = substr_replace($form_text, '
@if (!Larapack::form()->isConfirm())
<script src="https://www.google.com/recaptcha/api.js" async defer></script>
<div class="g-recaptcha" data-sitekey="{{ config(\'const.re_captcha.sitekey\') }}"></div>
{{ Larapack::form()->error(\'g-recaptcha-response\') }}
@endif', $replace_pos + strlen($replace_base), 0);
                    }

                    if(!empty($table_data["front"]["options"]["hideForm"])) {
                        foreach($table_data["front"]["options"]["hideForm"] as $hideForm) {
                            $form_text = str_replace($this->createAdminFormText($hideForm, $table_data["fileds"][$hideForm], $table_data), "", $form_text);
                        }
                    }

                    file_put_contents(resource_path($form_name), $form_text);
                }
            }

            $index_name = "views/" . $view . "/index.blade.php";
            if(!file_exists(resource_path($index_name))) {
                $is_register = ($table_data["front"]["type"] == "user_form") ? "register_" : "";
                $post_action = (!empty($table_data["front"]["options"]["confirm"])) ? "confirm" : "store";
                $post_text = (!empty($table_data["front"]["options"]["confirm"])) ? "内容を確認する" : "内容を送信する";
                $index_text = <<<EOT
@extends('layouts.app')

@section('content')

<pre>
    {$table_data["title"]}

    @if (!Larapack::form()->isConfirm())
        入力画面
    @else
        確認画面
    @endif

    @if (!Larapack::form()->isConfirm())
        {{ Form::open(['route' => ['{$view}.{$is_register}{$post_action}'], 'method' => 'post', 'id' => '{$view}_form']) }}
            @include('{$view}._form')
        {{ Form::close() }}

        <div class="btn01">
            <a href="javascript:void(0);" onclick="$('#{$view}_form').submit();">{$post_text}</a>
        </div>
    @else
        {{ Form::open(['route' => ['{$view}.{$is_register}store'], 'method' => 'post', 'id' => '{$view}_form']) }}
            @include('{$view}._form')

            {{-- 戻るフラグ --}}
            {{ Form::hidden('back', 1, ['id' => 'hiddenBack']) }}
            {{-- その他の入力値 --}}
            {{ Larapack::form()->hiddenVars(['back']) }}
        {{ Form::close() }}

        <div class="btnSet clearfix">
            <div class="btn01">
                <a href="javascript:void(0);" onclick="$('#hiddenBack').val('0');$('#{$view}_form').submit();">送信</a>
            </div>
            <div class="btn02">
                <a href="javascript:void(0);" onclick="$('#hiddenBack').val('1');$('#{$view}_form').submit();">修正する</a>
            </div>
        </div>
    @endif
</pre>

@endsection
EOT;
                file_put_contents(resource_path($index_name), $index_text);
            }

            $complete_name = "views/" . $view . "/complete.blade.php";
            if(!file_exists(resource_path($complete_name))) {
                $main_text = <<<EOT
    お問い合わせ完了

    お問い合わせが完了いたしました。
    3営業日以内に返信が無い場合、返信先メールアドレスの入力をお間違いになられているか、お問い合わせフォームの不具合により弊社にてメールが確認できていないことが考えられます。
    その場合、恐れ入りますが、再度お問い合わせをお願いいたします。

    <a href="{{ secure_url('/') }}">トップへ戻る</a>
EOT;
                if($table_data["front"]["type"] == "user_form") {
                    $main_text = <<<EOT
    ユーザ登録完了

    ユーザ登録が完了いたしました。
    下記よりログインください。

    <a href="{{ secure_url('/{$view}/login') }}">ログイン</a>
EOT;
                }
                $complete_text = <<<EOT
@extends('layouts.app')

@section('content')

<pre>
    {$table_data["title"]}
    
{$main_text}
</pre>

@endsection
EOT;
                file_put_contents(resource_path($complete_name), $complete_text);
            }
        }

        if($table_data["front"]["type"] == "user_form") {
            $email_verification_name = "views/" . $view . "/email_verification.blade.php";
            if(!file_exists(resource_path($email_verification_name))) {
                $email_verification_text = <<<EOT
@extends('layouts.app')

@section('content')

<pre>
    ユーザ情報81メール認証

    @if (!\$isError)
        メールの確認が終わりました。
    @else
        URLが間違っています
    @endif
</pre>

@endsection
EOT;
                file_put_contents(resource_path($email_verification_name), $email_verification_text);
            }
        }
    }

    /**
     * values.php作成
     */
    private function createValuesFile($value) {
        if(!is_array($value)) {
            $value = array($value);
        }

        $values_text = "";
        if(!file_exists(config_path("values.php"))) {
            $values_text = "<?php
return [
];
";
        }
        else {
            $values_text = file_get_contents(config_path("values.php"));
        }

        foreach($value as $values_data) {
            if(strpos($values_text, $values_data["name"]) !== false) {
                $this->error('Already Exist value!:' . $values_data["name"]);
                continue;
            }

            $value_text = "\n";
            if(!empty($values_data["comment"])) {
                $value_text .= "\t// " . $values_data["comment"] . "\n";
            }

            if(is_array($values_data["value"])) {
                $value_text .= "\t'" . $values_data["name"] . "' => [\n";
                foreach($values_data["value"] as $value_key => $value_val) {
                    $value_text .= "\t\t'" . $value_key . "' => '" . $value_val . "',\n";
                }
                $value_text .= "\t],\n";
            }
            else {
                $value_text .= "\t'" . $values_data["name"] . "' => '" . $values_data["value"] . "',\n";
            }

            $values_text = str_replace("];", $value_text . "];", $values_text);
        }

        file_put_contents(config_path("values.php"), $values_text);

        exec("php artisan config:cache");
    }

    /**
     * ckeditorのinstall
     */
    private function ckeditorInstall($value) {
        $dest = (!empty($value["dest"])) ? $value["dest"] : "/js/admin/";
        if(!file_exists(public_path($dest))) {
            $this->laravel['files']->makeDirectory(public_path($dest), 0755, true);
        }

        if(!file_exists(realpath(public_path($dest)) . "/ckeditor")) {
            $version = (!empty($value["version"])) ? $value["version"] : "4.10.0";
            $type = (!empty($value["type"])) ? $value["type"] : "full";
            exec("cd " . public_path($dest) . " && wget https://download.cksource.com/CKEditor/CKEditor/CKEditor%20" . $version . "/ckeditor_" . $version . "_" . $type . ".zip");
            exec("cd " . public_path($dest) . " && unzip ckeditor_" . $version . "_" . $type . ".zip");
            exec("cd " . public_path($dest) . " && rm ckeditor_" . $version . "_" . $type . ".zip");
        }

        $this->ckeditor_path = $dest;
    }
}
