<?php

namespace Uehi\Larapack\Models;

use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\File;

/**
 * ファイルアップロード処理を含む
 * Trait FileUploads
 * @package Uehi\Larapack\Models
 */
trait FileUploads
{

    /**
     * fillableにcolumn追加したかどうか
     * @var bool
     */
    private $addedUploadColumns = false;

    /*
    |--------------------------------------------------------------------------
    | model上書き用
    |--------------------------------------------------------------------------
    */

    /**
     * アップロード対象のcolumnをget
     * @return array : 以下のようにstorageのpublic/private どちらに保存するか指定
     *      ['public_image1', 'public_image2'];
     *      ['public_image1', 'public_image2', 'private' => ['private_image1', 'private_image2']];
     *      ['public' => ['public_image1', 'public_image2'], 'private' => ['private_image1', 'private_image2']];
     */
    protected function getUploadColumns()
    {
        return [];
    }

    /**
     * privateなfileの表示権限を返す。trueの場合表示可能
     *      /uploader/read で使われる。
     * @param $column
     * @return bool
     */
    public function privateAuthorize($column)
    {
        return true;
    }

    /*
    |--------------------------------------------------------------------------
    | アップロードパラメータ保持用のaccessor
    |--------------------------------------------------------------------------
    */

    /**
     * アップロードパラメータ保持用のaccessor名
     */
    private function getUploadSupportColumns()
    {
        $inputSuffix = Config::get('const.uploads.input_suffix');
        $deleteSuffix = Config::get('const.uploads.delete_suffix');
        $columns = [];
        $normalized = $this->normalizedUploadColumns();
        foreach ($normalized as $type => $columnsInType) {
            foreach ($columnsInType as $column) {
                $columns[] = "{$column}{$inputSuffix}";
                $columns[] = "{$column}{$deleteSuffix}";
            }
        }
        return $columns;
    }

    protected $uploadInfo = [];
    protected function setUploadInfo($key, $value)
    {
        $this->uploadInfo[$key] = $value;
    }
    protected function getUploadInfo($key)
    {
        return !empty($this->uploadInfo[$key]) ? $this->uploadInfo[$key] : null;
    }

    /**
     * override アップロードパラメータを保持するためattributes拡張
     * @param array $attributes
     * @return $this
     */
    public function fill(array $attributes)
    {
        parent::fill($attributes);

        // ファイルアップに必要な情報をオブジェクト内に格納しておく
        foreach ($this->getUploadSupportColumns() as $column) {
            if (!empty($attributes[$column])) {
                $this->setUploadInfo($column, $attributes[$column]);
            }
        }


        return $this;
    }

    /*
    |--------------------------------------------------------------------------
    | boot
    |--------------------------------------------------------------------------
    */

    /**
     * boot
     */
    public static function bootFileUploads()
    {
        $self = new self;
        $uploadColumns = $self->normalizedUploadColumns();
        $inputSuffix = Config::get('const.uploads.input_suffix');
        $deleteSuffix = Config::get('const.uploads.delete_suffix');

        // ファイル名が渡ってきたら本来の保存領域に入れる
        static::saved(function ($record) use ($uploadColumns, $inputSuffix, $deleteSuffix) {
            if (!empty($uploadColumns)) {
                $isUpdate = false;
                foreach ($uploadColumns as $type => $columns) {
                    foreach ($columns as $column) {
                        // requestから保存ファイル名、削除フラグある場合取得
                        $inputColumn = "{$column}{$inputSuffix}";
                        $deleteColumn = "{$column}{$deleteSuffix}";
                        $tmpName = !empty($record->getUploadInfo($inputColumn)) ? $record->getUploadInfo($inputColumn) : null;
                        $deleteFlag = !empty($record->getUploadInfo($deleteColumn));

                        // 存在する場合
                        if (!empty($tmpName)) {
                            // 新しいファイル(tmp dirが含まれている時tmpだと判断)ある場合tmpから移動
                            if (static::isTmpPath($tmpName)) {
                                if ($type === 'private') {
                                    $tmpPath = storage_path('app/' . $tmpName);
                                } else {
                                    $tmpPath = public_path($tmpName);
                                }

                                if (is_file($tmpPath)) {
                                    if ($type === 'private') {
                                        $baseDir = storage_path('app/');
                                        $saveDir = $record->getTable() . '/' . floor($record->id / 1000) . '/';
                                    } else {
                                        $baseDir = public_path('/');
                                        $saveDir = 'storage/' . $record->getTable() . '/' . floor($record->id / 1000) . '/';
                                    }

                                    if (!file_exists($baseDir . $saveDir)) {
                                        File::makeDirectory($baseDir . $saveDir, 0707, true);
                                    }

                                    $newName = $saveDir . basename($tmpPath);
                                    if (rename($tmpPath, $baseDir . $newName)) {
                                        chmod($baseDir . $newName, 0707);
                                        // privateの場合sessionを消しておく
                                        static::deletePrivateSession($tmpName);
                                    }

                                    // modelにset
                                    $record->{$column} = $newName;
                                    $isUpdate = true;
                                }
                            }
                        }

                        // deleteフラグがある場合null set
                        if ($deleteFlag && !empty($record->{$column})) {
                            $record->{$column} = null;
                            $isUpdate = true;
                        }

                        // 指定fieldがdirtyかどうか
                        // dirtyなら以前のファイルは削除
                        if ($record->isDirty($column)) {
                            $original = $record->getOriginal($column);
                            $path = public_path($original);
                            if (!empty($original) && $original !== $record->{$column} && is_file($path)) {
                                unlink($path);
                            }
                        }

                    }
                }

                // 変更あればsave
                if ($isUpdate) {
                    $record->save();
                }
            }
            return true;
        });

    }

    /*
    |--------------------------------------------------------------------------
    | ファイル格納関連
    |--------------------------------------------------------------------------
    */

    /**
     * form値をtmpディレクトリに移動
     * FIXME: publicディレクトリ固定なので悩ましい
     * @param $column
     * @return bool
     */
    public function uploadToTmp($column)
    {
        $request = app('request');
        $params = $request->all();

        // hasManyの場合, $columnはchilds[1][image]のような形で来るので本来のcolumn名を抽出
        if (preg_match('/^(.+)\[([0-9]+)\]\[(.+)\]$/', $column, $m)) {
            $hasMany = true;
            $childProp = $m[1];
            $childNum = $m[2];
            $childColumn = $m[3];
            $value = !empty($params[$childProp][$childNum][$childColumn]) ? $params[$childProp][$childNum][$childColumn] : null;
        } else {
            $hasMany = false;
            $value = !empty($params[$column]) ? $params[$column] : null;
        }

        if (!empty($value)) {
            if (!$hasMany && ($type = $this->getUploadType($column))) {
                // 通常
                $path = ($type === 'public' ? 'public/' : '') . Config::get('const.uploads.tmp_dir');
                $filename = $request->file($column)->store($path);
            } elseif ($hasMany && ($type = $this->getUploadType($childColumn))) {
                // hasMany
                $path = ($type === 'public' ? 'public/' : '') . Config::get('const.uploads.tmp_dir');
                $filename = $request->file($childProp)[$childNum][$childColumn]->store($path);
            } else {
                return false;
            }

            if ($type === 'public') {
                // public/tmp/~~.png => storage/tmp/~~.png
                $ret = preg_replace('/^public\//', 'storage/', $filename);
            } else {
                // tmp/~~.png
                $ret = $filename;
            }
            return $ret;
        }
        return false;
    }

    /*
    |--------------------------------------------------------------------------
    | 共通処理
    |--------------------------------------------------------------------------
    */

    /**
     * filenameがtmpかどうかを判断
     * tmp dirが含まれている時tmpだと判断
     * @param $filename
     */
    public static function isTmpPath($filename)
    {
        $publicTmp = 'storage/' . Config::get('const.uploads.tmp_dir') . '/';
        $privateTmp = Config::get('const.uploads.tmp_dir') . '/';
        return strpos($filename, $publicTmp) === 0 || strpos($filename, $privateTmp) === 0;
    }

    /**
     * 指定columnの保存がpublicかprivateかを返す
     * falseの場合、そのcolumnはupload columnではない
     * @param $column
     * @return bool|int|string
     */
    public function getUploadType($column)
    {
        $normalized = $this->normalizedUploadColumns();
        foreach ($normalized as $type => $upColumns) {
            foreach ($upColumns as $upColumn) {
                if ($upColumn === $column) {
                    return $type;
                }
            }
        }
        return false;
    }

    /**
     * getUploadColumnsにて指定された配列をpublic/private分類した以下の形に整形
     *      ['public' => ['public_image1', 'public_image2'], 'private' => ['private_image1', 'private_image2']]
     * @return array
     */
    private function normalizedUploadColumns()
    {
        $normalized = ['public' => [], 'private' => []];
        foreach ($this->getUploadColumns() as $key => $val) {
            if (is_array($val)) {
                foreach ($val as $childVal) {
                    if ($key === 'private') {
                        // private
                        $normalized['private'][] = $childVal;
                    } else {
                        // public
                        $normalized['public'][] = $childVal;
                    }
                }
            } else {
                // public
                $normalized['public'][] = $val;
            }
        }
        return $normalized;
    }

    /**
     * sessionに保存しているfilepathを消す
     * priavteの場合,sessionにfilepathを保存している
     * @param $filename
     */
    private static function deletePrivateSession($filename)
    {
        $key = config('const.uploads.private_session_key');
        $sessionValue = app('request')->session()->get($key);
        if (empty($sessionValue) || !is_array($sessionValue)) {
            return true;
        }
        foreach ($sessionValue as $key => $val) {
            if ($val === $filename) {
                unset($sessionValue[$key]);
            }
        }
        return app('request')->session()->put($key, $sessionValue);
    }

}