<?php

namespace Uehi\Larapack\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\Config;

class InstallCommand extends Command
{
    /**
     * コマンド名
     * @var string
     */
    protected $name = 'larapack:install';

    /**
     * 説明
     * @var string
     */
    protected $description = 'Install Larapack';

    /**
     * Execute
     * @return void
     */
    public function handle()
    {
        $this->publishDatabase();
        $this->createFiles();
        $this->createFileDirectories();
        $this->publishFiles();
    }

    /**
     * DBテーブルを作成し、データを入れる
     * @return void
     */
    public function publishDatabase()
    {
        $this->call('migrate', ['--path' => str_replace(base_path(), '', __DIR__) . '/../../database/migrations/']);
        $this->call('db:seed', ['--class' => \Uehi\Larapack\Database\Seeds\AdminUserTableSeeder::class]);
    }

    /**
     * 必要なファイルを外出しする
     * @return void
     */
    public function publishFiles()
    {
        $this->call('vendor:publish', ['--tag' => 'larapack']);
    }

    /**
     * ファイルアップ関連ディレクトリを作成
     */
    public function createFileDirectories()
    {
        // storageを外出し
        $this->call('storage:link');

        $directories = [
            ["upload tmp dir", upload_tmp_dir()],
        ];

        foreach ($directories as $directory) {
            $this->createDir($directory[0], $directory[1]);
        }


    }

    /**
     * 必要なファイルを作成
     */
    public function createFiles()
    {
        $files = [
            ['AdminUser Model', app_path('Models') . '/AdminUser.php', $this->getStub('AdminUser')],
            ['AdminUser Controller', app_path('Http/Controllers/Admin') . '/AdminUsersController.php', $this->getStub('AdminUsersController')],
            ['AdminUser Request', app_path('Http/Requests') . '/AdminUserRequest.php', $this->getStub('AdminUserRequest')],
        ];

        foreach ($files as $file) {
            $this->putFile($file[0], $file[1], $file[2]);
        }
    }

    /**
     * ファイル作成
     * @param $logText: 標準出力に出す名前
     * @param $path: ファイルパス
     * @param $content: ファイルに記述するtext
     */
    protected function putFile($logText, $path, $content)
    {
        // 既に存在する場合なにもしない
        if (file_exists($path)) {
            $this->line("<info>{$logText} file has already exists</info>");
            return;
        }

        // ディレクトリなければ作成
        $this->makeDir(dirname($path));

        // ファイル作成
        $this->laravel['files']->put(
            $path,
            $content
        );
        $this->line("<info>{$logText} file was created:</info> " . $path);
    }

    /**
     * ファイル作成
     * @param $logText: 標準出力に出す名前
     * @param $path: ディレクトリパス
     */
    protected function createDir($logText, $path)
    {
        // 既に存在する場合なにもしない
        if (file_exists($path)) {
            $this->line("<info>{$logText} directory has already exists</info>");
            return;
        }

        // ディレクトリ作成
        $this->laravel['files']->makeDirectory($path, 0707, true);

        $this->line("<info>{$logText} directory was created:</info> " . $path);
    }

    /**
     * stubの中身を取得
     * @param $name
     * @return mixed
     */
    protected function getStub($name)
    {
        return $this->laravel['files']->get(__DIR__ . "/stubs/{$name}.stub");
    }

    /**
     * ディレクトリ作成
     * あれば何もしない
     * @param $path
     */
    protected function makeDir($path)
    {
        if (!file_exists($path)) {
            // 再帰的に作成
            $this->laravel['files']->makeDirectory($path, 0755, true);
        }
    }

}
