<?php

namespace Uehi\Larapack\Middleware;

use Closure;
use Illuminate\Support\Facades\Route;

/**
 * Class PrefixBasicAuthMiddleware
 * @package Uehi\Larapack\Middleware
 * (※) config/const.php にて以下指定
 * 'prefix_basic_auth' => [
 *      'env' => [
 *           // 'env(複数の場合カンマ区切り)' => '設定値で記述'
 *           'develop,dev,staging' => ['*'],
 *           'production' => ['*'],
 *       ],
 *       // ['id', 'パスワード']を複数指定可
 *       'idpass' => [
 *           ['admin', 'admin']
 *      ],
 * ],
 * (※) 設定値の書き方
 *      例1) 全てかける               : ['*'] ...... *要素を配列にいれると全てかける
 *      例2) 全て認証外す             : [] or false ...... emptyだと認証がなにもかからない
 *      例3) adminのみかける          : ['deny' => ['admin']] ...... denyにadminを含める
 *      例4) 表側とapiだけ認証外す    : ['allow' => ['api', '']] ...... allowに表側(prefixなし'')とapiを含める
 *      例5) 表側とapiだけ認証外す    : ['allow' => ['api', '']] ...... allowに表側(prefixなし'')とapiを含める
 */
class PrefixBasicAuthMiddleware
{

    /**
     * Handle an incoming request.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \Closure  $next
     * @return mixed
     */
    public function handle($request, Closure $next)
    {
        // config
        $idpass = config('const.prefix_basic_auth.idpass');
        $settings = config('const.prefix_basic_auth.env');
        $ip = config('const.prefix_basic_auth.ip');

        if(in_array($request->ip(), $ip)) {
            return $next($request);
        }

        // 対象の設定値を取得
        $setting = $this->getSetting($settings);
        if (empty($setting)) {
            // ない場合なにもしない
            return $next($request);
        }

        // 対象のenvの場合
        if ($this->isTargetPrefix($setting)) {
            // 入力値
            $requestUser = $request->getUser();
            $requestPass = $request->getPassword();

            // 該当するid, passがあるか
            $success = false;
            foreach ($idpass as $val) {
                if ($val[0] === $requestUser && $val[1] === $requestPass) {
                    $success = true;
                    break;
                }
            }

            if (!$success) {
                $header = ['WWW-Authenticate' => 'Basic'];
                // メッセージ吐き出し
                return response('Authorization Required', 401, $header);
            }
        }

        return $next($request);
    }

    /**
     * 現在のenvから設定値を取得
     * @param $settings
     * @return bool
     */
    private function getSetting($settings)
    {
        $currentEnv = app()->environment();
        foreach ($settings as $envs => $setting) {
            $envs = explode(',', $envs);
            if (in_array($currentEnv, $envs)) {
                return $setting;
            }
        }
        return false;
    }

    /**
     * 現在のprefixがBasicかけるprefixかどうか
     * @param $prefix
     * @return bool
     */
    private function isTargetPrefix($setting)
    {
        $currentPrefix = Route::getCurrentRoute()->getPrefix();

        // 該当のprefixでない場合スルー
        if (in_array('*', $setting)) {
            // 全てかける
            return true;
        } elseif (!empty($setting['deny']) && in_array($currentPrefix, $setting['deny'])) {
            // 現在のprefixがdenyに含まれる場合、basicかける
            return true;
        } elseif (!empty($setting['allow']) && !in_array($currentPrefix, $setting['allow'])) {
            // 現在のprefixがallowに含まれない場合、basicかける
            return true;
        } else {
            // それ以外かけない
            return false;
        }

    }

}
