<?php

namespace Uehi\Larapack\Controllers;

use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Request;
use Illuminate\Support\Facades\Validator;
use ReflectionClass;

class UploaderController extends Controller
{

    /**
     * アップロード処理
     * post
     *      image(カラム名)
     *      model
     *      request
     *      column
     *      error_options
     *      image_options
     */
    public function upload()
    {
        // validate
        // もともとのFormRequestからファイルアップcolumnのruleのみを抜粋する
        $params = Request::all();

        // hasManyかどうか
        $hasMany = false;
        if (preg_match('/^(.+)\[([0-9]+)\]\[(.+)\]$/', $params['column'], $m)) {
            $hasMany = true;
            $childProp = $m[1];
            $childNum = $m[2];
            $childColumn = $m[3];
        }

        $formRequest = $params['request'];
        $formRequest = new $formRequest;
        $column = $params['column'];
        $rules = $formRequest->rules();
        if ($hasMany) {
            $ruleKey = "{$childProp}.*.{$childColumn}";
            $rules = !empty($rules[$ruleKey]) ? [$ruleKey => $rules[$ruleKey]] : [];
        } else {
            $rules = !empty($rules[$column]) ? [$column => $rules[$column]] : [];
        }
        $validator = Validator::make($params, $rules, $formRequest->messages(), $formRequest->attributes());

        // view
        $errorOptions = is_array($params['error_options']) ? $params['error_options'] : json_decode($params['error_options'], true);
        if (isset($params['image_options'])) {
            // 画像
            $viewName = !empty($params['htmlTemplate']) ? $params['htmlTemplate'] : 'larapack::uploader.image';
            $imageOptions = is_array($params['image_options']) ? $params['image_options'] : json_decode($params['image_options'], true);
            $viewData = [
                'column' => $column,
                'filename' => null,
                'errorOptions' => $errorOptions,
                'imageOptions' => $imageOptions,
            ];
        } else {
            // ファイル
            $viewName = !empty($params['htmlTemplate']) ? $params['htmlTemplate'] : 'larapack::uploader.document';
            $viewData = [
                'column' => $column,
                'filename' => null,
                'errorOptions' => $errorOptions,
            ];
        }

        // validate
        if ($validator->fails()) {
            app('view')->share('errors', $validator->messages()); // shareに登録されないので手動で
            return view($viewName, $viewData)->withErrors($validator->messages());
        } else {
            // ファイルをtmp領域に移す
            $modelName = $params['model'];
            $model = new $modelName;
            if ($hasMany) {
                $childModelName = (new ReflectionClass($model->$childProp()->getRelated()))->getName();
                $childModel = new $childModelName;
                $filename = $childModel->uploadToTmp($column);
                $storageType = $childModel->getUploadType($column);
            } else {
                $filename = $model->uploadToTmp($column);
                $storageType = $model->getUploadType($column);
            }

            $viewData['filename'] = $filename;
            $viewData['storageType'] = $storageType;

            // image src / documentリンク を格納
            $src = url("/{$filename}"); // public
            if ($storageType === 'private') {
                // private
                $src = url("/uploader/readTmp/{$filename}");
            }
            $viewData['src'] = $src;

            // private の場合、filenameをsessionに保存しておく
            // sessionに入ったfilenameだけ表示できるようにするということ
            if ($storageType === 'private') {
                $this->addPrivateSession($filename);
            }

            return view($viewName, $viewData);
        }
    }

    /**
     * privateのtmp用ファイルを返す
     * ※private領域専用
     * @param $filename
     */
    public function readTmp($filename)
    {
        // sessionに入っていなければだめ
        if (!$this->containsPrivateSession($filename)) {
            abort(403, 'has no authorization to read');
        }
        return response()->file(storage_path("app/{$filename}"));
    }

    /**
     * privateのファイルを返す
     * ※private領域専用
     * @param $table
     * @param $id
     * @param $column
     * @return mixed
     */
    public function read($table, $id, $column)
    {
        $model = app('larapack')->util()->getModelByTableName($table);
        $record = $model::findOrfail($id);
 
        // 閲覧権限なければ403

        if (!$record->privateAuthorize($column)) {
            abort(403, 'has no authorization to read');
        }

        return response()->file(storage_path("app/{$record->{$column}}"));
    }

    /**
     * tmp画像パスをsessionに保存
     * privateのtmp用
     * @param $filename
     * @return mixed
     */
    private function addPrivateSession($filename)
    {
        $key = config('const.uploads.private_session_key');
        $sessionValue = app('request')->session()->get($key);
        if (empty($sessionValue)) $sessionValue = [];
        $sessionValue[] = $filename;
        return app('request')->session()->put($key, $sessionValue);
    }

    /**
     * tmp画像パスがsessionにあるかチェック
     * privateのtmp用
     * @param $filename
     * @return bool
     */
    private function containsPrivateSession($filename)
    {
        $key = config('const.uploads.private_session_key');
        $sessionValue = app('request')->session()->get($key);
        if (!is_array($sessionValue)) return false;
        return in_array($filename, $sessionValue);

    }

}
