/**
 * jQuery.inputFileResizer.js
 * an image resizing in input type file
 * @author Hiroki Ueno
 */
(function($){

	/**
	 * default params
	 */
	var defaults = {
		maxWidth: 1500,
		maxHeight: 1500,
		// callback
		onStart: null,
		onComplete: null,
		onError: null
	};

	$.inputFileResizer = function(elem , options){
		this.init(elem, options);
	}

	// ---------------------------------------------------

	/**
	 * plugin detail
	 */
	$.extend($.inputFileResizer.prototype, {

		/**
		 * initialize
		 */
		init: function(elem, options){

			/**
			 * target object
			 */
			this.elem = elem;

			/**
			 * options
			 */
			this.options = $.extend(defaults, options);

			/**
			 * do
			 */
			var that = this;
			if(that.options.onStart){
				that.options.onStart(that.elem);
			}
			that.resize();

		},

		/**
		 * resize
		 */
		resize: function(){
			// type check
			var file = $(this.elem).get(0).files[0];
			// androidでtype=nullになるので、その場合ファイル名から取得
			var type = file.type;
			if(!type) type = getImageType(file.name);

			if (!type.match(/^image\/(png|jpeg|gif)$/)){
				// callback
				if(this.options.onError){
					this.options.onError('not image! file type is :' + type);
				}
				return;
			}

			// 画像縮小
			var img = new Image();
			var reader = new FileReader();
			var maxWidth = this.options.maxWidth;
			var maxHeight = this.options.maxHeight;

			var that = this;
			reader.onload = function(e) {
				var data = e.target.result;
				// androidでmimetypeが含まれないものがあったので
				var dataSplit = data.split(',');
				if(dataSplit[0].match(/^data:base64$/)){
					data = 'data:' + type + ';base64,' + dataSplit[1];
				}

				img.onload = function() {

					// gifの場合は本処理はしない
					if (data.split(',')[0].match('gif')) {
						// callback
						if(that.options.onComplete){
							that.options.onComplete(null, null, null);
						}
						return;
					}

					var iw = img.naturalWidth, ih = img.naturalHeight;
					var width = iw, height = ih;

					var orientation;

					// JPEGの場合には、EXIFからOrientation（回転）情報を取得
					if (data.split(',')[0].match('jpeg')) {
						orientation = getOrientation(data);
					}
					// JPEG以外や、JPEGでもEXIFが無い場合などには、標準の値に設定
					orientation = orientation || 1;

					// 90度回転など、縦横が入れ替わる場合には事前に最大幅、高さを入れ替えておく
					if (orientation > 4) {
						var tmpMaxWidth = maxWidth;
						maxWidth = maxHeight;
						maxHeight = tmpMaxWidth;
					}

					if(width > maxWidth || height > maxHeight) {
						var ratio = width/maxWidth;
						if(ratio <= height/maxHeight) {
							ratio = height/maxHeight;
						}
						width = Math.floor(img.width/ratio);
						height = Math.floor(img.height/ratio);
					}

					var canvas = $('<canvas>');
					var ctx = canvas[0].getContext('2d');
					ctx.save();

					// EXIFのOrientation情報からCanvasを回転させておく
					transformCoordinate(canvas, width, height, orientation);

					// 縮小のimgをdrawする
					img.width = width;
					img.height = height;
					ctx.drawImage(img, 0, 0, width, height);

					// プレビューするために<img>用のDataURLを作成
					// (スマホなどの狭小画面でも画像の全体図が見れるように、解像度を保ったたま縮小表示したいので)
					var displaySrc = '';
					if (data.split(',')[0].match('jpeg')) {
						displaySrc = ctx.canvas.toDataURL(type, .9);
					} else {
						displaySrc = ctx.canvas.toDataURL(type);
					}
					var displayImg = $('<img>').attr({'src': displaySrc, 'id': 'preview'}).css({'maxWidth':'90%', 'maxHeight':'90%'});

					// FormDataに縮小後の画像データを追加する
					// Blob形式にすることで、サーバ側は従来のままのコードで対応可能
					var blob = dataURLtoBlob(displaySrc);

					// ↑※AndroidでBlobアップすると空画像になるバグがあるので断念

					// callback
					if(that.options.onComplete){
						// that.options.onComplete(displaySrc, displayImg, file);
						that.options.onComplete(blob, displayImg, file);
					}

				}
				img.src = data;
			}
			reader.readAsDataURL(file);
		}

	});

	// JPEGのEXIFからOrientationのみを取得する
	function getOrientation(imgDataURL){
		var byteString = atob(imgDataURL.split(',')[1]);
		var orientaion = byteStringToOrientation(byteString);
		return orientaion;

		function byteStringToOrientation(img){
			var head = 0;
			var orientation;
			while (1){
				if (img.charCodeAt(head) == 255 & img.charCodeAt(head + 1) == 218) {break;}
				if (img.charCodeAt(head) == 255 & img.charCodeAt(head + 1) == 216) {
					head += 2;
				}
				else {
					var length = img.charCodeAt(head + 2) * 256 + img.charCodeAt(head + 3);
					var endPoint = head + length + 2;
					if (img.charCodeAt(head) == 255 & img.charCodeAt(head + 1) == 225) {
						var segment = img.slice(head, endPoint);
						var bigEndian = segment.charCodeAt(10) == 77;
						if (bigEndian) {
							var count = segment.charCodeAt(18) * 256 + segment.charCodeAt(19);
						} else {
							var count = segment.charCodeAt(18) + segment.charCodeAt(19) * 256;
						}
						for (i=0;i<count;i++){
							var field = segment.slice(20 + 12 * i, 32 + 12 * i);
							if ((bigEndian && field.charCodeAt(1) == 18) || (!bigEndian && field.charCodeAt(0) == 18)) {
								orientation = bigEndian ? field.charCodeAt(9) : field.charCodeAt(8);
							}
						}
						break;
					}
					head = endPoint;
				}
				if (head > img.length){break;}
			}
			return orientation;
		}
	}

	// iPhoneのサブサンプリングを検出
	function detectSubsampling(img) {
		var iw = img.naturalWidth, ih = img.naturalHeight;
		if (iw * ih > 1024 * 1024) {
			var canvas = $('<canvas>');
			canvas[0].width = canvas[0].height = 1;
			var ctx = canvas[0].getContext('2d');
			ctx.drawImage(img, -iw + 1, 0);
			return ctx.getImageData(0, 0, 1, 1).data[3] === 0;
		} else {
			return false;
		}
	}

	// iPhoneの縦画像でひしゃげて表示される問題
	function detectVerticalSquash(img, iw, ih) {
		var canvas = $('<canvas>');
		canvas[0].width = 1;
		canvas[0].height = ih;
		var ctx = canvas[0].getContext('2d');
		ctx.drawImage(img, 0, 0);
		var data = ctx.getImageData(0, 0, 1, ih).data;
		var sy = 0;
		var ey = ih;
		var py = ih;
		while (py > sy) {
			var alpha = data[(py - 1) * 4 + 3];
			if (alpha === 0) {
				ey = py;
			} else {
				sy = py;
			}
			py = (ey + sy) >> 1;
		}
		var ratio = (py / ih);
		return (ratio===0) ? 1 : ratio;
	}

	function transformCoordinate(canvas, width, height, orientation) {
		if (orientation > 4) {
			canvas[0].width = height;
			canvas[0].height = width;
		} else {
			canvas[0].width = width;
			canvas[0].height = height;
		}
		var ctx = canvas[0].getContext('2d');
		switch (orientation) {
			case 2:
				// horizontal flip
				ctx.translate(width, 0);
				ctx.scale(-1, 1);
				break;
			case 3:
				// 180 rotate left
				ctx.translate(width, height);
				ctx.rotate(Math.PI);
				break;
			case 4:
				// vertical flip
				ctx.translate(0, height);
				ctx.scale(1, -1);
				break;
			case 5:
				// vertical flip + 90 rotate right
				ctx.rotate(0.5 * Math.PI);
				ctx.scale(1, -1);
				break;
			case 6:
				// 90 rotate right
				ctx.rotate(0.5 * Math.PI);
				ctx.translate(0, -height);
				break;
			case 7:
				// horizontal flip + 90 rotate right
				ctx.rotate(0.5 * Math.PI);
				ctx.translate(width, -height);
				ctx.scale(-1, 1);
				break;
			case 8:
				// 90 rotate left
				ctx.rotate(-0.5 * Math.PI);
				ctx.translate(-width, 0);
				break;
			default:
				break;
		}
	}

	function dataURLtoArrayBuffer(data) {
		var byteString = atob(data.split(',')[1]);
		var ab = new ArrayBuffer(byteString.length);
		var ia = new Uint8Array(ab);
		for (var i = 0; i < byteString.length; i++) {
			ia[i] = byteString.charCodeAt(i);
		}
		return ab
	}

	function dataURLtoBlob(data) {
		var mimeString = data.split(',')[0].split(':')[1].split(';')[0];
		var ab = dataURLtoArrayBuffer(data);
		var bb = (window.BlobBuilder || window.WebKitBlobBuilder || window.MozBlobBuilder);
		if (bb) {
			bb = new (window.BlobBuilder || window.WebKitBlobBuilder || window.MozBlobBuilder)();
			bb.append(ab);
			return bb.getBlob(mimeString);
		} else {
			bb = new Blob([ab], {
				'type': (mimeString)
			});
			return bb;
		}
	}

	// 拡張子からmimeTypeを取得
	function getImageType(fileName) {
		var ext = null;
		if (!fileName) return null;
		var fileTypes = fileName.split(".");
		var len = fileTypes.length;
		if (len === 0) return null;
		ext = fileTypes[len - 1];
		ext = ext.toLowerCase();
		switch (ext) {
			case "jpg":
			case "jpeg":
				return "image/jpeg";
			case "gif":
				return "image/gif";
			case "png":
				return "image/png";
			default:
				return null;
		}
	}

	// ---------------------------------------------------

	/**
	 * bind
	 */
	$.fn.inputFileResizer = function(options) {
		return new $.inputFileResizer(this, options);
	};

})(jQuery);
