<?php

namespace Uehi\Larapack\Models;

use ErrorException;
use Illuminate\Database\Eloquent\Relations\Relation;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;
use InvalidArgumentException;
use ReflectionClass;
use ReflectionMethod;

trait LarapackModel
{
    use AdminEditors;

    /**
     * boot
     */
    public static function bootLarapackModel()
    {
    }

    /**
     * 検索クエリを生成
     * @param array $params
     *      order: 配列指定
     *          ['updated_at' => 'DESC', 'id' => 'ASC']
     * @param bool $admin
     */
    public static function createSearchQuery($params = [], $admin = false)
    {
        $query = static::query();

        // public
        if (static::hasColumn('public')) {
            if ($admin) {
                if (!empty($params['public'])) {
                    $query->where('public', $params['public']);
                }
            } else {
                $query->where('public', 1);
            }
        }

        // keywords
        if (!empty($params['keywords'])) {
            static::createKeywordsQuery($query, $params['keywords'], false, $admin);
        }

        // order
        if (!empty($params['sort']) && !empty($params['sort_dir'])) {
            // sort, sort_dirある場合$params['order']より優先
            if (!static::hasColumn($params['sort']) || !in_array(strtolower($params['sort_dir']), ['desc', 'asc'])) {
                throw new InvalidArgumentException();
            }
            $query->orderBy($params['sort'], $params['sort_dir']);
        } else {
            $order = null;
            if (!empty($params['order'])) {
                $order = $params['order'];
            } elseif (!empty(static::defaultOrder())) {
                $order = static::defaultOrder();
            }

            if (!empty($order)) {
                if (is_array($order)) {
                    foreach ($order as $field => $sort) {
                        $query->orderBy($field, $sort);
                    }
                } else {
                    $query->orderByRaw($order);
                }
            }
        }

        return $query;
    }

    /**
     * defaultのorderを指定
     * overrideして使う
     * @return null
     */
    public static function defaultOrder()
    {
        return null;
    }

    /*
    |--------------------------------------------------------------------------
    | キーワード検索関連
    |--------------------------------------------------------------------------
    */

    public static function createKeywordsQuery($query, $paramKeywords, $or = false, $admin = false)
    {
        // スペース区切りのキーワード検索
        $keyword = mb_convert_kana($paramKeywords, 's');
        $keywords = preg_split('/[\s]+/', $keyword, -1, PREG_SPLIT_NO_EMPTY);
        $targetFields = static::getKeywordsTargets();
        if (empty($targetFields[0]) && empty($targetFields[1])) {
            return $query;
        }

        // 全角半角無関係に検索できるようにcollation指定
        $collation = '';
        $dbConfig = DB::connection()->getConfig();
        // 以下2種類のみ対応
        if (in_array($dbConfig['collation'], ['utf8mb4_general_ci', 'utf8_general_ci'])) {
            $collation = 'COLLATE ' . str_replace('general', 'unicode', $dbConfig['collation']);
        }

        $conditions = [];
        foreach ($keywords as $kwd) {
            if (empty($kwd)) continue;
            $oneKwdCond = [];
            if (!empty($targetFields[0])) {
                foreach ($targetFields[0] as $field) {
                    $oneKwdCond["({$field}) {$collation} LIKE ?"] = ['%' . $kwd . '%'];
                }
            }
            if ($admin && !empty($targetFields[1])) {
                foreach ($targetFields[1] as $field) {
                    $oneKwdCond["({$field}) {$collation} LIKE ?"] = ['%' . $kwd . '%'];
                }
            }
            $conditions[] = $oneKwdCond;
        }

        if (!$or) {
            // and: キーワードを全て持っているレコードのみ抽出
            foreach ($conditions as $cond) {
                $query->where(function($q) use ($cond) {
                    foreach ($cond as $sql => $val) {
                        $q->orWhereRaw($sql, $val);
                    }
                });
            }
        } else {
            // or: キーワードのいずれかを持っているレコードのみ
            foreach ($conditions as $cond) {
                foreach ($cond as $sql => $val) {
                    $query->orWhereRaw($sql, $val);
                }
            }
        }

        return $query;
    }

    /**
     * キーワード検索の対象となるフィールドを設定
     * overrideして使う
     * @return array [['通常時kwd1', '通常時kwd2'...], ['admin時kwd1', 'admin時kwd2'...]]
     *      default) [['title', 'text'], ['note']]
     */
    protected static function getKeywordsTargets()
    {
        $regularTargets = [];
        if (static::hasColumn('title')) {
            $regularTargets[] = 'title';
        }
        if (static::hasColumn('text')) {
            $regularTargets[] = 'text';
        }

        $adminTargets = [];
        if (static::hasColumn('note')) {
            $regularTargets[] = 'note';
        }
        return [$regularTargets, $adminTargets];
    }

    /*
    |--------------------------------------------------------------------------
    | list取得関連
    |--------------------------------------------------------------------------
    */

    /**
     * 一覧を取得
     * @param mixed $list : 文字列渡すとそのcolumnが配列のvalueなる
     *      true渡すと'title'にする
     * @param bool $admin
     * @return array
     */
    public static function getPublic($list = false, $admin = false)
    {
        if ($list === true) {
            $list = 'title';
        }
        $query = static::createSearchQuery([], $admin);
        $data = $query->get();
        $ret = [];
        foreach ($data as $val) {
            $ret[$val->id] = $list ? $val->{$list} : $val;
        }
        return $ret;
    }

    /*
    |--------------------------------------------------------------------------
    | 共通method
    |--------------------------------------------------------------------------
    */

    /**
     * table名取得
     * @return mixed
     */
    public static function getTableName()
    {
        return ((new self)->getTable());
    }

    /**
     * カラム名が存在するかを返す
     * @param $columnName
     * @return mixed
     */
    public static function hasColumn($columnName)
    {
        return Schema::hasColumn(static::getTableName(), $columnName);
    }

    /**
     * カラム名を取得
     * @return mixed
     */
    public static function getColumns()
    {
        return Schema::getColumnListing(static::getTableName());
    }

    /**
     * 全リレーションを取得
     * @return array
     */
    public function getAllRelations() {

        $model = new static;

        $relations = [];

        foreach((new ReflectionClass($model))->getMethods(ReflectionMethod::IS_PUBLIC) as $method) {
            if ($method->class != get_class($model) ||
                !empty($method->getParameters()) ||
                $method->getName() == __FUNCTION__) {
                continue;
            }

            try {
                $return = $method->invoke($model);

                if ($return instanceof Relation) {
                    $relations[$method->getName()] = [
                        'type' => (new ReflectionClass($return))->getShortName(),
                        'model' => (new ReflectionClass($return->getRelated()))->getName()
                    ];
                }
            } catch(ErrorException $e) {}
        }

        return $relations;
    }

}