<?php

namespace Uehi\Larapack\Models;

use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\File;

/**
 * ファイルアップロード処理を含む
 * Trait FileUploads
 * @package Uehi\Larapack\Models
 */
trait FileUploads
{

    /**
     * fillableにcolumn追加したかどうか
     * @var bool
     */
    private $addedUploadColumns = false;

    /*
    |--------------------------------------------------------------------------
    | model上書き用
    |--------------------------------------------------------------------------
    */

    /**
     * アップロード対象のcolumnをget
     * @return array
     */
    protected function getUploadColumns()
    {
        return [];
    }

    /*
    |--------------------------------------------------------------------------
    | アップロードパラメータ保持用のaccessor
    |--------------------------------------------------------------------------
    */

    /**
     * アップロードパラメータ保持用のaccessor名
     */
    private function getUploadSupportColumns()
    {
        $inputSuffix = Config::get('const.uploads.input_suffix');
        $deleteSuffix = Config::get('const.uploads.delete_suffix');
        $columns = [];
        foreach ($this->getUploadColumns() as $column) {
            $columns[] = "{$column}{$inputSuffix}";
            $columns[] = "{$column}{$deleteSuffix}";
        }
        return $columns;
    }

    /**
     * override アップロードパラメータを保持するためappends拡張
     * @param array $appends
     * @return $this
     */
//    public function setAppends(array $appends)
//    {
//        $supportColumns = $this->getUploadSupportColumns();
//        if (!empty($supportColumns)) {
//            $this->appends = array_unique(array_merge($this->appends, $supportColumns));
//        }
//
//        foreach ($supportColumns as $column) {
//            $functionName ='set' . studly_case($column) . 'Attribute';
//            $this->{$functionName} = function($value) use ($column) {
//                $this->attributes[$column] = $value;
//            };
//        }
//
//        return $this;
//    }

    protected $uploadInfo = [];
    protected function setUploadInfo($key, $value)
    {
        $this->uploadInfo[$key] = $value;
    }
    protected function getUploadInfo($key)
    {
        return !empty($this->uploadInfo[$key]) ? $this->uploadInfo[$key] : null;
    }

    /**
     * override アップロードパラメータを保持するためattributes拡張
     * @param array $attributes
     * @return $this
     */
    public function fill(array $attributes)
    {
        parent::fill($attributes);

//        foreach ($this->getUploadSupportColumns() as $column) {
//            if (!empty($attributes[$column])) {
//                $this->setAttribute($column, $attributes[$column]);
//            }
//        }

        // ファイルアップに必要な情報をオブジェクト内に格納しておく
        foreach ($this->getUploadSupportColumns() as $column) {
            if (!empty($attributes[$column])) {
                $this->setUploadInfo($column, $attributes[$column]);
            }
        }


        return $this;
    }

    /*
    |--------------------------------------------------------------------------
    | boot
    |--------------------------------------------------------------------------
    */

    /**
     * boot
     */
    public static function bootFileUploads()
    {
        $self = new self;
        $uploadColumns = $self->getUploadColumns();
        $inputSuffix = Config::get('const.uploads.input_suffix');
        $deleteSuffix = Config::get('const.uploads.delete_suffix');

        // ファイル名が渡ってきたら本来の保存領域に入れる
        static::saved(function ($record) use ($uploadColumns, $inputSuffix, $deleteSuffix) {
            if (!empty($uploadColumns)) {
                $isUpdate = false;
                foreach ($uploadColumns as $column) {
                    // requestから保存ファイル名、削除フラグある場合取得
                    $inputColumn = "{$column}{$inputSuffix}";
                    $deleteColumn = "{$column}{$deleteSuffix}";
                    $tmpName = !empty($record->getUploadInfo($inputColumn)) ? $record->getUploadInfo($inputColumn) : null;
                    $deleteFlag = !empty($record->getUploadInfo($deleteColumn));

                    // 存在する場合
                    if (!empty($tmpName)) {
                        // 新しいファイル(tmp dirが含まれている時tmpだと判断)ある場合tmpから移動
                        if (strpos($tmpName, Config::get('const.uploads.tmp_dir') . '/') !== false) {
                            $tmpPath = public_path($tmpName);
                            if (is_file($tmpPath)) {
                                $baseDir = public_path('/');
                                $saveDir = 'storage/' . $record->getTable() . '/' . floor($record->id / 1000) . '/';
                                if (!file_exists($baseDir . $saveDir)) {
                                    File::makeDirectory($baseDir . $saveDir, 0707, true);
                                }

                                $newName = $saveDir . basename($tmpPath);
                                if (rename($tmpPath, $baseDir . $newName)) {
                                    chmod($baseDir . $newName, 0707);
                                }

                                // modelにset
                                $record->{$column} = $newName;
                                $isUpdate = true;
                            }
                        }
                    }

                    // deleteフラグがある場合null set
                    if ($deleteFlag && !empty($record->{$column})) {
                        $record->{$column} = null;
                        $isUpdate = true;
                    }

                    // 指定fieldがdirtyかどうか
                    // dirtyなら以前のファイルは削除
                    if ($record->isDirty($column)) {
                        $original = $record->getOriginal($column);
                        $path = public_path($original);
                        if (!empty($original) && $original !== $record->{$column} && is_file($path)) {
                            unlink($path);
                        }
                    }

                }

                // 変更あればsave
                if ($isUpdate) {
                    $record->save();
                }
            }
            return true;
        });

    }

    /*
    |--------------------------------------------------------------------------
    | ファイル格納関連
    |--------------------------------------------------------------------------
    */

    /**
     * form値をtmpディレクトリに移動
     * FIXME: publicディレクトリ固定なので悩ましい
     * @param $column
     * @return bool
     */
    public function uploadToTmp($column)
    {
        $request = app('request');
        $params = $request->all();

        // hasManyの場合, $columnはchilds[1][image]のような形で来るので本来のcolumn名を抽出
        if (preg_match('/^(.+)\[([0-9]+)\]\[(.+)\]$/', $column, $m)) {
            $hasMany = true;
            $childProp = $m[1];
            $childNum = $m[2];
            $childColumn = $m[3];
            $value = !empty($params[$childProp][$childNum][$childColumn]) ? $params[$childProp][$childNum][$childColumn] : null;
        } else {
            $hasMany = false;
            $value = !empty($params[$column]) ? $params[$column] : null;
        }

        if (!empty($value) &&
            (
                (!$hasMany && in_array($column, $this->getUploadColumns()))
                ||
                ($hasMany && in_array($childColumn, $this->getUploadColumns()))
            )
        ) {
            // 移動
            $path = 'public/' . Config::get('const.uploads.tmp_dir');
            if ($hasMany) {
                $filename = $request->file($childProp)[$childNum][$childColumn]->store($path);
            } else {
                $filename = $request->file($column)->store($path);
            }
            // public/tmp/~~.png => storage/tmp/~~.png
            $ret = preg_replace('/^public\//', 'storage/', $filename);
            return $ret;
        }
        return false;
    }

    /*
    |--------------------------------------------------------------------------
    | 共通処理
    |--------------------------------------------------------------------------
    */


}