<?php

namespace Uehi\Larapack\Models;

use Illuminate\Database\Eloquent\SoftDeletingScope;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Route;
use Illuminate\Support\Facades\Schema;

/**
 * 管理画面で対象のカラムが存在した場合データの登録者ID、更新者ID、削除者IDを自動的に書き込む
 * Trait EditAdminIds
 * @package Uehi\Larapack\Models
 */
trait AdminEditors
{

    /**
     * boot
     */
    public static function bootAdminEditors()
    {
        $self = new self;
        $tableName = $self->getTable();

        $currentRoute = Route::getCurrentRoute();
        $prefix = !empty($currentRoute) ? $currentRoute->getPrefix() : null;
        $user = Auth::guard('admin')->user();
        if ($prefix === 'admin' && !empty($user)) {
            // カラム名とそのカラム名を持っているか
            $createdIdColumn = $self->getCreatedIdColumn();
            $hasCreatedColumn = Schema::hasColumn($tableName, $createdIdColumn);
            $updatedIdColumn = $self->getUpdatedIdColumn();
            $hasUpdatedColumn = Schema::hasColumn($tableName, $updatedIdColumn);
            $deletedIdColumn = $self->getDeletedIdColumn();
            $hasDeletedColumn = Schema::hasColumn($tableName, $deletedIdColumn);
            // softDeletesを利用しているか
            $useSoftDeletes = self::hasGlobalScope(SoftDeletingScope::class);

            // created_idカラムある場合管理者ID埋める
            if ($hasCreatedColumn) {
                static::creating(function ($record) use ($user, $createdIdColumn, $updatedIdColumn, $hasUpdatedColumn) {
                    $record->{$createdIdColumn} = $user->id;
                    // updatedColumnある場合こちらも一緒に
                    if ($hasUpdatedColumn) {
                        $record->{$updatedIdColumn} = $user->id;
                    }
                    return true;
                });
            }

            // updated_idカラムある場合管理者ID埋める

            if ($hasUpdatedColumn) {
                static::updating(function ($record) use ($user, $updatedIdColumn) {
                    $record->{$updatedIdColumn} = $user->id;
                    return true;
                });
            }

            // deleted_idカラムあり、softDeletes利用の場合管理者ID埋める

            if ($hasDeletedColumn && $useSoftDeletes) {
                static::deleting(function ($record) use ($user, $deletedIdColumn) {
                    $record->{$deletedIdColumn} = $user->id;
                    $record->save(); // ここでsaveしないと保存されない
                    return true;
                });
            }

        }

    }

    /**
     * created_idのカラム名を返す
     * 変更する場合、modelにconst CREATED_IDを設定する
     * @return string
     */
    public function getCreatedIdColumn()
    {
        return defined('static::CREATED_ID') ? static::CREATED_ID : 'created_id';
    }

    /**
     * updated_idのカラム名を返す
     * 変更する場合、modelにconst UPDATED_IDを設定する
     * @return string
     */
    public function getUpdatedIdColumn()
    {
        return defined('static::UPDATED_ID') ? static::UPDATED_ID : 'updated_id';
    }

    /**
     * deleted_idのカラム名を返す
     * 変更する場合、modelにconst DELETED_IDを設定する
     * @return string
     */
    public function getDeletedIdColumn()
    {
        return defined('static::DELETED_ID') ? static::DELETED_ID : 'deleted_id';
    }

}