<?php

namespace Uehi\Larapack\Commands;

use Illuminate\Console\Command;

class ViewMakeCommand extends Command
{
    /**
     * The console command signature.
     *   full: 全てのviewを外出しする
     *   single: 設定画面などフォーム1画面の機能を出したい場合
     * @var string
     */
    protected $signature = 'larapackmake:view {name : The name of model class.} {--full : Whether creating all view.} {--single : in case form page only.}';

    /**
     * 説明
     * @var string
     */
    protected $description = 'Create a new view files for Larapack';

    /**
     * Execute
     * @return void
     */
    public function fire()
    {
        $name = str_plural(snake_case(trim($this->input->getArgument('name'))));
        $isFull = $this->option('full');
        $isSingle = $this->option('single');
        $this->createFiles($name, $isFull, $isSingle);
    }

    /**
     * 必要なファイルを作成
     * @param $dirName
     * @param $isFull
     * @param $isSingle
     */
    public function createFiles($dirName, $isFull, $isSingle)
    {
        $baseDir = resource_path("views/admin/{$dirName}/");
        $files = [
            ['_form', $baseDir . '_form.blade.php', $this->getViewContent('partials/admin/_form')],
        ];

        if (!$isSingle) {
            $files[] = ['_list', $baseDir . '_list.blade.php', $this->getViewContent('partials/admin/_list')];
            $files[] = ['_search', $baseDir . '_search.blade.php', $this->getViewContent('partials/admin/_search')];
        }

        if ($isFull) {
            $files[] = ['create', $baseDir . 'create.blade.php', $this->getViewContent('layouts/create')];
            if (!$isSingle) {
                $files[] = ['index', $baseDir . 'index.blade.php', $this->getViewContent('layouts/index')];
            }
        }

        foreach ($files as $file) {
            $this->putFile($file[0], $file[1], $file[2]);
        }
    }

    /**
     * ファイル作成
     * @param $logText: 標準出力に出す名前
     * @param $path: ファイルパス
     * @param $content: ファイルに記述するtext
     */
    protected function putFile($logText, $path, $content)
    {
        // 既に存在する場合なにもしない
        if (file_exists($path)) {
            $this->line("<info>{$logText} file has already exists</info>");
            return;
        }

        // ディレクトリなければ作成
        $this->makeDir(dirname($path));

        // ファイル作成
        $this->laravel['files']->put(
            $path,
            $content
        );
        $this->line("<info>{$logText} file was created:</info> " . $path);
    }

    /**
     * viewファイルの中身を取得
     * @param $name
     * @return mixed
     */
    protected function getViewContent($name)
    {
        return $this->laravel['files']->get(dirname(dirname(dirname(__FILE__))) . "/resources/views/{$name}.blade.php");
    }

    /**
     * ディレクトリ作成
     * あれば何もしない
     * @param $path
     */
    protected function makeDir($path)
    {
        if (!file_exists($path)) {
            // 再帰的に作成
            $this->laravel['files']->makeDirectory($path, 0755, true);
        }
    }

}
