<?php

namespace Uehi\Larapack;


use BadMethodCallException;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\HtmlString;
use Illuminate\View\View;
use Symfony\Component\Debug\Exception\ClassNotFoundException;
use Symfony\Component\Debug\Exception\FatalErrorException;

class Form
{

    /**
     * Laravelcollective::Form
     * @var \Illuminate\Foundation\Application|mixed|null
     */
    private $form = null;

    /**
     * isConfirm
     * @var null
     */
    private $isConfirm = null;

    /**
     * Larapack constructor.
     */
    public function __construct()
    {
        $this->form = app('form');
    }

    /*
    |--------------------------------------------------------------------------
    | confirm判別
    |--------------------------------------------------------------------------
    */

    /**
     * 確認画面,詳細画面等の表示のみ行うフラグ
     */
    public function isConfirm()
    {
        // 明示的にsetされている場合
        if (is_bool($this->isConfirm)) {
            return $this->isConfirm;
        }
        // 通常はactionで判定
        $actionName = app('view')->shared('actionName');
        return !in_array($actionName, ['create', 'edit']);
    }

    /**
     * 明示的にconfirm状態にする
     */
    public function setIsConfirm($value)
    {
        $this->isConfirm = $value;
    }

    /*
    |--------------------------------------------------------------------------
    | form要素 laravelcollective::formのラッパー含む
    |--------------------------------------------------------------------------
    */

    /**
     * @param array $options
     * @return string
     */
    public function open(array $options = [])
    {
        if ($this->isConfirm()) {
            return '';
        }
        return $this->form->open($options);
    }

    /**
     * @param $model
     * @param array $options
     * @return string
     */
    public function model($model, array $options = [])
    {
        if ($this->isConfirm()) {
            return '';
        }
        return $this->form->model($model, $options);
    }

    /**
     * @return mixed
     */
    public function close()
    {
        if ($this->isConfirm()) {
            return '';
        }
        return $this->form->close();
    }

    /**
     * @param $name
     * @param null $value
     * @param array $options
     * @return mixed
     */
    public function hidden($name, $value = null, $options = [])
    {
        if ($this->isConfirm()) {
            return '';
        }
        return $this->form->hidden($name, $value, $options);
    }

    /**
     * @param $name
     * @param null $value
     * @param array $options
     * @return mixed
     */
    public function text($name, $value = null, $options = [])
    {
        if ($this->isConfirm()) {
            return $this->getDisplayInput($name);
        }
        return $this->form->text($name, $value, $options);
    }

    /**
     * @param $name
     * @param array $options
     * @return mixed
     */
    public function password($name, $options = [])
    {
        if ($this->isConfirm()) {
            return '********(表示しません)';
        }
        return $this->form->password($name, $options);
    }

    /**
     * email
     * @param $name
     * @param null $value
     * @param array $options
     * @return mixed
     */
    public function email($name, $value = null, $options = [])
    {
        if ($this->isConfirm()) {
            return $this->getDisplayInput($name);
        }
        return $this->form->email($name, $value, $options);
    }

    /**
     * tel
     * @param $name
     * @param null $value
     * @param array $options
     * @return mixed
     */
    public function tel($name, $value = null, $options = [])
    {
        if ($this->isConfirm()) {
            return $this->getDisplayInput($name);
        }
        return $this->form->tel($name, $value, $options);
    }

    /**
     * number
     * @param $name
     * @param null $value
     * @param array $options
     * @return mixed
     */
    public function number($name, $value = null, $options = [])
    {
        if ($this->isConfirm()) {
            return $this->getDisplayInput($name);
        }
        return $this->form->number($name, $value, $options);
    }

    /**
     * date
     * @param $name
     * @param null $value
     * @param array $options
     * @return mixed
     */
    public function date($name, $value = null, $options = [])
    {
        if ($this->isConfirm()) {
            return $this->getDisplayInput($name);
        }
        return $this->form->date($name, $value, $options);
    }

    /**
     * datetime
     * @param $name
     * @param null $value
     * @param array $options
     * @return mixed
     */
    public function datetime($name, $value = null, $options = [])
    {
        if ($this->isConfirm()) {
            return $this->getDisplayInput($name);
        }
        return $this->form->datetime($name, $value, $options);
    }

    /**
     * datetimeLocal
     * @param $name
     * @param null $value
     * @param array $options
     * @return mixed
     */
    public function datetimeLocal($name, $value = null, $options = [])
    {
        if ($this->isConfirm()) {
            return $this->getDisplayInput($name);
        }
        return $this->form->datetimeLocal($name, $value, $options);
    }

    /**
     * time
     * @param $name
     * @param null $value
     * @param array $options
     * @return mixed
     */
    public function time($name, $value = null, $options = [])
    {
        if ($this->isConfirm()) {
            return $this->getDisplayInput($name);
        }
        return $this->form->time($name, $value, $options);
    }

    /**
     * url
     * @param $name
     * @param null $value
     * @param array $options
     * @return mixed
     */
    public function url($name, $value = null, $options = [])
    {
        if ($this->isConfirm()) {
            return $this->getDisplayInput($name);
        }
        return $this->form->url($name, $value, $options);
    }

    /**
     * Create a textarea input field.
     *
     * @param  string $name
     * @param  string $value
     * @param  array  $options
     * @param array $ck
     *      'src' => 'ckeditor.jsの場所',
     *      'class' => 'class名(なければckeditor)',
     * @return string
     */
    public function textarea($name, $value = null, $options = [], array $ck = [])
    {
        if ($this->isConfirm()) {
            $value = $this->getDisplayInput($name);
            if (empty($value)) {
                return '';
            } elseif (!empty($ck)) {
                return new HtmlString($value);
            } else {
                return new HtmlString(nl2br(e($value)));
            }
        }

        if (!empty($ck)) {
            app('larapack')->appendJs($ck['src']);
            // class
            $ckClass = !empty($ck['class']) ? $ck['class'] : 'ckeditor';
            $options['class'] = !empty($options['class']) ? "{$options['class']} {$ckClass}" : $ckClass;
        }
        return $this->form->textarea($name, $value, $options);
    }

    /**
     * @param $name
     * @param array $list
     * @param null $selected
     * @param array $selectAttributes
     * @param array $optionsAttributes
     * @return string
     */
    public function select($name, $list = [], $selected = null, array $selectAttributes = [], array $optionsAttributes = [])
    {
        if ($this->isConfirm()) {
            return $this->getDisplaySelect($name, $list);
        }
        return $this->form->select($name, $list, $selected, $selectAttributes, $optionsAttributes);
    }

    /**
     * プルダウン日付
     * @param $name
     * @param array $options
     *      'empty' = 空を用意するか
     *      'minYear' = minimum year
     *      'maxYear' = max year
     *      'orderYear' = 'asc' or 'desc' 年を昇順降順どちらにするか
     *      'format' = '<div class="form-inline">{{year}} 年 {{month}} 月 {{day}} 日</div>'
     *      'start' = $minYear年を何月何日から選択可能にするか 例) [3]: 3/1から, [6, 15] 6/15から
     *      'end' = $maxYearの年を何月何日まで選択可能にするか 例) [3]: 3/31まで, [6, 15] 6/15まで
     * @param null $selected "yyyy-mm-dd"で指定
     * @param array $selectAttributes
     *      ['year' => [], 'month' => [], 'day' => []] で指定
     * @param array $optionsAttributes
     *      ['year' => [], 'month' => [], 'day' => []] で指定
     * @return string
     */
    public function selectDate($name, array $options = [], $selected = null, array $selectAttributes = [], array $optionsAttributes = [])
    {
        // year, month, dayのname
        $nameY = "{$name}_year";
        $nameM = "{$name}_month";
        $nameD = "{$name}_day";

        // options default
        $empty = current(Config::get('values.selectEmpty'));
        $defaults = [
            'empty' => $empty,
            'minYear' => 1980,
            'maxYear' => date('Y'),
            'orderYear' => 'asc',
            'format' => '<div class="form-inline">{{year}} 年 {{month}} 月 {{day}} 日</div>'
        ];
        $options = array_merge($defaults, $options);
        // selectAttributes default
        $defaults = [
            'year' => ['class' => 'form-control'],
            'month' => ['class' => 'form-control'],
            'day' => ['class' => 'form-control']
        ];
        $selectAttributes = array_merge($defaults, $selectAttributes);
        // optionsAttributes default
        $defaults = ['year' => [], 'month' => [], 'day' => []];
        $optionsAttributes = array_merge($defaults, $optionsAttributes);

        // selectedの値
        $ymd = !empty($selected) ? $selected : $this->getDisplayInput($name);
        $y = $m = $d = null;
        $ymdArr = explode('-', $ymd);
        if (!empty($ymdArr[0])) {
            $y = $ymdArr[0];
        }
        if (!empty($ymdArr[1])) {
            $m = sprintf('%d', $ymdArr[1]);
        }
        if (!empty($ymdArr[2])) {
            $d = sprintf('%d', $ymdArr[2]);
        }

        if ($this->isConfirm()) {
            if (empty($y) || empty($m) || empty($d)) {
                return '';
            }
            $str = $this->replaceTemplate($options['format'], [
                'year' => $y,
                'month' => $m,
                'day' => $d,
            ]);
            return new HtmlString($str);
        }

        // selectAttributes
        $yearAttr = !empty($selectAttributes['year']) ? $selectAttributes['year'] : [];
        $monthAttr = !empty($selectAttributes['month']) ? $selectAttributes['month'] : [];
        $dayAttr = !empty($selectAttributes['day']) ? $selectAttributes['day'] : [];
        // optionsAttributes
        $optionsYearAttr = !empty($optionsAttributes['year']) ? $optionsAttributes['year'] : [];
        $optionsMonthAttr = !empty($optionsAttributes['month']) ? $optionsAttributes['month'] : [];
        $optionsDayAttr = !empty($optionsAttributes['day']) ? $optionsAttributes['day'] : [];

        // y, m, dそれぞれのjs用classを決定する
        $uid = uniqid();
        $yearClass = "selectDateYear_{$uid}";
        $monthClass = "selectDateMonth_{$uid}";
        $dayClass = "selectDateDay_{$uid}";
        $hiddenClass = "selectDateYmd_{$uid}";
        $yearAttr['class'] = !empty($yearAttr['class']) ? "{$yearAttr['class']} {$yearClass}" : $yearClass;
        $monthAttr['class'] = !empty($monthAttr['class']) ? "{$monthAttr['class']} {$monthClass}" : $monthClass;
        $dayAttr['class'] = !empty($dayAttr['class']) ? "{$dayAttr['class']} {$dayClass}" : $dayClass;

        // 最初の年の選択可能な年月ある場合
        $startM = $startD = 0;
        if (!empty($options['start'])) {
            if (count($options['start']) === 2) {
                $startM = $options['start'][0];
                $startD = $options['start'][1];
            } else {
                $startM = $options['start'][0];
            }
        }
        // 最終年の選択可能な年月ある場合
        $endM = $endD = 0;
        if (!empty($options['end'])) {
            if (count($options['end']) === 2) {
                $endM = $options['end'][0];
                $endD = $options['end'][1];
            } else {
                $endM = $options['end'][0];
            }
        }

        // select list
        $yearList = app('larapack')->util()->getSelectNum($options['minYear'], $options['maxYear']);
        if (strtolower($options['orderYear']) !== 'asc') {
            $yearList = array_reverse($yearList);
        }
        if (!empty($options['empty'])) {
            $yearList = ['' => $options['empty']] + $yearList;
        }
        $monthList  = app('larapack')->util()->getSelectNum(1, 12);
        $dayList  = app('larapack')->util()->getSelectNum(1, 31);
        // year/month/day select
        $yearForm = $this->form->select($nameY, $yearList, $y, $yearAttr, $optionsYearAttr);
        $monthForm = $this->form->select($nameM, $monthList, $m, $monthAttr, $optionsMonthAttr);
        $dayForm = $this->form->select($nameM, $dayList, $d, $dayAttr, $optionsDayAttr);
        // ymd結合したhidden
        $hiddenForm = $this->form->hidden($name, $ymd, ['class' => $hiddenClass]);

        $ret = $this->replaceTemplate($options['format'], [
            'year' => $yearForm,
            'month' => $monthForm,
            'day' => $dayForm,
        ]);
        $ret .= $hiddenForm;

        // js
        $js = '<script type="text/javascript">';
        $js .= "
            var isEmptyName = " . (!empty($options['empty']) ? "'{$options['empty']}'" : 'false') . ";
            jQuery(function($) {
                // 選択可能な最初の年月
                var startY = " . $options['minYear'] . ";
                var startM = " . $startM . ";
                var startD = " . $startD . ";
                // 選択可能な最終年月
                var endY = " . $options['maxYear'] . ";
                var endM = " . $endM . ";
                var endD = " . $endD . ";

                // hiddenのvalueをselectの値に変更する
                var changeYmdHidden = function(){
                    var y = $('.{$yearClass}').val();
                    var m = $('.{$monthClass}').val();
                    var d = $('.{$dayClass}').val();

                    // 値が全て揃っていなければ空を渡す
                    ymd = '';
                    if (!y || !m || !d) {
                        // 何もしない
                    } else {
                        m = ('0' + m).slice(-2);
                        d = ('0' + d).slice(-2);
                        ymd = y + '-' + m + '-' + d;
                    }

                    $('.{$hiddenClass}').attr('value', ymd);
                    return true;
                }

                // y,m,dのselectをdisableにする
                var disableYmdSelect = function() {
                    $('select.{$yearClass}').attr('disabled', 'disabled');
                    $('select.{$monthClass}').attr('disabled', 'disabled');
                    $('select.{$dayClass}').attr('disabled', 'disabled');
                    return true;
                }

                // 属するformのsubmit時にy,m,dのselectは送信しないようdisableにする
                var form = $('select.{$yearClass}').closest('form');
                form.submit(function(){
                    return disableYmdSelect();
                });

                // 年変更時
                $('select.{$yearClass}').change(function(){
                    startEndMDCheck();
                    leapYearCheck();
                    changeYmdHidden();
                });
                // 月変更時
                $('select.{$monthClass}').change(function(){
                    startEndMDCheck();
                    leapYearCheck();
                    changeYmdHidden();
                });
                // 初期値のday
                $('select.{$dayClass}').val('{$d}');
                $('select.{$dayClass}').change(function(){
                    changeYmdHidden();
                });

                // 初回実行
                leapYearCheck();
                startEndMDCheck();

                // 最初、最終年月チェック
                function startEndMDCheck(){
                    var y = $('.{$yearClass}').val();
                    var currentMonthSelect = $('.{$monthClass}').val();
                    var first = 1;
                    var last = 12;
                    $('.{$monthClass}').empty();
                    if(isEmptyName) $('.{$monthClass}').append('<option>' + isEmptyName + '</option>');
                    // 最初年選択時に最初表示月指定ある場合
                    if (parseInt(y) == startY && (0 < startM && startM <= 12)) {
                        first = startM;
                    }
                    // 最終年選択時に最終表示月指定ある場合
                    if (parseInt(y) == endY && (0 < endM && endM <= 12)) {
                        last = endM;
                    }
                    for (var i = first; i <= last; i++) {
                        var selected = (parseInt(currentMonthSelect) == i) ? ' selected=\'selected\' ' : '';
                        $('.{$monthClass}').append('<option value=\'' + i + '\'' + selected + '>' + i + '</option>');
                    }
                }

                // うるう年判定して日付を表示
                function leapYearCheck(){
                    // dayプルダウン初期化
                    var currentDaySelect = $('.{$dayClass}').val();
                    $('.{$dayClass}').empty();
                    if(isEmptyName) $('.{$dayClass}').append('<option>' + isEmptyName + '</option>');

                    var y = $('.{$yearClass}').val();
                    var m = $('.{$monthClass}').val();
                    var first = 1;
                    var last;
                    if (m == 2 && (y % 400 == 0 || (y % 4 == 0 && y % 100 != 0))) {
                        last = 29;
                    } else {
                        last = new Array(31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31)[m - 1];
                    }

                    // 最初表示年月で最終選択日の指定がある場合その日が最後
                    if (parseInt(y) == startY && parseInt(m) == startM && (0 < startD && startD <= last)) {
                        first = startD;
                    }
                    // 最終表示年月で最終選択日の指定がある場合その日が最後
                    if (parseInt(y) == endY && parseInt(m) == endM && (0 < endD && endD <= last)) {
                        last = endD;
                    }

                    for (var i = first; i <= last; i++) {
                        selected = (parseInt(currentDaySelect) == i) ? ' selected=\'selected\' ' : '';
                        $('.{$dayClass}').append('<option value=\'' + i + '\'' + selected + '>' + i + '</option>');
                    }
                }
            });
        ";
        $js .= '</script>';
        app('larapack')->appendJsBlock($js);

        return new HtmlString($ret);
    }

    /**
     * @param $name
     * @param null $value
     * @param array $options
     *      dpOptions: '{a:b, c:d}'等のハッシュで指定するとjquery.datepickerのオプションとなる
     *      time: trueにするとdatetimeのpickerになる
     * @return mixed
     */
    public function textDate($name, $value = null, $options = [])
    {
        if ($this->isConfirm()) {
            $format = !empty($options['time']) ? 'Y年m月d日 H:i:s' : 'Y年m月d日';
            $value = $this->getDisplayInput($name);
            return !empty($value) ? date($format, strtotime($value)) : null;
        }

        // time flag
        $time = !empty($options['time']);
        if (isset($options['time'])) {
            unset($options['time']);
        }

        // input id
        if (empty($options['id'])) {
            $options['id'] = "{$name}_" . uniqid();
        }
        $selector = "#{$options['id']}";

        // css&javascript読み込み
        $js = '';
        app('larapack')->appendCss('//ajax.googleapis.com/ajax/libs/jqueryui/1.11.4/themes/smoothness/jquery-ui.css');
        app('larapack')->appendJs(url('/larapack/js/jquery.ui.datepicker-ja.js'));
        if ($time) {
            app('larapack')->appendCss(url('/larapack/js/jquery-ui-timepicker-addon.css'));
            app('larapack')->appendJs(url('/larapack/js/jquery-ui-timepicker-addon.js'));
            $jOpt = "{
                timeOnlyTitle: '時間',
                timeText: '時間',
                hourText: '時',
                minuteText: '分',
                secondText: '秒',
                closeText: '閉じる',
                currentText: '現在',
                showButtonPanel: true,
                dateFormat: 'yy-mm-dd',
                showSecond: true,
                controlType: 'select',
                timeFormat: 'HH:mm:ss'
            }";
            if (!empty($options['dpOption'])) {
                $jOpt = "jQuery.extend({$jOpt}, {$options['dpOption']})";
            }
            $js = "
                jQuery(function($){
                    $(document).on('focus', '{$selector}', function(){
                        var options = {$jOpt};
                        $(this).datetimepicker(options);
                    });
                });
            ";
        }else{
            $jOpt = "{
                closeText: '閉じる',
                currentText: '今日',
                showButtonPanel: true,
                dateFormat: 'yy-mm-dd'
            }";
            if (!empty($options['dpOption'])) {
                $jOpt = "$.extend({$jOpt}, {$options['dpOption']})";
            }
            $js = "
                jQuery(function($){
                    var options = {$jOpt};
                    $(document).on('focus', '{$selector}', function(){
                        $(this).datepicker(options);
                    });
                });
            ";
        }

        if (!empty($options['dpOption'])) {
            unset($options['dpOption']);
        }

        // js書き出し
        app('larapack')->appendJsBlock('<script type="text/javascript">' . $js . '</script>');
        return $this->form->text($name, $value, $options);
    }

    /**
     * @param $name
     * @param array $list
     * @param null $selected
     * @param array $options
     *      template: '<label>{{radio}}{{legend}}</label>' 各radioのformat
     * @return string
     */
    public function radio($name, $list = [], $selected = null, array $options = [])
    {
        if ($this->isConfirm()) {
            return $this->getDisplaySelect($name, $list);
        }

        $defaults = [
            'template' => '<label>{{radio}}{{legend}}</label>',
        ];
        $options = array_merge($defaults, $options);

        // class defaultでつける
        $options['class'] = !empty($options['class']) ? "{$options['class']} radio-inline" : 'radio-inline';
        // template
        $template = $options['template'];
        unset($options['template']);

        $ret = '';
        foreach ($list as $value => $legend) {
            $ret .= $this->replaceTemplate($template, [
                'radio' => $this->form->radio($name, $value, ($value == $selected), $options),
                'legend' => $legend,
            ]);
        }
        return new HtmlString($ret);
    }

    /**
     * @param $name
     * @param null $checked
     * @param array $options
     *      template: '<label>{{checkbox}}{{legend}}</label>'
     * @param array $legends
     *      ['入力画面のlabel', '確認画面チェックした場合のlabel', '確認画面チェックしない場合のlabel']
     * @return mixed
     */
    public function checkbox($name, $checked = null, $options = [], $legends = ['EDIT', 'YES', 'NO'])
    {
        if ($this->isConfirm()) {
            $value = $this->getDisplayInput($name);
            return !empty($value) ? $legends[1] : $legends[2];
        }

        $defaults = [
            'template' => '<label>{{checkbox}}{{legend}}</label>',
        ];
        $options = array_merge($defaults, $options);

        // class defaultでつける
        $options['class'] = !empty($options['class']) ? "{$options['class']} checkbox-inline" : 'checkbox-inline';
        // template
        $template = $options['template'];
        unset($options['template']);

        $ret = $this->form->hidden($name, 0); // チェック->未チェックに更新時0を送るため
        $ret .= $this->replaceTemplate($template, [
            'checkbox' => $this->form->checkbox($name, 1, $checked, $options),
            'legend' => $legends[0],
        ]);
        return new HtmlString($ret);
    }

    /**
     * 画像アップロード
     * @param $name
     * @param array $options
     *      model: モデル名を明示的に指定する場合(App\Models\Example)
     *      Request: validationするrequestを明示的に指定する場合(App\Http\Requests\ExampleRequest)
     * @param array $imageOptions
     * @return HtmlString
     */
    public function imageFile($name, array $options = [], array $imageOptions = [])
    {
        if ($this->isConfirm()) {
            if ($filename = $this->getDisplayInput($name)) {
                return app('html')->image("/{$filename}", '', $imageOptions);
            } else {
                return '';
            }
        }

        // inputのclass
        $uid = uniqid();
        $fileClass = "inputFile_{$uid}";
        $options['class'] = !empty($options['class']) ? "{$options['class']} {$fileClass}" : $fileClass;
        // 画像表示する部分のdivタグのclassを指定
        $divClass = "inputFileShow_{$uid}";

        // model求める(\が消えるのでreplace)
        if (!empty($options['model'])) {
            $model = str_replace('\\', '\\\\', $options['model']);
            unset($options['model']);
        } else {
            // default($recordから取得)
            $record = app('view')->shared('record');
            $model = str_replace('\\', '\\\\', get_class($record));
        }

        // request求める
        if (!empty($options['request'])) {
            $request = str_replace('\\', '\\\\', $options['request']);
            unset($options['request']);
        } else {
            // default(model名から取得)
            $arr = explode('\\\\', $model);
            $modelname = end($arr);
            $request = "App\\\\Http\\\\Requests\\\\{$modelname}Request";
        }

        // js
        app('larapack')->appendJs(url('/larapack/js/jquery.inputFileResizer.js'));
        $js = app('view')->make('larapack::uploader.image_js', [
            'model' => $model,
            'request' => $request,
            'column' => $name,
            'fileClass' => $fileClass,
            'divClass' => $divClass,
            'imageOptions' => $imageOptions,
        ]);

        // 初期値
        $errors = app('view')->shared('errors');
        if (count($errors) > 0) {
            // validationエラー時は前回の入力値から取る
            $filename = $this->getDisplayInput($name . app('config')->get('const.uploads.input_suffix'));
            $isDeleted = $this->getDisplayInput($name . app('config')->get('const.uploads.delete_suffix'));
        } else {
            // 初回表示時
            $filename = $this->getDisplayInput($name);
            $isDeleted = false;
        }
        // 初期表示
        $initHtml = app('view')->make('larapack::uploader.image', [
            'filename' => $filename,
            'column' => $name,
            'imageOptions' => $imageOptions,
            'isDeleted' => $isDeleted,
        ]);

        $ret = '';
        $ret .= $js; // inline
        $ret .= "<div class='{$divClass}'>{$initHtml}</div>";
        $ret .= $this->form->file($name, $options);
        return new HtmlString($ret);
    }

    /**
     * ファイル
     * @param $name
     * @param array $options
     * @return HtmlString
     */
    public function documentFile($name, array $options = [])
    {
        if ($this->isConfirm()) {
            if ($filename = $this->getDisplayInput($name)) {
                $html = "<a href='/{$filename}' target='_blank'>ファイル</a>";
                return new HtmlString($html);
            } else {
                return '';
            }
        }

        // inputのclass
        $uid = uniqid();
        $fileClass = "inputFile_{$uid}";
        $options['class'] = !empty($options['class']) ? "{$options['class']} {$fileClass}" : $fileClass;
        // 画像表示する部分のdivタグのclassを指定
        $divClass = "inputFileShow_{$uid}";

        // model求める(\が消えるのでreplace)
        if (!empty($options['model'])) {
            $model = str_replace('\\', '\\\\', $options['model']);
            unset($options['model']);
        } else {
            // default($recordから取得)
            $record = app('view')->shared('record');
            $model = str_replace('\\', '\\\\', get_class($record));
        }

        // request求める
        if (!empty($options['request'])) {
            $request = str_replace('\\', '\\\\', $options['request']);
            unset($options['request']);
        } else {
            // default(model名から取得)
            $arr = explode('\\\\', $model);
            $modelname = end($arr);
            $request = "App\\\\Http\\\\Requests\\\\{$modelname}Request";
        }

        // js
        $js = app('view')->make('larapack::uploader.document_js', [
            'model' => $model,
            'request' => $request,
            'column' => $name,
            'fileClass' => $fileClass,
            'divClass' => $divClass,
        ]);

        // 初期値
        $errors = app('view')->shared('errors');
        if (count($errors) > 0) {
            // validationエラー時は前回の入力値から取る
            $filename = $this->getDisplayInput($name . app('config')->get('const.uploads.input_suffix'));
            $isDeleted = $this->getDisplayInput($name . app('config')->get('const.uploads.delete_suffix'));
        } else {
            // 初回表示時
            $filename = $this->getDisplayInput($name);
            $isDeleted = false;
        }
        // 初期表示
        $initHtml = app('view')->make('larapack::uploader.document', [
            'filename' => $filename,
            'column' => $name,
            'isDeleted' => $isDeleted,
        ]);

        $ret = '';
        $ret .= $js; // inline
        $ret .= "<div class='{$divClass}'>{$initHtml}</div>";
        $ret .= $this->form->file($name, $options);
        return new HtmlString($ret);
    }

    /**
     * @param $latName
     * @param $lonName
     * @param $zoomName
     * @param array $options
     *      'size' mapのサイズ ['width' => 123, 'height' => 456]
     *      'default' 初期位置 ['lat' => 33.5903547, 'lon' => 130.4017155, 'zoom' => 13]
     * @param bool $addressSearch 地図検索機能を入れるか
     *      true : 地図検索のinput type textを生成
     *      ['field' => 'フィールドのid'] 該当のidのinputを利用した検索
     *      ['field' => ['フィールド1のid', 'フィールド2のid' .. ]] 該当のidのinput(複数)を文字列連結して検索
     *      ['field' => ['フィールド1のid', 'フィールド2のid' .. ], 'prefix' => 'prefix文字列'] prefix文字列 + input(複数)を文字列連結して検索
     * @return string
     */
    public function map($latName, $lonName, $zoomName, array $options = [], $addressSearch = false)
    {
        // 設定値($optionsで上書き可能)
        $size = ['width' => 555, 'height' => 350];
        $default = app('config')->get('const.map_position');
        if (!empty($options['size'])) {
            $size = $options['size'];
            unset($options['size']);
        }
        if (!empty($options['default'])) {
            $default = $options['default'];
            unset($options['default']);
        }

        // 値
        $valueLat = $this->getDisplayInput($latName);
        $valueLon = $this->getDisplayInput($lonName);
        $valueZoom = $this->getDisplayInput($zoomName);

        if ($this->isConfirm()) {
            if ($valueLat && $valueLon && $valueZoom) {
                $url = app('larapack')->util()->gmapStaticUrl($valueLat, $valueLon, $valueZoom, $size['width'], $size['height']);
                $ret = "<div><img src='{$url}'></div>";
                return new HtmlString($ret);
            }else{
                return '';
            }
        }

        $ret = "";

        // class指定に利用
        $uniqId = uniqid();
        $latId = "{$latName}_{$uniqId}";
        $lonId = "{$lonName}_{$uniqId}";
        $zoomId = "{$zoomName}_{$uniqId}";
        $mapDivId = "gmap_{$uniqId}";
        $searchBtnId = "gmapsearchbtn_{$uniqId}";

        // js読み込み
        app('larapack')->appendJs(app('larapack')->util()->gmapJsUrl());
        app('larapack')->appendJs(url('/larapack/js/gmap.v3.js'));
        $mainJs = "
            // google map --------------------------------------------------------------
            var lat = " . ($valueLat ? $valueLat : $default['lat']) . ";
            var lon = " . ($valueLon ? $valueLon : $default['lon']) . ";
            var zoom = " . ($valueZoom ? $valueZoom : $default['zoom']) . ";
            FormGm = new Gmap($('#{$mapDivId}').get(0), {centerLatLng: [lat, lon], zoom: zoom, width: $('#{$mapDivId}').css('width'), height: $('#{$mapDivId}').css('height')});
            FormGm.show();
            google.maps.event.addListener(FormGm.map, 'idle', function(){
                var centerObj = FormGm.map.getCenter();

                $('#{$latId}').val(centerObj.lat());
                $('#{$lonId}').val(centerObj.lng());
                $('#{$zoomId}').val(FormGm.map.getZoom());

                if(FormMarker) FormGm.removeOverlay(FormMarker);
                FormMarker = FormGm.setMarker([centerObj.lat(), centerObj.lng()]);
            });
        ";

        $searchJs = "";
        $searchBtn = "<a href='javascript:void(0);' id='{$searchBtnId}' class='btn btn-primary'>地図を検索</a>";
        if (!empty($addressSearch)) {
            // 他のテキストフォームを使う
            $searchStrJs = "";
            if (is_array($addressSearch) && !empty($addressSearch['field'])) {
                $prefixStr = (!empty($addressSearch['prefix'])) ? $addressSearch['prefix'] : '';
                if (!is_array($addressSearch['field'])) {
                    $searchStrJs = "
                        var sad = '{$prefixStr}' + $('#{$addressSearch['field']}').val();
                    ";
                } else {
                    // field指定が複数ある場合この入力値を連結
                    $searchStrJs = "
                        var sad = '{$prefixStr}';
                        var target = " . json_encode($addressSearch['field']) . ";
                        for (var i = 0; i < target.length; i++) {
                            var inputObj = $('[name=\'' + target[i] + '\']');
                            if(inputObj[0].nodeName.toLowerCase() === 'input') {
                                // input
                                sad = sad + inputObj.val();
                            } else {
                                // select
                                sad = sad + inputObj.children(':selected').text();
                            }
                        }
                    ";
                }
                $ret .= '<div class="mb10">';
                $ret .= $searchBtn;
                $ret .= '</div>';
            } else {
                // テキストフォームを用意
                $searchId = "form_map_search_{$uniqId}";
                $ret .= '<div class="row">';
                $ret .= '<div class="col-xs-6">';
                $ret .= '<div class="input-group mb10">';
                $ret .= "<input type='text' name='form_map_search' id='{$searchId}' class='form-control notEnterSubmit'>";
                $ret .= '<div class="input-group-btn">';
                $ret .= $searchBtn;
                $ret .= '</div>';
                $ret .= '</div>';
                $ret .= '</div>';
                $ret .= '</div>';
                $searchStrJs = "
                    var sad = $('#{$searchId}').val();
                ";
            }

            // 検索用のJS
            $searchJs = "
                // 住所から検索
                $('#{$searchBtnId}').click(function() {
                    {$searchStrJs}
                    var geocoder = new google.maps.Geocoder();
                    geocoder.geocode({'address': sad}, function(results, status) {
                        if (status == google.maps.GeocoderStatus.OK) {
                            FormGm.map.setCenter(results[0].geometry.location);
                            FormMarker.setPosition(results[0].geometry.location);

                            var p = FormMarker.position;
                            $('#{$latId}').val(p.lat());
                            $('#{$lonId}').val(p.lng());
                        } else {
                            alert('住所から場所を特定できませんでした。最初にビル名などを省略し、番地までの検索などでお試しください。');
                        }
                    });
                    return false;
                });
            ";
        }

        // js
        $js = '';
        $js .= '<script type="text/javascript">';
        $js .= "
            jQuery(function($) {
                FormGm = null;
                FormMarker = null;
                {$mainJs}
                {$searchJs}
            });
        ";
        $js .= '</script>';
        app('larapack')->appendJsBlock($js);

        $ret .= "<div id='{$mapDivId}' style='overflow: hidden; width:{$size['width']}px; height:{$size['height']}px;'>Loading...</div>";
        $ret .= $this->form->hidden($latName, null, ['id' => $latId]);
        $ret .= $this->form->hidden($lonName, null, ['id' => $lonId]);
        $ret .= $this->form->hidden($zoomName, null, ['id' => $zoomId]);

        return new HtmlString($ret);
    }

    /**
     * 多対多checkbox
     * @param $name: modelのfunction名と同じにする
     *      function categories() ... => categories
     *      "categories[]"のように"[]"をつけてもつけなくてもいい
     * @param array $list
     * @param $selected
     * @param array $options
     *      'separator': 確認表記時の区切り文字列
     * @return HtmlString
     */
    public function multiCheckbox($name, $list = [], $selected = null, array $options = [])
    {
        // 名前を複数表記と通常と統一
        $name = str_replace('[]', '', $name);
        $nameMultiple = "{$name}[]";

        $separator = ' / ';
        if (!empty($options['separetor'])) {
            $separator = $options['separetor'];
            unset($options['separetor']);
        }

        if ($this->isConfirm()) {
            $ret = [];
            $value = $this->getDisplayInput($name);
            if (!empty($value)) {
                $items = $value->toArray();
                foreach ($items as $item) {
                    $ret[] = $list[$item['id']];
                }
            }
            return implode($separator, $ret);
        }

        // selectedは必ず配列にする
        if (!is_array($selected) && !empty($selected)) {
            $selected = [$selected];
        } else {
            $selected = [];
        }
        // selectedが存在しない場合,recordの初期値を入れておく
        if (empty($selected)) {
            $record = app('view')->shared('record');
            if (!empty($record->{$name})) {
                $items = $record->{$name}->toArray();
                foreach ($items as $item) {
                    $selected[] = $item['id'];
                }
            }
        }

        // class defaultでつける
        $options['class'] = !empty($options['class']) ? "{$options['class']} checkbox-inline" : 'checkbox-inline';

        $ret = '';
        $ret .= $this->form->hidden($name, ''); // 全て未チェックでもパラメータ飛ばすように(全チェック削除のため)
        foreach ($list as $id => $title) {
            $ret .= '<label>';
            $ret .= $this->form->checkbox($nameMultiple, $id, (!empty($selected) ? in_array($id, $selected) : null), $options);
            $ret .= $title;
            $ret .= '</label>';
        }

        return new HtmlString($ret);
    }

    /*
    |--------------------------------------------------------------------------
    | 共通系
    |--------------------------------------------------------------------------
    */

    /**
     * エラーを表示
     * @param $name
     * @return string
     */
    public function error($name, $format = '<div class="error-message">%s</div>')
    {
        // hasManyの場合, "["や"]"つなぎを.に変換
        $hasManyName = null;
        if (preg_match('/^(.+)\[([0-9]+)\]\[(.+)\]$/', $name, $m)) {
            $hasManyName = "{$m[1]}.{$m[2]}.{$m[3]}";
        }

        if (!$errors = app('view')->shared('errors')) {
            return '';
        }
        if (!$msg = !empty($hasManyName) ? $errors->first($hasManyName) : $errors->first($name)) {
            return '';
        }
        return new HtmlString(sprintf($format, $msg));
    }

    /**
     * 表示するための値(詳細の場合record値、それ以外は入力値)を取得
     * @param $name
     * @return mixed
     */
    public function getDisplayInput($name)
    {
        return $this->form->getValueAttribute($name);
    }

    /**
     * 表示するための値(詳細の場合record値、それ以外は入力値)をリストから取得
     * @param $name
     * @param $list
     * @return mixed|string
     */
    public function getDisplaySelect($name, $list)
    {
        $value = $this->getDisplayInput($name);
        return !empty($list[$value]) ? $list[$value] : null;
    }

    /**
     * テンプレートの置換
     * @param $format
     * @param $replaces:
     *      ['year' => '文字列'] で {{year}}が'文字列'に変換される
     * @return mixed
     */
    private function replaceTemplate($format, $replaces)
    {
        $ret = $format;
        foreach ($replaces as $search => $replace) {
            $ret = str_replace("{{{$search}}}", $replace, $ret);
        }
        return $ret;
    }

    /**
     * 全フォーム値のinput hidden タグを生成
     *
     * @param array|null $excepts 除外する配列※複数あれば配列で指定
     * @return string HtmlString
     */
    public function hiddenVars($excepts = null)
    {
        $input = \Session::get('_old_input');
        if (!is_array($excepts)) {
            $excepts[] = $excepts;
        }
        $excepts = array_merge(['_method', '_token'], $excepts);

        // 生成するhidden name, valueを配列で取得
        $retArr = [];
        if (!empty($input)) {
            foreach ($input as $key => $val) {
                if (!is_array($val)) {
                    $retArr[] = ['name' => $key, 'value' => $val];
                } else {
                    $retArr = array_merge($retArr, $this->multiArrayHiddenVars($val, $key));
                }
            }
        }

        $ret = '';
        foreach ($retArr as $arr) {
            if (!in_array($arr['name'], $excepts)) {
                $ret .= $this->form->hidden($arr['name'], $arr['value']) . "\n";
            }
        }

        return new HtmlString($ret);
    }

    /**
     *
     * @param $array
     * @param $parentKey
     */
    private function multiArrayHiddenVars($array, $parentKey)
    {
        // 配列の深さを調べる
        $depth = $this->arrayDepth($array);

        $ret = [];
        if ($depth === 1) {
            foreach ($array as $val) {
                // multiCheckbox(多対多)想定 keyはparentKey[]となる
                $ret[] = ['name' => "{$parentKey}[]", 'value' => $val];
            }
        } else {
            // 1対多想定
            $ret = array_merge($ret, $this->multiArrayHiddenVarsRecursive($array, $parentKey));
        }
        return $ret;
    }

    /**
     * 配列のキーを[]で連結したhiddenを返す
     * @param $array
     */
    private function multiArrayHiddenVarsRecursive($array, $parentKey)
    {
        if (!is_array($array)) {
            return false;
        }

        $ret = [];
        if (is_array($array)) {
            foreach ($array as $key => $val) {
                if (is_array($val)) {
                    $ret = array_merge($ret, $this->multiArrayHiddenVarsRecursive($val, "{$parentKey}[{$key}]"));
                } else {
                    $ret[] = ['name' => "{$parentKey}[{$key}]", 'value' => $val];
                }
            }
        }
        return $ret;
    }

    /**
     * 配列の深さを求める
     * @param $a
     * @param int $c
     * @return int|mixed
     */
    private function arrayDepth($a, $c = 0)
    {
        if (is_array($a) && count($a)) {
            ++$c;
            $_c = array($c);
            foreach ($a as $v) {
                if (is_array($v) && count($v)) {
                    $_c[] = $this->arrayDepth($v, $c);
                }
            }
            return max($_c);
        }
        return $c;
    }

}