<?php
namespace Cbase\View\Helper;
use Cake\View\Helper;
use Cake\View\View;
use Cake\Core\Configure;
use Cake\ORM\TableRegistry;
use Cake\Utility\Inflector;
use Cake\View\Form\NullContext;
use Cake\I18n\Date;
use Cake\I18n\Time;
use Cake\I18n\FrozenDate;
use Cake\I18n\FrozenTime;
/**
 * 入力画面と確認画面でフォームを切り替える
 *
 * @author ueno
 *
 * @property FormHelper $Form
 * @property HtmlHelper $Html
 * @property PrivateHelper $Private
 * @property MinifyHelper $Minify
 */
class CformHelper extends Helper
{

    /**
     * @var array helpers
     */
    public $helpers = ['Form', 'Html', 'Cbase.Private', 'Cbase.Minify'];

    /**
     * 確認画面かどうかの判定
     *
     * @return boolean
     */
    public function isConfirm()
    {
        return !empty($this->_View->get('confirmForm')) && $this->_View->get('confirmForm') === true;
    }

    /**
     * text
     * @param string $fieldName
     * @param array $options
     *      $options['link'] = trueにすると確認画面でリンク表示になる
     * @return string
     */
    public function text($fieldName, array $options = [])
    {
        if ($this->isConfirm()) {
            if (!$value = $this->_formValue($fieldName)) {
                return '&nbsp;';
            } elseif (!empty($options['link'])) {
                return $this->Html->link($value, $value, ['target' => '_blank']);
            } else {
                return $value;
            }
        }

        if (!empty($options['link'])) unset($options['link']);
        return $this->Form->text($fieldName, $options);
    }

    /**
     * password
     * @param string $fieldName
     * @param array $options
     * @return string
     */
    public function password($fieldName, array $options = [])
    {
        if ($this->isConfirm()) {
            if (!$value = $this->_formValue($fieldName)) return '&nbsp;';

            // 伏せて表示
            $length = mb_strlen($value);
            $str = '';
            for($i = 0; $i < $length; $i++){
                $str .= '*';
            }
            return $str;
        }
        return $this->Form->password($fieldName, $options);
    }

    /**
     * textarea
     * @param string $fieldName
     * @param array $options
     * @param array $wysiwyg array(
     *      'src' => 'ckeditor.jsの場所',
     *      'class' => 'class名(なければckeditor)',
     * @return string
     */
    public function textarea($fieldName, array $options = [], array $wysiwyg = [])
    {
        if ($this->isConfirm()) {
            $value = $this->_formValue($fieldName);
            if (!empty($wysiwyg)) {
                return !$value ? '&nbsp;' : $value;
            } else {
                return !$value ? '&nbsp;' : nl2br($value);
            }
        }

        $ret = "";
        if (!empty($wysiwyg)) {
            $ret .= $this->Html->script($wysiwyg['src'], ['block' => true]);
            // class
            if (!empty($wysiwyg['class'])) {
                $options['class'] = (!empty($options['class'])) ? "{$options['class']} {$wysiwyg['class']}" : $wysiwyg['class'];
            } else {
                $options['class'] = (!empty($options['class'])) ? "{$options['class']} ckeditor" : 'ckeditor';
            }
        }
        $ret .= $this->Form->textarea($fieldName, $options);
        return $ret;
    }

    /**
     * checkbox
     * @param string $fieldName
     * @param array $options
     * @param array $display label array('入力画面のlabel', '確認画面チェックした場合のlabel', '確認画面チェックしない場合のlabel')
     * @return string
     */
    public function checkbox($fieldName, array $options = [], array $display = ['editLabel', 'YESLabel', 'NOLabel'])
    {
        if ($this->isConfirm()) {
            return (!$this->_formValue($fieldName)) ? $display[2] : $display[1];
        }

        $ret = "";
        $ret .= '<label>';
        $ret .= $this->Form->checkbox($fieldName, $options);
        $ret .= $display[0];
        $ret .= '</label>';
        return $ret;
    }

    /**
     * 画像アップロード
     * @param string $fieldName
     * @param array $options
     * @param array $size 画像のサイズ
     * @param array $imageOptions
     * @return string
     */
    public function imageFile($fieldName, array $options = [], array $size = [160, null], array $imageOptions = [])
    {
        if (!empty($size[0])) {
            $imageOptions['width'] = $size[0];
        }
        if (!empty($size[1])) {
            $imageOptions['height'] = $size[1];
        }

        if ($this->isConfirm()) {
            if ($image = $this->_formValue($fieldName)) {
                return $this->Html->image("/{$image}", $imageOptions);
            } else {
                return '-';
            }
        }

        $defaults = [
            'label' => false,
            'error' => false
        ];
        $options = array_merge($defaults, $options);

        $ret = '';
        if ($image = $this->_formValue($fieldName)) {
            $ret .= $this->Html->image("/{$image}", $imageOptions);
            $ret .= '<br>';
            $ret .= '<label>';
            $ret .= $this->Form->checkbox("{$fieldName}_shouldDelete", ['label' => false]);
            $ret .= 'この画像を削除';
            $ret .= '</label>';
        }
        $ret .= $this->Form->file("{$fieldName}_file", $options);
        $ret .= $this->Form->hidden($fieldName);

        return $ret;
    }

    /**
     * 画像アップロードAjax版
     * @param string $fieldName
     * @param array $options
     * @param array $size 画像のサイズ
     * @param array $imageOptions
     * @return string
     *
     * views/pages/upload_file.ctpを配置。内容は以下
     * <?php echo $cform->imageFileAjax(h($fieldName), aa('isAjax', true)); ?>
     */
    public function imageFileAjax($fieldName, array $options = [], array $size = [160, null], array $imageOptions = [])
    {
        if (!empty($size[0])) {
            $imageOptions['width'] = $size[0];
        }
        if (!empty($size[1])) {
            $imageOptions['height'] = $size[1];
        }

        if ($this->isConfirm()) {
            if ($image = $this->_formValue($fieldName)) {
                return $this->Html->image("/{$image}", $imageOptions);
            } else {
                return '-';
            }
        }

        $defaults = [
            'label' => false,
            'error' => false
        ];
        $options = array_merge($defaults, $options);

        // inputのclass
        $uid = uniqid();
        $fn = str_replace('.', '_', $fieldName);
        $fileClass = "inputFileAjax_{$fn}_{$uid}";
        $options['class'] = !empty($options['class']) ? "{$options['class']} {$fileClass}" : $fileClass;
        // 画像表示する部分のdivタグのclassを指定
        $divClass = "FileUploadDiv_{$fn}_{$uid}";

        // table名求める
        $entity = $this->_View->get('formEntity');
        $tableName = $entity->source();

        $ret = '';
        // 画像アップjs
        $ret .= $this->_View->element('Cbase.upload_image_js', [
            'fieldName' => $fieldName,
            'tableName' => $tableName,
            'fileSel' => ".{$fileClass}",
            'photoAreaSel' => ".{$divClass}",
            'size' => $size,
            'imageOptions' => $imageOptions,
        ]);

        // 初期表示
        $initHtml = $this->_View->element('Cbase.upload_image_box', [
            'fieldName' => $fieldName,
            'tableName' => $tableName,
            'imageOptions' => $imageOptions,
        ]);

        $ret .= "<div class='{$divClass}'>{$initHtml}</div>";
        $ret .= $this->Form->file("{$fieldName}_file", $options);

        return $ret;
    }

    /**
     * 通常のfileアップロード&確認画面でのダウンロードリンク
     * @param array $fileField = [$fileFieldName, $fileOptions = []]
     * @param array $nameField = [$nameFieldName, $nameOptions = []]
     * @return string
     */
    public function downloadFile(array $fileField = [], array $nameField = [])
    {
        if ($this->isConfirm()) {
            if ($file = $this->_formValue($fileField[0])) {
                $name = (!empty($nameField[0]) && $this->_formValue($nameField[0])) ? h($this->_formValue($nameField[0])) : 'ファイル';
                return $this->Html->link($name, "/{$file}", ['target' => '_blank']);
            } else {
                return '-';
            }
        }

        // option
        $fileOptions = (!empty($fileField[1]) && is_array($fileField[1])) ? $fileField[1] : [];
        $nameOptions = (!empty($nameField[1]) && is_array($nameField[1])) ? $nameField[1] : [];
        // default options
        $defaults = [
            'label' => false,
            'error' => false
        ];
        $fileOptions = array_merge($defaults, $fileOptions);

        $ret = '';

        if (!empty($nameField[0])) {
            $ret .= '<label>';
            $ret .= 'ファイル名: ';
            $ret .= '<br>';
            $ret .= $this->Form->text($nameField[0], $nameOptions);
            $ret .= '</label>';
            $ret .= '<br>';
        }

        // ファイルあればリンクを表示
        if ($file = $this->_formValue($fileField[0])) {
            $ret .= $this->Html->link('ファイル', "/{$file}", ['target' => '_blank']);
            $ret .= '<br>';
            $ret .= '<label>';
            $ret .= $this->Form->checkbox("{$fileField[0]}_shouldDelete", ['label' => false, 'class' => 'form-control checkbox-inline']);
            $ret .= 'このファイルを削除';
            $ret .= '</label>';
        }
        // ファイル名入力あればファイル選択labelを表示
        if (!empty($nameField[0])) {
            $ret .= "<br>";
            $ret .= '<label>';
            $ret .= 'ファイル選択: ';
            $ret .= "<br>";
        }
        $ret .= $this->Form->file("{$fileField[0]}_file", $fileOptions);
        if (!empty($nameField[0])) {
            $ret .= '</label>';
        }
        $ret .= $this->Form->hidden($fileField[0]);

        return $ret;
    }

    /**
     * 通常のfileアップロード&確認画面でのダウンロードリンク
     * @param array $fileField = [$fileFieldName, $fileOptions = []]
     * @param array $nameField = [$nameFieldName, $linkOptions = []]
     * @param array $options
     * @return string
     */
    public function downloadFileAjax(array $fileField = [], array $nameField = [])
    {
        if ($this->isConfirm()) {
            if ($file = $this->_formValue($fileField[0])) {
                $name = (!empty($nameField[0]) && $this->_formValue($nameField[0])) ? h($this->_formValue($nameField[0])) : 'ファイル';
                return $this->Html->link($name, "/{$file}", ['target' => '_blank']);
            } else {
                return '-';
            }
        }

        // option
        $fileOptions = (!empty($fileField[1]) && is_array($fileField[1])) ? $fileField[1] : [];
        $nameOptions = (!empty($nameField[1]) && is_array($nameField[1])) ? $nameField[1] : [];
        // default options
        $defaults = [
            'label' => false,
            'error' => false
        ];
        $fileOptions = array_merge($defaults, $fileOptions);

        // inputのclass
        $uid = uniqid();
        $fn = str_replace('.', '_', $fileField[0]);
        $fileClass = "inputFileAjax_{$fn}_{$uid}";
        $fileOptions['class'] = !empty($options['class']) ? "{$options['class']} {$fileClass}" : $fileClass;
        // 画像表示する部分のdivタグのclassを指定
        $divClass = "FileUploadDiv_{$fn}_{$uid}";

        // table名求める
        $entity = $this->_View->get('formEntity');
        $tableName = $entity->source();

        $ret = '';

        if (!empty($nameField[0])) {
            $ret .= '<label>';
            $ret .= 'ファイル名: ';
            $ret .= "<br>";
            $ret .= $this->Form->text($nameField[0], $nameOptions);
            $ret .= '</label>';
            $ret .= "<br>";
        }

        // 画像アップjs
        $ret .= $this->_View->element('Cbase.upload_file_js', [
            'fieldName' => $fileField[0],
            'tableName' => $tableName,
            'fileSel' => ".{$fileClass}",
            'fileAreaSel' => ".{$divClass}",
        ]);

        // 初期表示
        $initHtml = $this->_View->element('Cbase.upload_file_box', [
            'fieldName' => $fileField[0],
            'tableName' => $tableName,
        ]);

        $ret .= "<div class='{$divClass}'>{$initHtml}</div>";
        $ret .= $this->Form->file("{$fileField[0]}_file", $fileOptions);

        return $ret;
    }

    /**
     * select
     * @param string $fieldName
     * @param array $options
     * @param array $attributes trueにすると<select multiple>になる,'checkbox'にするとcheckboxになる
     * @return string
     */
    public function select($fieldName, array $options = [], array $attributes = [])
    {
        $separator = !empty($attributes['separator']) ? $attributes['separator'] : null;
        unset($attributes['separator']);
        // multiple checkboxかどうか
        $isMultiCheckbox = (!empty($attributes['multiple']) && $attributes['multiple'] == 'checkbox');

        if ($this->isConfirm()) {
            $value = $this->_formValue($fieldName);
            if (!$isMultiCheckbox) {
                if (!$value) {
                    return !empty($attributes['empty']) ? $attributes['empty'] : '未選択';
                } else {
                    return $options[$value];
                }
            } else {
                if (empty($value)) return '-';

                // マルチ選択の場合
                if (empty($separator)) $separator = ' ';
                $str = '';
                foreach ($value as $val) {
                    if (!empty($options[$val])) {
                        if (!empty($str)) $str .= $separator;
                        $str .= $options[$val];
                    }
                }
                return empty($str) ? '-' : $str;
            }
        }

        // multiでない場合
        if (!$isMultiCheckbox) {
            return $this->Form->select($fieldName, $options, $attributes);
        } else {
            // multiple=checkboxの場合
            // templateある場合、それはめる
            $template = null;
            if (!empty($attributes['template'])) {
                $template = $attributes['template'];
            } else {
                // classは各checkboxに当てる
                $class = !empty($attributes['class']) ? " class='{$attributes['class']}'" : '';
                $template = [
                    'checkbox' => '<input type="checkbox" name="{{name}}" value="{{value}}"{{attrs}}' . $class . '">',
                    'checkboxWrapper' => '{{label}}'
                ];
            }
            return $this->Form->input($fieldName, [
                'options' => $options,
                'type' => 'select',
                'multiple' => 'checkbox',
                'label' => false, // 最初のhiddenのlabelを表示しない
                'templates' => $template
            ]);
        }
    }

    /**
     * selectDate
     * @param string $fieldName
     * @param array $options inputのoptions
     * @param array $customs
     *      $customs['start'] = $minYear年を何月何日から選択可能にするか 例) [3]: 3/1から, [6, 15] 6/15から
     *      $customs['end'] = $maxYearの年を何月何日まで選択可能にするか 例) [3]: 3/31まで, [6, 15] 6/15まで
     * @return string
     */
    public function selectDate($fieldName, array $options = [], array $customs = [])
    {
        if ($this->isConfirm()) {
            $value = $this->_formValue($fieldName);
            if (!empty($value['year']) && !empty($value['month']) && !empty($value['day'])) {
                return "{$value['year']}年{$value['month']}月{$value['day']}日";
            } else {
                return $value;
            }
        }

        $defaults = [
            'type' => 'date',
            'dateFormat' => 'YMD',
            'label' => false,
            'monthNames' => false,
            'empty' => '----',
            'minYear' => 1980,
            'maxYear' => date('Y'),
            'orderYear' => 'asc',
            'year' => ['class' => 'form-control'],
            'month' => ['class' => 'form-control'],
            'day' => ['class' => 'form-control'],
            'templates' => [
                'dateWidget' => '{{year}} 年 {{month}} 月 {{day}} 日'
            ],
            'error' => false
        ];
        $options = array_merge($defaults, $options);

        // y, m, dそれぞれのjs用classを決定する
        $uid = uniqid();
        $yearClass = "selectDateYear_{$uid}";
        $monthClass = "selectDateMonth_{$uid}";
        $dayClass = "selectDateDay_{$uid}";
        $options['year']['class'] .= " {$yearClass}";
        $options['month']['class'] .= " {$monthClass}";
        $options['day']['class'] .= " {$dayClass}";

        // 最初の年の選択可能な年月ある場合
        $startM = $startD = 0;
        if (!empty($customs['start'])) {
            if (count($customs['start']) === 2) {
                $startM = $customs['start'][0];
                $startD = $customs['start'][1];
            } else {
                $startM = $customs['start'][0];
            }
        }

        // 最終年の選択可能な年月ある場合
        $endM = $endD = 0;
        if (!empty($customs['end'])) {
            if (count($customs['end']) === 2) {
                $endM = $customs['end'][0];
                $endD = $customs['end'][1];
            } else {
                $endM = $customs['end'][0];
            }
        }

        // js
        $js = $this->Html->scriptBlock($this->Minify->scriptBlock("
            var isEmptyName = " . (!empty($options['empty']) ? "'{$options['empty']}'" : 'false') . ";

            jQuery(function($) {

                // 選択可能な最初の年月
                var startY = " . $options['minYear'] . ";
                var startM = " . $startM . ";
                var startD = " . $startD . ";

                // 選択可能な最終年月
                var endY = " . $options['maxYear'] . ";
                var endM = " . $endM . ";
                var endD = " . $endD . ";

                // 年変更時
                $('select.{$yearClass}').change(function(){
                    startEndMDCheck();
                    leapYearCheck();
                });
                // 月変更時
                $('select.{$monthClass}').change(function(){
                    startEndMDCheck();
                    leapYearCheck();
                });

                // 初回実行
                leapYearCheck();
                startEndMDCheck();

                // 最初、最終年月チェック
                function startEndMDCheck(){
                    var y = $('.{$yearClass}').val();
                    var currentMonthSelect = $('.{$monthClass}').val();
                    var first = 1;
                    var last = 12;
                    $('.{$monthClass}').empty();
                    if(isEmptyName) $('.{$monthClass}').append('<option>' + isEmptyName + '</option>');
                    // 最初年選択時に最初表示月指定ある場合
                    if (parseInt(y) == startY && (0 < startM && startM <= 12)) {
                        first = startM;
                    }
                    // 最終年選択時に最終表示月指定ある場合
                    if (parseInt(y) == endY && (0 < endM && endM <= 12)) {
                        last = endM;
                    }
                    for (var i = first; i <= last; i++) {
                        var selected = (parseInt(currentMonthSelect) == i) ? ' selected=\'selected\' ' : '';
                        $('.{$monthClass}').append('<option value=\'' + i + '\'' + selected + '>' + i + '</option>');
                    }
                }

                // うるう年判定して日付を表示
                function leapYearCheck(){
                    // dayプルダウン初期化
                    var currentDaySelect = $('.{$dayClass}').val();
                    $('.{$dayClass}').empty();
                    if(isEmptyName) $('.{$dayClass}').append('<option>' + isEmptyName + '</option>');

                    var y = $('.{$yearClass}').val();
                    var m = $('.{$monthClass}').val();
                    var first = 1;
                    var last;
                    if (m == 2 && (y % 400 == 0 || (y % 4 == 0 && y % 100 != 0))) {
                        last = 29;
                    } else {
                        last = new Array(31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31)[m - 1];
                    }

                    // 最初表示年月で最終選択日の指定がある場合その日が最後
                    if (parseInt(y) == startY && parseInt(m) == startM && (0 < startD && startD <= last)) {
                        first = startD;
                    }
                    // 最終表示年月で最終選択日の指定がある場合その日が最後
                    if (parseInt(y) == endY && parseInt(m) == endM && (0 < endD && endD <= last)) {
                        last = endD;
                    }

                    for (var i = first; i <= last; i++) {
                        selected = (parseInt(currentDaySelect) == i) ? ' selected=\'selected\' ' : '';
                        $('.{$dayClass}').append('<option value=\'' + i + '\'' + selected + '>' + i + '</option>');
                    }
                }
            });
        "), ['block' => true]);

        $ret = "";
        // js
        $ret .= $js;
        // input
        $ret .= $this->Form->input($fieldName, $options);

        return $ret;
    }

    /**
     * jQuery.datetimepickerから日時を選択し、textに格納するフォーム
     * @param string $fieldName
     * @param array $options
     *      'dpOption' : '{a:b, c:d}'等のハッシュで指定するとjquery.datepickerのオプションとなる
     * @param boolean $time
     * @return string
     */
    public function textDate($fieldName, array $options = [], $time = false)
    {
        if ($this->isConfirm()) {
            $format = ($time) ? 'Y年m月d日 H:i:s' : 'Y年m月d日';
            return (!$value = $this->_formValue($fieldName)) ? '&nbsp;' : h($this->Private->getFormatDate($format, $value));
        }

        // inputのidを決定する
        if (empty($options['id'])) {
            $options['id'] = "{$fieldName}_" . uniqid();
        }
        $selector = "#" . h($options['id']);

        $ret = "";

        // css&javascript読み込み
        $http = isset($_SERVER['HTTPS']) ? 'https' : 'http';
        $ret .= $this->Html->css($http.'://ajax.googleapis.com/ajax/libs/jqueryui/1.11.4/themes/smoothness/jquery-ui.css', ['block' => true]);
        $ret .= $this->Html->script('/cbase/js/jquery.ui.datepicker-ja', ['block' => true]);
        if ($time) {
            $ret .= $this->Html->css('/cbase/js/jquery-ui-timepicker-addon.css', ['block' => true]);
            $ret .= $this->Html->script('/cbase/js/jquery-ui-timepicker-addon', ['block' => true]);
            $jOpt = "{
                timeOnlyTitle: '時間',
                timeText: '時間',
                hourText: '時',
                minuteText: '分',
                secondText: '秒',
                closeText: '閉じる',
                currentText: '現在',
                showButtonPanel: true,
                dateFormat: 'yy-mm-dd',
                showSecond: true,
                controlType: 'select',
                timeFormat: 'HH:mm:ss'
            }";
            if (!empty($options['dpOption'])) {
                $jOpt = "jQuery.extend({$jOpt}, {$options['dpOption']})";
            }
            $ret .= $this->Html->scriptBlock($this->Minify->scriptBlock("
                jQuery(function($){
                    $(document).on('focus', '{$selector}', function(){
                        var options = {$jOpt};
                        $(this).datetimepicker(options);
                    });
                });
            "), ['block' => true]);
        }else{
            $jOpt = "{
                closeText: '閉じる',
                currentText: '今日',
                showButtonPanel: true,
                dateFormat: 'yy-mm-dd'
            }";
            if (!empty($options['dpOption'])) {
                $jOpt = "$.extend({$jOpt}, {$options['dpOption']})";
            }
            $ret .= $this->Html->scriptBlock($this->Minify->scriptBlock("
                jQuery(function($){
                    var options = {$jOpt};
                    $(document).on('focus', '{$selector}', function(){
                        $(this).datepicker(options);
                    });
                });
            "), ['block' => true]);
        }

        if (!empty($options['dpOption'])) {
            unset($options['dpOption']);
        }

        // value指定ある場合取得
        if (isset($options['value']) && !isset($options['val'])) {
            $options['val'] = $options['value'];
            unset($options['value']);
        }
        if (!isset($options['val'])) {
            $options['val'] = $this->Form->context()->val($fieldName);
        }
        if (!isset($options['val']) && isset($options['default'])) {
            $options['val'] = $options['default'];
        }
        unset($options['value'], $options['default']);
        // value存在する場合でobjectの場合文字列に変換
        if (!empty($options['val'])) {
            $options['val'] = $this->_objectToStr($options['val']);
        }

        $ret .= $this->Form->text($fieldName, $options);
        return $ret;
    }

    /**
     * hidden
     * @param string $fieldName
     * @param array $options
     * @return string|null
     */
    public function hidden($fieldName, array $options = [])
    {
        if ($this->isConfirm()) {
            // 確認画面では何もしない
            return null;
        }
        return $this->Form->hidden($fieldName, $options);
    }

    /**
     * radio
     * @param string $fieldName
     * @param array $options
     * @param array $attributes
     *      'empty'を指定すると確認画面でemptyだった場合に表示
     * @return string
     */
    public function radio($fieldName, array $options = [], array $attributes = [])
    {
        if ($this->isConfirm()) {
            $emptyMsg = empty($attributes['empty']) ? '未選択' : $attributes['empty'];
            $value = $this->_formValue($fieldName);
            return (!$value && empty($options[$value])) ? $emptyMsg : $options[$value];
        }

        return $this->Form->radio($fieldName, $options, $attributes);
    }

    /**
     * 座標の入力フォーム(google map api version 3用)
     * @param $fieldNames array(緯度、経度、ズーム)
     * @param $options
     *      'size' mapのサイズ array(幅, 高さ)で指定可能
     *      'default' 初期位置 array(緯度, 経度, ズーム)で指定可能
     * @param $addressSearch
     *      true : 地図検索フォームを生成
     *      ['field' => 'フィールドのid'] 該当のidのフィールドのinputを利用
     *      ['field' => ['フィールド1のid', 'フィールド2のid' .. ]] 該当のidのフィールドのinput(複数)を文字列連結して使用
     *      ['field' => ['フィールド1のid', 'フィールド2のid' .. ], 'prefix' => 'prefix文字列'] prefix文字列に該当のフィールド名のinput(複数)を文字列連結して使用
     * @param int $mode
     *      0 : 地図を常に表示
     *      1 : 地図を常に非表示
     *      2 : 新規作成時は表示、更新時は非表示
     *      3 : 緯度か経度が空の時は表示、その他は非表示
     * @return string
     */
    public function map(array $fieldNames = [], array $options = [], $addressSearch = false, $mode = 0)
    {
        $http = isset($_SERVER['HTTPS']) ? 'https' : 'http';

        // 設定値($optionsで上書き可能)
        $size = [555, 350];
        $default = [35.7090259, 139.7319925, 4];
        if (!empty($options['size'])) $size = $options['size'];
        if (!empty($options['default'])) $default = $options['default'];

        if ($this->isConfirm()) {
            $valueLat = $this->_formValue($fieldNames[0]);
            $valueLon = $this->_formValue($fieldNames[1]);
            $valueZoom = $this->_formValue($fieldNames[2]);
            if ($valueLat && $valueLon && $valueZoom) {
                $ret = "<img src='{$http}://maps.google.com/maps/api/staticmap?";
                $ret .= "center={$valueLat},{$valueLon}&amp;zoom={$valueZoom}&amp;size={$size[0]}x{$size[1]}";
                $ret .= "&amp;maptype=roadmap&amp;markers=color:red|{$valueLat},{$valueLon}&amp;region=jp' />";
                return "<div>{$ret}</div>";
            }else{
                return '<div>-</div>';
            }
        }

        // value取得
        $valueLat = $this->Form->context()->val($fieldNames[0]);
        $valueLon = $this->Form->context()->val($fieldNames[1]);
        $valueZoom = $this->Form->context()->val($fieldNames[2]);

        $ret = "";

        // ID属性指定に利用
        $uniqId = uniqid();
        $fieldIds = ["{$fieldNames[0]}_{$uniqId}", "{$fieldNames[1]}_{$uniqId}", "{$fieldNames[2]}_{$uniqId}"];

        $dispBtnId = "gmapdispbtn_{$uniqId}";

        // 地図表示ボタン
        $ret .= "<a href='javascript:void(0);' id='{$dispBtnId}' class='btn btn-primary'>地図を表示</a>";

        // js読み込み
        $mainJs = "
            // google map --------------------------------------------------------------
            var lat = ". ($valueLat ? $valueLat : $default[0]) . ";
            var lon = ". ($valueLon ? $valueLon : $default[1]) . ";
            var zoom = ". ($valueZoom ? $valueZoom : $default[2]) . ";
            CformGm = new Gmap($('#map').get(0), {centerLatLng: [lat, lon], zoom: zoom, width: $('#map').css('width'), height: $('#map').css('height')});
            CformGm.show();
            google.maps.event.addListener(CformGm.map, 'idle', function(){
                var centerObj = CformGm.map.getCenter();

                $('#".$fieldIds[0]."').val(centerObj.lat());
                $('#".$fieldIds[1]."').val(centerObj.lng());
                $('#".$fieldIds[2]."').val(CformGm.map.getZoom());

                if(CformMarker) CformGm.removeOverlay(CformMarker);

                CformMarker = CformGm.setMarker([centerObj.lat(), centerObj.lng()]);
            });
        ";

        $dispJs = "$('#map').show();";
        $searchDivId = "gmapsearchdiv_{$uniqId}";
        $searchJs = "";
        $searchBtn = $this->Html->link('地図を検索', 'javascript:void(0);', array('id' => 'getad', 'class' => 'btn btn-primary getad'));
        if (!empty($addressSearch)) {
            // 他のテキストフォームを使う
            $searchStrJs = "";
            if (is_array($addressSearch) && !empty($addressSearch['field'])) {
                $prefixStr = (!empty($addressSearch['prefix'])) ? $addressSearch['prefix'] : '';
                if (!is_array($addressSearch['field'])) {
                    $searchStrJs = "
                        var sad = '".$prefixStr."' + $('#".$addressSearch['field']."').val();
                    ";
                } else {
                    // field指定が複数ある場合この入力値を連結
                    $searchStrJs = "
                        var sad = '".$prefixStr."';
                        var cameledArr = ".json_encode($addressSearch['field']).";
                        for (var i = 0; i < cameledArr.length; i++) {
                            var inputObj = $('#' + cameledArr[i]);
                            if(inputObj[0].nodeName.toLowerCase() === 'input') {
                                // input
                                sad = sad + inputObj.val();
                            } else {
                                // select
                                sad = sad + inputObj.children(':selected').text();
                            }
                        }
                    ";
                }
                $ret .= "<div class='mb10' id='{$searchDivId}' style='display:none;'>";
                $ret .= $searchBtn;
                $ret .= '</div>';
            } else {
                // テキストフォームを用意
                $searchId = 'cform_map_search_' . uniqid();
                $ret .= "<div class='row' id='{$searchDivId}' style='display:none;'>";
                $ret .= '<div class="col-xs-6">';
                $ret .= '<div class="input-group mb10">';
                $ret .= $this->Form->text('cform_map_search', array('id' => $searchId, 'class' => 'form-control'));
                $ret .= '<div class="input-group-btn">';
                $ret .= $searchBtn;
                $ret .= '</div>';
                $ret .= '</div>';
                $ret .= '</div>';
                $ret .= '</div>';
                $searchStrJs = "
                    var sad = $('#".$searchId."').val();
                ";
            }

            // 検索用のJS
            $searchJs = "
                // 住所から検索
                $('#getad').click(function() {
                    ".$searchStrJs."
                    var geocoder = new google.maps.Geocoder();
                    geocoder.geocode({'address': sad}, function(results, status) {
                        if (status == google.maps.GeocoderStatus.OK) {
                            CformGm.map.setCenter(results[0].geometry.location);
                            CformMarker.setPosition(results[0].geometry.location);

                            var p = CformMarker.position;
                            $('#".$fieldIds[1]."').val(p.lat());
                            $('#".$fieldIds[2]."').val(p.lng());
                        } else {
                            alert('住所から場所を特定できませんでした。最初にビル名などを省略し、番地までの検索などでお試しください。');
                        }
                    });
                    return false;
                });
            ";
            $dispJs .= "
            $('#{$searchDivId}').show();";
        }

        // 常に地図を表示するかどうか
        $dispAlways = false;
        if(empty($mode)) {
            $dispAlways = true;
        }
        else if($mode == 2) {
            if(empty($this->Form->context()->val("id"))) {
                $dispAlways = true;
            }
        }
        else if($mode == 3) {
            if(empty($valueLat) || empty($valueLon)) {
                $dispAlways = true;
            }
        }

        // js
        $js = '';
        $js .= "
        function initMap_{$uniqId}() {";
        $js .= $this->appendJsOnEvent('/cbase/js/gmap.v3.js', "initMap2_{$uniqId}");
        $js .= "
        }";
        $js .= "
        function initMap2_{$uniqId}() {
            CformGm = null;
            CformMarker = null;
            {$mainJs}
            {$searchJs}
        }";
        $js .= "
        jQuery(function($) {";
        if(!$dispAlways) {
            $js .= "
            $('#{$dispBtnId}').click(function() {";
        }
        $js .= "
            {$dispJs}";
        $js .= $this->appendJsOnEvent("{$http}://maps.google.com/maps/api/js?language=ja&region=jp&key=" . GOOGLE_MAP_API_KEY . "&callback=initMap_" . $uniqId);
        $js .= "
            $('#{$dispBtnId}').hide();";
        if(!$dispAlways) {
            $js .= "
            });";
        }
        $js .= "
        });";

        // js
        $ret .= $this->Html->scriptBlock($this->Minify->scriptBlock($js), array('inline' => false));

        $ret .= "<div id='map' style='overflow: hidden; width:{$size[0]}px; height:{$size[1]}px; display:none;'>Loading...</div>";
        $ret .= $this->Form->hidden($fieldNames[0], ['id' => $fieldIds[0]]);
        $ret .= $this->Form->hidden($fieldNames[1], ['id' => $fieldIds[1]]);
        $ret .= $this->Form->hidden($fieldNames[2], ['id' => $fieldIds[2]]);

        return $ret;
    }

    /**
     * 確認画面またはエラーがない場合は出力しないバリデーションエラーメッセージ
     * 出力する場合は先頭に<br />が付加される
     * @param string $field
     * @param string|null $text
     * @param array $options
     * @param string $format string format specifiers
     * @return string|null
     */
    public function error($field, $text = null, array $options = [], $format = '%s')
    {
        if($this->isConfirm()) return null;
        elseif(!$this->Form->error($field, $text, $options)) return null;

        return sprintf($format, $this->Form->error($field, $text, $options));
    }

    /**
     * Field名をIDのキャメルに変換
     *
     * @param string $fieldName
     * @return string
     */
    private function _cameled($fieldName)
    {
        $names = explode('.', $fieldName);
        $cameled = '';
        foreach($names as $name){
            $cameled .= Inflector::camelize($name);
        }

        return $cameled;
    }

    /**
     * 指定のfieldのvalueを返す
     * show/deleteConfirmのform内の要素でない場合, View内の$formEntityを使う
     */
    private function _formValue($field)
    {
        $entity = $this->_View->get('formEntity');
        if (preg_match("/^(.+)\._ids$/", $field, $m)) {
            // habtm
            $assoc = $m[1];
            $value = [];
            foreach ($entity->{$assoc} as $obj) {
                $value[] = $obj->id;
            }
            return $value;
        } elseif (preg_match("/^(.+)\.(.+)\.(.+)$/", $field, $m)) {
            // hasMany
            list($f, $assoc, $key, $fieldName) = $m;
            return $entity->{$assoc}[$key]->{$fieldName};
        } else {
            if (empty($entity->{$field})) {
                return null;
            }
            $value = $entity->{$field};
            // objectは文字列に変更
            $value = $this->_objectToStr($value);
            return h($value);
        }
    }

    /**
     * DateやDateTimeをform値として扱いにくいので文字列に変換
     */
    private function _objectToStr($obj)
    {
        $value = $obj;
        if ($obj instanceof Date) {
            $value = $obj->i18nFormat('yyyy-MM-dd');
        } elseif ($obj instanceof Time) {
            $value = $obj->i18nFormat('yyyy-MM-dd HH:mm:ss');
        } elseif ($obj instanceof FrozenDate) {
            $value = $obj->i18nFormat('yyyy-MM-dd');
        } elseif ($obj instanceof FrozenTime) {
            $value = $obj->i18nFormat('yyyy-MM-dd HH:mm:ss');
        }
        return $value;
    }

    /**
     * jsをイベントなどでappendする
     */
    public function appendJsOnEvent($src, $onload = false)
    {
        $script = '
        {
            var s = document.createElement("script");
            s.type = "text/javascript";
            s.src = "' . $src . '";';
        if(!empty($onload)) {
            $script .= '
            s.onload = ' . $onload . ';';
        }
        $script .= '
            document.getElementsByTagName("head")[0].appendChild(s);
        }';
        return $script;
    }
}
