<?php
namespace Cbase\Model\Behavior;

use ArrayObject;
use Cake\Event\Event;
use Cake\ORM\Behavior;
use Cake\ORM\Entity;
use Cake\ORM\Table;
use Cake\Utility\Text;
use Cake\Utility\Inflector;
use Cake\ORM\TableRegistry;

/**
 * MultiBlock behavior
 * N段フォームでsequenceの付与や存在しないブロックの削除処理
 *
 * @author ueno
 */
class MultiBlockBehavior extends Behavior
{

    /**
     * Default configuration.
     *
     * @var array
     */
    protected $_defaultConfig = [
        'targets' => []
    ];

    public function initialize(array $config)
    {
        $targets = [];
        foreach ($config as $tableName => $sequenceField) {
            $propertyName = Inflector::underscore($tableName);
            $targets[] = compact('tableName', 'propertyName', 'sequenceField');
        }
        $this->config('targets', $targets, false);
    }

    /**
     * beforeMarshal callbackメソッド
     *
     * @param Event $event
     * @param ArrayObject $data オブジェクトなので内部をいじると呼び出し元に反映される
     * @param ArrayObject $options
     */
    public function beforeMarshal(Event $event, ArrayObject $data, ArrayObject $options)
    {
        foreach ($this->config('targets') as $target) {
            // $this->request->dataの配列の順にsequenceをつけておく
            if (!empty($data[$target['propertyName']])) {
                $num = 1;
                foreach ($data[$target['propertyName']] as $key => $val) {
                    $data[$target['propertyName']][$key][$target['sequenceField']] = $num;
                    $num++;
                }
            }
        }
    }

    /**
     * afterSave
     * パラメータが渡ってきていないブロックは削除する
     *
     * @param Event $event
     * @param Entity $entity
     * @param ArrayObject $options
     * @return bool
     */
    public function afterSave(Event $event, Entity $entity, ArrayObject $options)
    {
        foreach ($this->config('targets') as $target) {
            if (empty($entity->{$target['propertyName']})) {
                continue;
            }

            // idを取り出す
            $ids = [];
            foreach ($entity->{$target['propertyName']} as $hasManyEntity) {
                $ids[] = $hasManyEntity->id;
            }

            // hasManyテーブルの外部キーと、親ID
            $foreignKey = $this->_table->hasMany($target['tableName'])->foreignKey();
            $parentId = $entity->id;

            // 保存されたブロック以外は削除する条件
            // 何も渡ってきていない場合全て
            $conditions = [$foreignKey => $parentId];
            if (!empty($ids)) {
                $conditions += [
                    'NOT' => ['id IN' => $ids]
                ];
            }

            // 削除
            $hasManyTable = TableRegistry::get($target['tableName']);
            $deleteDatas = $hasManyTable->find('all')->where($conditions)->toArray();
            // beforeDelete, afterDeleteを動かすためにひとつずつ削除
            // ここだけのためにhasMany cascadeCallbacksしたくもないので
            foreach ($deleteDatas as $deleteEntity) {
                $hasManyTable->delete($deleteEntity);
            }
        }
        return true;
    }

}
