<?php
namespace Cbase\Controller\Component;

use Cake\Controller\Component;
use Cake\Event\Event;
use Cake\Core\Configure;
use Cake\Utility\Text;
use Cake\Utility\Inflector;

/**
 * Privateコンポネント
 * @author ueno
 *
 * controller内で共通で利用する処理を記述した
 *
 * @property PrivateComponent $Private
 */
class PrivateComponent extends Component {

    /**
     * @var array 利用するcomponent
     */
    public $components = ['Flash'];

    /**
     * @var string メイン利用モデル名
     */
    public $mainTable;

    /**
     * @var array indexへのリダイレクトURL(ルーティング配列形式)
     */
    public $redirectUrl;

    /**
     * @var Controller コントローラのインスタンス
     */
    protected $controller;

    /**
     * @var array 設定
     */
    public $privateSettings = [];

    /**
     * @var object formにsetするEntity
     */
    public $formEntity = null;

    /**
     * @var object formにsetするToken, Sessionのキーに
     */
    public $formToken = null;


    /**
    * initialize
     *
     * @param Controller $controller
    */
    public function initialize(array $config)
    {

        // コントローラのインスタンスを格納
        $this->controller = $this->_registry->getController();

        // メインtable
        $this->mainTable = $this->controller->name;

        // indexへのリダイレクトURL(例:コントローラ名:Spots =>'/admin/spots/index')
        $this->redirectUrl = $redirectUrl = [
            // 'controller' => Inflector::tableize($this->controller->name),
            'controller'    => $this->request->controller,
            'action'        => 'index',
            'prefix'        => 'admin'
        ];
        $this->controller->set(compact("redirectUrl"));

        // controllerで設定されたsettingをマージする
        $defaultSettings = [
            'useConfirm' => true, // 確認画面を利用するか
            'inputPreview' => false, // 入力画面にプレビュー表示ボタンを置くか
            'inputBack' => true, // 入力画面に戻るボタンを置くか
        ];
        $this->privateSettings = !empty($this->controller->privateSettings) ? array_merge($defaultSettings, $this->controller->privateSettings) : $defaultSettings;
        $this->controller->set('privateSettings', $this->privateSettings);

        // formTokenあれば保持する
        $this->formToken = $this->request->data('formToken');
    }

    /**
     * beforeRender
     */
    public function beforeRender(Event $event)
    {
        // form値をset
        $this->controller->set('formEntity', $this->formEntity);
        $this->controller->set('formToken', $this->formToken);
    }

    /**
     * privateSettingの値を代入し、controller->setする
     * @param mixed $key
     * @param mixed $value
     */
    public function setSettingValue($key, $value)
    {
        $this->privateSettings[$key] = $value;
        $this->controller->set('privateSettings', $this->privateSettings);
    }

    /**
     * リファラーチェック
     *
     * @param string $action リンク元のアクション名
     * @return boolean
     */
    public function checkReferer($action)
    {
        $referer = explode('/', $this->controller->referer());
        return in_array($action, $referer);
    }

    /**
     * メール送信処理を実装誰か実装してくれ
     * @param string $subject 件名
     * @param array|string $to 宛先
     * @param array $data テンプレートで利用するデータ
     * @param string $template テンプレート名
     * @param string $config Config/email.phpのconfigの指定
     * @return array
     */
    public function sendMail($subject, $to, $data, $template, $config = 'default')
    {

        App::import('Behavior', 'Cbase.AdditionalValidation');
        $email = new CakeEmail($config);
        $email->helpers(['Html', 'Cbase.Private']);
        $email->emailPattern(AdditionalValidationBehavior::emailPattern);
        $email->subject($subject);
        $email->to($to);
        $email->template($template);
        $email->viewVars(['content' => $data]);
        return $email->send();

    }

    /**
     * 一覧に戻ってきた時にsessionからパラメータを格納する処理
     */
    public function setStoredQuery()
    {
        $currentQuery = $this->request->query;

        // Sessionのキーにprefixを付加することで表側と管理側等の混在を可能にする
        $prefix = !empty($this->request->prefix) ? $this->request->prefix : '';
        $sessionName = "{$prefix}_query";

        // backフラグがある場合は除外
        if (!empty($currentQuery['back'])) {
            unset($currentQuery['back']);
        }

        // 格納
        $this->request->session()->write($sessionName, $currentQuery);
    }

    /**
     * 一覧に戻ってきた時に格納したsessionパラメータを取得
     * @return mixed
     */
    public function getStoredQuery()
    {
        $prefix = !empty($this->request->prefix) ? $this->request->prefix : '';
        $sessionName = "{$prefix}_query";
        $query = $this->request->session()->read($sessionName);
        return !empty($query) ? $query : [];
    }

    /////////////////////////////////////////////////////////////////////////////////////////////////
    ///// controller action support /////////////////////////////////////////////////////////////////
    /////////////////////////////////////////////////////////////////////////////////////////////////

    /**
     * 公開、非公開の更新処理
     *
     * @param int|string $id
     * @param string $field
     */
    public function changePublic($id, $field = 'hid')
    {

        // データ取得（なければなにもしない）
        if (!$id || !$data = $this->controller->{$this->mainTable}->find('first', ['conditions' => ["{$this->mainTable}.id" => $id]])) {
            die('error');
        }

        // 公開非公開の変更
        $this->controller->{$this->mainTable}->set('id', $data[$this->mainTable]['id']);
        if (!$this->controller->{$this->mainTable}->saveField($field, ((empty($data[$this->mainTable][$field])) ? date('Y-m-d H:i:s') : null))) {
            die('error');
        }
        exit;
    }

    /**
     * index
     *
     * @param array $options
     */
    public function index($options = [])
    {
        // 編集画面等から戻ったときにsessionからqueryを復元する
        if (!empty($this->request->query['back'])) {
            $this->request->query = $this->getStoredQuery();
        }

        // 検索フォームの初期値のために$this->dataへ格納
        $this->request->data = $this->request->query;

        // 編集画面等から戻るためsessionにquery格納処理
        $this->setStoredQuery();

    }

    /**
     * show
     *
     * @param array $options
     */
    public function show($id)
    {
        $this->deleteConfirm($id);
    }

    /**
     * add
     *
     * @param array $options
     */
    public function add($options = [])
    {
        $defaults = [
            'render'    => true
        ];
        $options = array_merge($defaults, $options);

        // form値
        if (!$this->formEntity = $this->getStoredEntity()) {
            $this->formEntity = $this->controller->{$this->mainTable}->newEntity();
        }

        if ($options['render']) {
            $this->adminRender();
        }
    }

    /**
     * edit
     * ファイルアップある場合$file = trueにする
     *
     * @param int|string $id
     * @param array $options
     */
    public function edit($id = null, $options = [])
    {

        $defaults = [
            'render'    => true
        ];
        $options = array_merge($defaults, $options);

        // データ取得（なければ一覧へ）
        if (!$id || !$this->formEntity = $this->controller->{$this->mainTable}->get($id, ['contain' => $this->getAssociationNames()])) {
            $this->adminRedirect();
        }

        if ($options['render']) {
            // addのviewを使用
            $this->adminRender();
        }

    }

    /**
     * copy
     * ファイルアップある場合$file = trueにする
     * @param int|string $id
     * @param array $options
     * $options['text']
     *      array('field名' => '文言')をセットするとfield名のデータ文字列に「文言」を追加
     *      例) array('title' => 'のコピー')
     */
    public function copy($id = null, $options = [])
    {

        $defaults = [
            'file'      => false,
            'render'    => true,
            'text'      => []
        ];
        $options = array_merge($defaults, $options);

        // editと同様の処理だが、ここでrenderはしない
        $this->edit($id, ['render' => false]);

        // idをunsetし、新規登録状態にする
        unset($this->formEntity->id);
        $this->formEntity->isNew(true);
        // fileはtmp領域にコピーする
        if (method_exists($this->controller->{$this->mainTable}, 'copyTmpFile')) {
            $this->controller->{$this->mainTable}->copyTmpFile($this->formEntity);
        }

        // $options['text']を付ける
        if(!empty($options['text'])){
            foreach ($options['text'] as $field => $text) {
                if(isset($this->formEntity->{$field})){
                    $this->formEntity->{$field} .= $text;
                }
            }
        }

        // render
        if ($options['render']) {
            // addのviewを使用
            $this->adminRender();
        }

    }

    /**
     * confirm
     *
     * @param array $options
     * @return boolean validationのステータス
     */
    public function confirm($options = [])
    {

        $defaults = [
            'render'    => true,
        ];
        $options = array_merge($defaults, $options);

        // フォーム値がなければ一覧へ
        if (!$this->request->is(['patch', 'post', 'put'])) {
            $this->adminRedirect();
        }

        // set entity
        if (empty($this->request->data['id'])) {
            // new
            $this->formEntity = $this->controller->{$this->mainTable}->newEntity();
        } else {
            // edit
            if (!$this->formEntity = $this->controller->{$this->mainTable}->get($this->request->data['id'], ['contain' => $this->getAssociationNames()])) {
                $this->adminRedirect();
            }
        }
        $this->formEntity = $this->controller->{$this->mainTable}->patchEntity($this->formEntity, $this->request->data, ['assosiated' => $this->getAssociationNames()]);

        // バリデーションエラーがあれば、フォーム画面に戻し、そうでなければ確認画面表示
        if ($this->formEntity->errors()) {
            $this->adminRender('add');
        } else {
            $this->setConfirmForm();

            // セッション保持用のtoken生成し、現在のEntityをsessionに格納
            $this->setStoredEntity();

            if ($options['render']) {
                $this->adminRender();
            }
        }

        // validation通ったかどうかを返すように
        return !$this->formEntity->errors();

    }

    /**
     * complete
     *
     * @param array $options
     * @return mixed $options['redirect'] => falseを指定している場合は保存に成功するとtrueを返す。未設定状態ではreturnせずにリダイレクトする。
     */
    public function complete($options = [])
    {

        // フォーム値がなければ一覧へ
        if (!$this->request->is(['patch', 'post', 'put'])) {
            $this->adminRedirect();
            return false;
        }

        $defaults = [
            'redirect'  => true,
            'onSuccess' => null
        ];
        $options = array_merge($defaults, $options);

        if (!$this->privateSettings['useConfirm']) {
            if (!$this->confirm(['render' => false])) {
                return false;
            }
        }

        // set
        $this->formEntity = $this->getStoredEntity();
        $this->deleteStoredEntity();

        // 登録、ファイルリネーム
        $msg = '';
        if ($flag = $this->controller->{$this->mainTable}->save($this->formEntity)) {
            $msg = $this->controller->subject . '情報' . (empty($this->request->data['id']) ? '登録' : '編集') . 'が完了しました。';
        }else{
            $msg = $this->controller->subject . '情報' . (empty($this->request->data['id']) ? '登録' : '編集') . 'に失敗しました。後ほどお試しください。';
        }
        $this->Flash->set($msg);

        // onSuccess callback
        if ($flag && !empty($options['onSuccess'])) {
            $options['onSuccess']($this->formEntity);
        }

        // リダイレクト
        if ($options['redirect']) {
            $this->adminRedirect(true);
        } else {
            return $flag;
        }

    }

    /**
     * deleteConfirm
     *
     * @param int|string $id
     * @param array $options
     */
    public function deleteConfirm($id, $options = [])
    {

        $defaults = [
            'render'    => true
        ];
        $options = array_merge($defaults, $options);

        // データ取得
        if (!$this->formEntity = $this->controller->{$this->mainTable}->get($id, ['contain' => $this->getAssociationNames()])) {
            $this->adminRedirect();
        }

        // 確認画面のビューを使用
        $this->setConfirmForm();
        if ($options['render']) {
            $this->adminRender();
        }

    }

    /**
     * deleteComplete
     *
     * @param int|string $id
     * @param array $options
     * @return mixed $options['redirect'] => falseを指定している場合は削除に成功するとtrueを返す。未設定状態ではreturnせずにリダイレクトする。
     */
    public function deleteComplete($id = null, $options = [])
    {
        $defaults = [
            'redirect'  => true
        ];
        $options = array_merge($defaults, $options);

        // リファラーチェック
        if (
            (!$this->privateSettings['useConfirm'] && !$this->checkReferer('index'))
            && ($this->privateSettings['useConfirm'] && !$this->checkReferer('deleteConfirm'))
        ) $this->adminRedirect();

        if ($this->privateSettings['useConfirm']) $id = $this->request->data['id'];

        // deletedに現在日時を格納
        if ($flag = $this->controller->{$this->mainTable}->softDelete($id)){
            $this->Flash->set("{$this->controller->subject}削除完了しました。");
        } else {
            $this->Flash->set("{$this->controller->subject}削除に失敗しました。後ほどお試しください。");
        }

        // リダイレクト
        return $options['redirect'] ? $this->adminRedirect(true) : $flag;
    }

    /**
     * render
     * @param string $name 'add' or 'confirm'
     *  nullの場合action名で判断
     *      action = add, edit  ====> add
     *      action = confirm, deleteConfirm  ====> confirm
     * @return mixed｜void renderできない場合はfalseが返る
     */
    public function adminRender($name = null)
    {
        if (empty($name)) {
            $action = $this->request->action;
            if(in_array($action, ['add', 'edit', 'copy'])) $name = 'add';
            elseif(in_array($action, ['confirm', 'deleteConfirm', 'show'])) $name = 'confirm';
            else return false;
        }

        // "Template/Admin/Controller名"ディレクトリに用意されているviewのリストを求める
        $viewFiles = [];
        // 該当のviewディレクトリパス(pluginかどうかで異なる)
        $viewDirPath = '';
        if (empty($this->request->params['plugin'])) {
            // プラグインなし
            $viewDirPath = APP . 'Template' . DS . 'Admin' . DS . $this->controller->name;
        } else {
            // プラグイン
            $viewDirPath = ROOT . 'plugins' . DS . Inflector::camelize($this->request->params['plugin']) . DS . 'src' . DS . 'Template' . DS . $this->controller->name;
        }
        if (file_exists($viewDirPath) && $dir = opendir($viewDirPath)) {
            while(($file = readdir($dir)) !== false){
                if($file != "." && $file != ".."){
                    $viewFiles[] = str_replace('.ctp', '', $file);
                }
            }
            closedir($dir);
        }

        // "Template/Admin/Controller名"ディレクトリ内で用意されていたらそちらを使う
        if (in_array($name, $viewFiles)) {
            $this->controller->render($name);
        } else {
            // 用意されていない場合で、"Template/Admin/Element"ディレクトリに用意していればそちらを使う、なければCbaseのファイルを使う
            $elementsFiles = [];
            $elementsDirPath = APP . 'Template' . DS . 'Admin' . DS . 'Element';
            if (file_exists($elementsDirPath) && $dir = opendir($elementsDirPath)) {
                while(($file = readdir($dir)) !== false){
                    if($file != "." && $file != ".."){
                        $elementsFiles[] = str_replace('.ctp', '', $file);
                    }
                }
                closedir($dir);
            }

            if (in_array($name, $elementsFiles)) {
                // elements内のファイル利用
                $this->controller->render("/../Template/Admin/Element/{$name}");
            } else {
                // cbase plugin内のファイル利用
                $this->controller->render("Cbase./Admin/Cbase/{$name}");
            }
        }

    }

    /**
     * indexにリダイレクトする
     * @param boolean $back trueだとurlにback:1を付加
     */
    public function adminRedirect($back = false)
    {
        ($back) ? $this->controller->redirect($this->redirectUrl + ['back' => true]) : $this->controller->redirect($this->redirectUrl);
    }

    /**
     * 確認画面表示フラグをセット
     */
    public function setConfirmForm()
    {
        $this->controller->set('confirmForm', true);
    }

    /**
     * containのためのmainmodelのassociationを取得
     */
    public function getAssociationNames()
    {
        $keys = $this->controller->{$this->mainTable}->associations()->keys();
        $assocNames = [];
        foreach ($keys as $key) {
            $assoc = $this->controller->{$this->mainTable}->association($key);
            $class = get_class($assoc);
            if (in_array($class, ['Cake\ORM\Association\BelongsTo', 'Cake\ORM\Association\BelongsToMany', 'Cake\ORM\Association\HasMany', 'Cake\ORM\Association\HasOne'])) {
                $assocNames[] = $assoc->name();
            }
        }
        return $assocNames;
    }

    /**
     * form操作時、画面持ち回りのためSessionに保持しているentityがあれば取得
     */
    public function getStoredEntity()
    {
        // form値
        if ($this->formToken) {
            return $this->request->session()->read("form.{$this->mainTable}.{$this->formToken}");
        } else {
            return null;
        }
    }

    /**
     * form操作時、画面持ち回りのためにSessionにentityを保存
     */
    public function setStoredEntity()
    {
        $this->formToken = Text::uuid();
        $this->request->session()->write("form.{$this->mainTable}.{$this->formToken}", $this->formEntity);
    }

    /**
     * form操作時、画面持ち回りのために保持したSessionを削除
     */
    public function deleteStoredEntity()
    {
        $this->request->session()->delete("form.{$this->mainTable}.{$this->formToken}");
    }

}

