<?php
namespace Cbase\Model\Validation;
use Cake\Validation\Validation;

class CbaseValidation extends Validation
{

    /**
     * @var string emailJPのチェックする正規表現
     */
    const emailPattern = '/(\\A(?:^([a-z0-9][a-z0-9_\\-\\.\\+\\?]*)@([a-z0-9][a-z0-9\\.\\-]{0,63}\\.(com|org|net|biz|info|name|net|pro|aero|coop|museum|[a-z]{2,4}))$)\\z)|(^[a-z0-9\?\.\/_-]{3,30}@(?:[a-z0-9][-a-z0-9]*\.)*(?:[a-z0-9][-a-z0-9]{0,62})\.(?:(?:[a-z]{2}\.)?[a-z]{2,4}|museum|travel)$)/i';

    ////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
    // 文字の種類関連のチェック
    // ひらがな、カタカナ、半角英数、数値、スペースのみ
    ////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

    /**
     * 全角ひらがなのみ
     * 全角ダッシュ「ー」のみ必要と考えられるので追加
     * @return boolean
     */
    public static function hiragana($value, $context)
    {
        return (bool) preg_match("/^(\xe3(\x81[\x81-\xbf]|\x82[\x80-\x93]|\x83\xbc))*$/", $value);
    }

    /**
     * 全角カタカナのみ
     * @return boolean
     */
    public static function katakana($value, $context)
    {
        //\xe3\x82\x9b 濁点゛
        //\xe3\x82\x9c 半濁点゜
        return (bool) preg_match("/^(\xe3(\x82[\xa1-\xbf]|\x83[\x80-\xb6]|\x83\xbc|\x82\x9b|\x82\x9c))*$/", $value);
    }

    /**
     * 全角ひらがな,ー，スペースのみ
     * @return boolean
     */
    public static function hiraganaPlus($value, $context)
    {
        return (bool) preg_match('/^([ぁ-ん]|[　 ]|[ー])*$/u', $value);
    }

    /**
     * 全角カタカナ,ー，スペースのみ
     * @return boolean
     */
    public static function katakanaPlus($value, $context)
    {
        return (bool) preg_match('/^([ァ-ヶ]|[　 ]|[ー])*$/u', $value);
    }

    /**
     * 半角英数のみ 0-9, a-z , A-Z
     * @return boolean
     */
    public static function alphaNumber($value, $context)
    {
        return (bool) preg_match("/^[a-zA-Z0-9]*$/", $value);
    }

    /**
     * 半角英数、記号の一部のみ 0-9, a-z , A-Z, -_
     * @return boolean
     */
    public static function alphaNumberPlus($value, $context)
    {
        return (bool) preg_match("/^[a-zA-Z0-9\-\_]*$/", $value);
    }

    /**
     * 数値のみ
     * @return boolean
     */
    public static function number($value, $context)
    {
        return (bool) preg_match( "/^[-+]?\\b[0-9]*\\.?[0-9]+\\b$/", $value);
    }

    ////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
    // 文字の形式関連のチェック
    // 電話番号、郵便番号、EMAIL、文字数、数字の範囲
    ////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
    /**
     * 電話またはFAX
     * @return boolean
     */
    public static function tel($value, $context) {
        $pattern = '/^(0\d{1,4}[\s-]?\d{1,4}[\s-]?\d{1,4}|\+\d{1,3}[\s-]?\d{1,4}[\s-]?\d{1,4}[\s-]?\d{1,4})$/';
        return (bool) preg_match($pattern, $value);
    }

    /**
     * 3つのtel入力を見て、電話番号として正しいかチェック
     *
     * @param Model &$model
     * @param mixed $value
     * @param array $fields フィールド名
     * @return boolean
     */
    public static function telThree($value, $fields, $context) {
        // ない場合OK
        if(empty($context['data'][$fields[0]]) || empty($context['data'][$fields[1]]) || empty($context['data'][$fields[2]])) return true;

        // チェック
        $tel = "{$context['data'][$fields[0]]}{$context['data'][$fields[1]]}{$context['data'][$fields[2]]}";
        return self::tel($tel, $context);
    }

    /**
     * 3つのtel入力を見て、空ならエラー
     *
     * @return boolean
     */
    public static function telThreeEmpty($value, $fields, $context) {
        return !empty($context['data'][$fields[0]]) && !empty($context['data'][$fields[1]]) && !empty($context['data'][$fields[2]]);
    }

    /**
     * 郵便番号
     * @return boolean
     */
    public static function zipCode($value, $context) {
        $pattern = '/^[0-9]{3}-?[0-9]{4}$/';
        return (bool) preg_match($pattern, $value);
    }

    /**
     * 2つのzipcode入力を見て、郵便番号として正しいかチェック
     *
     * @return boolean
     */
    public static function zipCodeTwo($value, $fields, $context) {
        // ない場合OK
        if(empty($context['data'][$fields[0]]) || empty($context['data'][$fields[1]])) return true;

        // チェック
        $zipCode = $context['data'][$fields[0]] . '-' . $context['data'][$fields[1]];
        return self::zipCode($zipCode, $context);
    }

    /**
     * 2つのzipcode入力を見て、空ならエラー
     *
     * @return boolean
     */
    public static function zipCodeTwoEmpty($value, $fields, $context) {
        return !empty($context['data'][$fields[0]]) && !empty($context['data'][$fields[1]]);
    }

    /**
     * Email(RFC準拠＆docomo,auの非準拠対応)
     * @return boolean
     */
    public static function emailJP($value, $context) {
        $pattern = self::emailPattern;
        return (bool) preg_match($pattern, $value);
    }

    /**
     * 全角対応文字数チェック(最大文字数)
     *
     * @return boolean
     */
    public static function maxlengthJP($value, $length, $context) {
        return (bool) (mb_strlen($value) <= $length);
    }

    /**
     * 全角対応文字数チェック(最小文字数)
     *
     * @return boolean
     */
    public static function minlengthJP($value, $length, $context) {
        return (bool) (mb_strlen($value) >= $length);
    }

    /**
     * 文字数が指定の範囲か
     * @return boolean
     */
    public static function betweenJP($value, $low, $high, $context) {
        return (mb_strlen($value) >= $low && mb_strlen($value) <= $high);
    }

    /**
     * 数字が指定の範囲か
     * @return boolean
     */
    public static function betweenNumber($value, $low, $high, $context) {
        if(!self::number($value, $context)) return false;
        $f = (float) $value;
        return ($f >= $low && $f <= $high);
    }

    ////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
    // ファイルアップロード関連チェック
    // 画像形式、ファイルサイズ
    ////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

    /**
     * Validation::uploadedFile()のラッパー
     * dataURLでアップされたものが必ずエラーになるので
     */
    public static function uploadedFile($file, array $options = [])
    {
        $options += [
            'minSize' => null,
            'maxSize' => null,
            'types' => null,
            'optional' => false,
        ];
        if (!is_array($file)) {
            return false;
        }
        $keys = ['error', 'name', 'size', 'tmp_name', 'type'];
        ksort($file);
        if (array_keys($file) != $keys) {
            return false;
        }
        if (!static::uploadError($file, $options['optional'])) {
            return false;
        }
        if ($options['optional'] && (int)$file['error'] === UPLOAD_ERR_NO_FILE) {
            return true;
        }
        if (isset($options['minSize']) && !static::fileSize($file, '>=', $options['minSize'])) {
            return false;
        }
        if (isset($options['maxSize']) && !static::fileSize($file, '<=', $options['maxSize'])) {
            return false;
        }
        if (isset($options['types']) && !static::mimeType($file, $options['types'])) {
            return false;
        }

        // dataURLでアップされたものが必ずfalseになるので
        // return is_uploaded_file($file['tmp_name']);
        return true;
    }

    // /**
    //  * アップロードするファイル形式がjpg,gif,pngであるかチェック
    //  * @param Model &$model
    //  * @param array $data
    //  * @return boolean
    //  */
    // public static function isImage(&$model, $data){
    //     $type = array('image/jpeg', 'image/gif', 'image/pjpeg', 'image/png', 'image/x-png');

    //     //$dataが２次元配列かを調べ、そうでなければ２次元配列にする
    //     foreach($data as $val){
    //         if(!is_array($val)){
    //             $data = array($data);
    //             break;
    //         }
    //     }

    //     foreach($data as $val){
    //         if(!empty($val['type'])){
    //             return in_array($val['type'], $type);
    //         }else{
    //             return true;
    //         }
    //     }
    // }

    // /**
    //  * アップロードするファイル形式がmp3, aac, m4a, wavであるかチェック
    //  * @param Model &$model
    //  * @param array $data
    //  * @return boolean
    //  */
    // public static function isAudio(&$model, $data){
    //     $type = array(
    //         'audio/mp3',
    //         'audio/mpeg',
    //         'audio/mpg',
    //         'audio/wav',
    //         'audio/x-mp3',
    //         'audio/x-mpeg',
    //         'audio/x-mpg',
    //         'audio/x-wav',
    //         'audio/aac',
    //         'audio/x-aac',
    //         'audio/x-m4a'
    //     );

    //     //$dataが２次元配列かを調べ、そうでなければ２次元配列にする
    //     foreach($data as $val){
    //         if(!is_array($val)){
    //             $data = array($data);
    //             break;
    //         }
    //     }

    //     foreach($data as $val){
    //         if(!empty($val['type'])){
    //             return in_array($val['type'], $type);
    //         }else{
    //             return true;
    //         }
    //     }
    // }

    // /**
    //  * 画像のサイズが指定サイズかをチェック
    //  *
    //  * @param Model &$model
    //  * @param array $data
    //  * @param int $size
    //  * @return boolean
    //  */
    // // public static function imageSize(&$model, $data, $size){
    // //     // 画像かどうかのチェック:画像でなければチェックしない
    // //     if(!$this->isImage($model, $data)) return true;

    // //     //$dataが２次元配列かを調べ、そうでなければ２次元配列にする
    // //     foreach($data as $val){
    // //         if(!is_array($val)){
    // //             $data = array($data);
    // //             break;
    // //         }
    // //     }

    // //     foreach($data as $val){
    // //         if(!empty($val['tmp_name'])){
    // //             // 実際の画像のw,h
    // //             list($w, $h) = getimagesize($val['tmp_name']);
    // //             $wTrue = empty($size[0]);
    // //             $hTrue = empty($size[1]);
    // //             if(!empty($size[0])) $wTrue = ($w == $size[0]);
    // //             if(!empty($size[1])) $hTrue = ($h == $size[1]);
    // //             return ($wTrue && $hTrue);
    // //         }else{
    // //             return true;
    // //         }
    // //     }
    // // }

    // /**
    //  * アップロードするファイル形式が以下のドキュメントであるかチェック
    //  * pdf,doc/docx,xls/xlsx,ppt/pptx,txt,zip,lzh
    //  * @param Model &$model
    //  * @param array $data
    //  * @return boolean
    //  */
    // public static function isFile(&$model, $data){
    //     $type = array(
    //         // PDF
    //         'application/pdf',
    //         // doc/docx
    //         'application/msword',
    //         'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
    //         // xls/xlsx
    //         'application/vnd.ms-excel',
    //         'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
    //         // ppt/pptx
    //         'application/vnd.ms-powerpoint',
    //         'application/vnd.openxmlformats-officedocument.presentationml.presentation',
    //         // txt
    //         'text/plain',
    //         // zip
    //         'application/zip',
    //         // lzh
    //         'application/lzh'
    //     );

    //     //$dataが２次元配列かを調べ、そうでなければ２次元配列にする
    //     foreach($data as $val){
    //         if(!is_array($val)){
    //             $data = array($data);
    //             break;
    //         }
    //     }

    //     foreach($data as $val){
    //         if(!empty($val['type'])){
    //             return in_array($val['type'], $type);
    //         }else{
    //             return true;
    //         }
    //     }
    // }

    // /**
    //  * アップロードするファイル形式が指定したmimetypeかチェック
    //  * @param Model &$model
    //  * @param array $data
    //  * @param array $type 受け入れるMimeTypeの配列
    //  * @return boolean
    //  */
    // public static function fileType(&$model, $data, $type){
    //     //$dataが２次元配列かを調べ、そうでなければ２次元配列にする
    //     foreach($data as $val){
    //         if(!is_array($val)){
    //             $data = array($data);
    //             break;
    //         }
    //     }

    //     foreach($data as $val){
    //         if(!empty($val['type'])){
    //             return in_array($val['type'], $type);
    //         }else{
    //             return true;
    //         }
    //     }
    // }

    // /**
    //  * ファイルのnullチェック
    //  * @param Model &$model
    //  * @param array $data
    //  * @return boolean
    //  */
    // public static function emptyFile(&$model, $data){

    //     //$dataが２次元配列かを調べ、そうでなければ２次元配列にする
    //     foreach($data as $key => $val){
    //         $field = $key;
    //         if(!is_array($val)){
    //             $data = array($data);
    //             break;
    //         }
    //     }

    //     foreach($data as $val){
    //         if(!empty($val['size'])){
    //             return ($val['size'] > 0);
    //         }else{
    //             // FileComponent依存部分

    //             // validation対象のrequst dataしか取得できないので,無理やり取得
    //             $Request = new CakeRequest();

    //             // hasmanyのデータなのか、そうでないのかを見極める
    //             // $Request->data[$model->name]のキーにfield名がなければhasMany
    //             $isHasMany = !in_array($field, array_keys($Request->data[$model->name]), true);

    //             if (!$isHasMany) {
    //                 // アップして保持している,かつ,このファイルを削除にチェックしていない
    //                 return (!empty($model->data[TMP_FILES . $model->name][$field]) && empty($model->data[DELETE_FILES . $model->name][$field]));
    //             } else {
    //                 // hasManyの場合、現在validationしているのが何番目のrequest dataかを見極める
    //                 $targetKey = null;
    //                 foreach ($Request->data[$model->name] as $reqKey => $reqVal) {
    //                     if ($reqVal == $model->data[$model->name]) {
    //                         $targetKey = $reqKey;
    //                         break;
    //                     }
    //                 }
    //                 if ($targetKey === null) return false;

    //                 // アップして保持している,かつ,このファイルを削除にチェックしていない
    //                 return (!empty($Request->data[TMP_FILES . $model->name][$targetKey][$field]) && empty($Request->data[DELETE_FILES . $model->name][$targetKey][$field]));
    //             }
    //         }
    //     }

    // }

    // /**
    //  * ファイルサイズが指定された容量以内かのチェック
    //  * @param Model $model
    //  * @param array $data
    //  * @param int $max
    //  * @return boolean
    //  */
    // public static function maxFileSize(&$model, $data, $max){

    //     //$dataが２次元配列かを調べ、そうでなければ２次元配列にする
    //     foreach($data as $val){
    //         if(!is_array($val)){
    //             $data = array($data);
    //             break;
    //         }
    //     }
    //     $error = array(1, 2, 3);
    //     foreach($data as $val){
    //         if(!empty($val['size'])){
    //             return ($val['size'] <= $max);
    //         }elseif(!empty($val['error']) && in_array($val['error'], $error)){
    //             return false;
    //         }else{
    //             return true;
    //         }
    //     }
    // }

    ////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
    // その他チェック
    ////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
    /**
     * 1つのチェックボックスの必須チエック
     * 値が1以外はfalseを返す
     *
     * @param Model $model
     * @param array $field
     * @return boolean Success
     */
    public static function checkAgree($value, $context) {
        return ($value == 1);
    }

    /**
     * 日付のfrom <= toかどうかをチェック
     * @param Model $model
     * @param array $value
     * @param array $fields = array(fromFieldName, toFieldName)
     * @return boolean
     */
    public static function compareFromTo(&$model, $value, $fields = array()) {
        // どちらかのデータがなければtrue
        if (!$model->data[$model->name][$fields[0]] || !$model->data[$model->name][$fields[1]]) return true;

        $from = strtotime($model->data[$model->name][$fields[0]]);
        $to   = strtotime($model->data[$model->name][$fields[1]]);

        return $from <= $to;
    }

    /**
     * 指定フィールドのどれかが入力されている場合に指定フィールド全て入力されているかをチェック
     * FIXME input type fileの場合の対応をしてない
     * @param Model $model
     * @param array $value
     * @param array $fields = array(field1, field2, ....)
     * @param array $options required=trueにすると指定フィールド全て必須
     * @return boolean
     */
    public static function anyFields(&$model, $value, $fields = array(), $options = array()){
        $nullCnt = 0;
        $notNullCnt = 0;

        foreach($fields as $field){
//          if(is_array($model->data[$model->name][$field])){
//              // input type fileの場合は空でも配列が入っているため
//              (empty($model->data[$model->name][$field]['name']) && empty($model->data[TMP_FILES.$model->name][$field])) ? $nullCnt++ : $notNullCnt++;
//          }else{
                (empty($model->data[$model->name][$field])) ? $nullCnt++ : $notNullCnt++;
//          }
        }

        if($options['required']){
            return $nullCnt === 0;
        }else{
            return (($notNullCnt === 0) || ($notNullCnt > 0 && $nullCnt === 0));
        }
    }

    /**
     * 該当フィールドと指定フィールド、いずれかを必須入力としてチェック
     * @param Model &$model
     * @param array $wordvalue1
     * @param string $fieldname2
     * @return boolean
     */
    public static function checkEither(&$model, $wordvalue1, $fieldname2) {
        $fieldname1 = key($wordvalue1);
        return ($model->data[$model->alias][$fieldname1] !='' || $model->data[$model->alias][$fieldname2] !='');
    }


    //メソッドはstaticである必要あり
    // public static function zip1($value, $context) {
    //     //きちんとboolで返さないとエラーになる
    //     return (bool) preg_match('/^[0-9]{3}$/', $value);
    // }
}
