<?php
namespace Cbase\Model\Behavior;

use ArrayObject;
use Cake\Event\Event;
use Cake\ORM\Behavior;
use Cake\ORM\Entity;
use Cake\ORM\Table;
use Cake\Utility\Text;
use finfo;

/**
 * File behavior
 *
 * @author ueno
 */
class FileBehavior extends Behavior
{

    /**
     * Default configuration.
     *
     * @var array
     */
    protected $_defaultConfig = [
        // fileがアップロードされるフィールド
        'targetFields' => [],
        // 受け入れるMimeType(これ以外の拡張子は処理を行わない)
        'acceptMimeTypes' => [
            // 画像
            'image/jpeg', 'image/gif', 'image/pjpeg', 'image/png', 'image/x-png',
            // 音声
            'audio/mp3', 'audio/mpeg', 'audio/mpg', 'audio/wav', 'audio/x-mp3', 'audio/x-mpeg', 'audio/x-mpg', 'audio/x-wav', 'audio/aac', 'audio/x-aac', 'audio/x-m4a',
            // 動画
            'video/mpeg', 'video/mp4', 'video/x-m4v', 'video/webm', 'video/ogg', 'video/quicktime', 'video/x-msvideo',
            // PDF, text
            'application/pdf', 'text/plain',
            // doc/docx, xls/xlsx, ppt/pptx
            'application/msword', 'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
            'application/vnd.ms-excel', 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
            'application/vnd.ms-powerpoint', 'application/vnd.openxmlformats-officedocument.presentationml.presentation',
            // archive
            'application/zip', 'application/lzh'
        ],
        // 除外する拡張子(※アップロードは受け入れるが、拡張子は付けずに配置する)
        'ignoreExtensions' => ['exe', 'php', 'cgi', 'php5', 'php4', 'php7', 'sh', 'pl', 'cgi', 'rb', 'py'],
        // ウェブルートまでのファイルパス(末尾に/つける)
        'webrootPath' => null,
        // ファイルが最終的にアップロードされるディレクトリのドキュメントルートからのパス(先頭に/つけない) デフォルトは"files/"
        'filesDirectory' => null,
        // ファイルが一時的にアップロードされるディレクトリのドキュメントルートからのパス(先頭に/つけない) デフォルトは"files/temp/"
        'tmpDirectory' => null,
        // ファイル削除時に指定するチェックボックスフィールドのサフィックス
        'shouldDeleteSuffix' => '_shouldDelete',
        // ファイルアップロードを受け入れるのフィールドのサフィックス
        'fileFieldSuffix' => '_file',
        // callback: ファイル削除時
        'afterDeleteFile' => null,
        // callback: ファイル登録し正規の場所に配置時
        'afterSaveFile' => null,
    ];

    public function initialize(array $config)
    {
        //nullが指定されているフィールドにデフォルト値を書き込む
        if (!isset($config['webrootPath']) || is_null($config['webrootPath'])) {
            $config['webrootPath'] = WWW_ROOT;
            $this->config('webrootPath', $config['webrootPath'], false);
        }

        if (!isset($config['tmpDirectory']) || is_null($config['tmpDirectory'])) {
            $config['tmpDirectory'] = 'files' . DS . 'tmp' . DS;
            $this->config('tmpDirectory', $config['tmpDirectory'], false);
        }
        if (!isset($config['filesDirectory']) || is_null($config['filesDirectory'])) {
            $config['filesDirectory'] = 'files' . DS;
            $this->config('filesDirectory', $config['filesDirectory'], false);
        }

        //アップロードするディレクトリの存在を確認する
        if (!is_dir($this->config('webrootPath') . $this->config('filesDirectory'))) {
            mkdir($this->config('webrootPath') . $this->config('filesDirectory'));
        }
        if (!is_dir($this->config('webrootPath') . $this->config('tmpDirectory'))) {
            mkdir($this->config('webrootPath') . $this->config('tmpDirectory'));
        }

    }

    /**
     * beforeMarshal callbackメソッド
     * $this->request->dataのarrayからEntityへの変換前に呼び出される
     * @param Event $event
     * @param ArrayObject $data オブジェクトなので内部をいじると呼び出し元に反映される
     * @param ArrayObject $options
     */
    public function beforeMarshal(Event $event, ArrayObject $data, ArrayObject $options)
    {

        foreach ($this->config('targetFields') as $fieldName) {

            //接尾辞のつかないフィールド名
            $originalFieldName = $fieldName;

            if ($this->config('fileFieldSuffix') !== null) {
                $fieldName .= $this->config('fileFieldSuffix');
            }

            //ファイルがアップロードされたかどうか確認
            if (!empty($data[$fieldName]['tmp_name']) || (!empty($data[$fieldName]) && is_string($data[$fieldName]))) {
                // dataURI stringでアップロードされたかどうか(ajax画像)
                $isUri = (!empty($data[$fieldName]) && is_string($data[$fieldName]));

                $tmpName = '';
                $fileName = '';
                $fileType = '';
                $extension = '';
                if ($isUri) {
                    // dataURI stringでアップロードされた場合(ajax画像)
                    list($tmpName, $fileType, $extension) = $this->_dataUriToFile($data[$fieldName]);
                    $fileName = basename($tmpName);
                } else {
                    // 通常時
                    $tmpName = $data[$fieldName]['tmp_name'];
                    $fileName = $data[$fieldName]['name'];

                    // MimeTypeの検証
                    $finfo = new finfo(FILEINFO_MIME_TYPE);
                    $fileType = $finfo->file($tmpName);
                    $extension = preg_filter('{^.*\.([^\.]+)$}', '$1', $fileName);
                }

                if (in_array($fileType, $this->config('acceptMimeTypes'))) {
                    //正しいファイルだと検証された

                    // tmp保存ファイル名とパス
                    $tmpDestName = $this->_generateTmpName($fieldName, $extension);
                    $tmpDestPath = $this->config('webrootPath') . $tmpDestName;

                    if ($this->_moveUploadedFile($tmpName, $tmpDestPath, $isUri)) {
                        chmod($tmpDestPath, 0777);
                        // moveに成功
                        // もともと消していたが、validationできないのでそのままにしておく
                        // unset($data[$fieldName]);

                        // validationチェックし、エラーがある場合何もしない
                        if ($isUri) {
                            // dataURLの場合validationのために通常のアップロードの形にする
                            $data[$fieldName] = [
                                'name' => basename($tmpDestPath),
                                'type' => $fileType,
                                'tmp_name' => $tmpDestPath,
                                'error' => 0,
                                'size' => filesize($tmpDestPath),
                            ];
                        }
                        // 対象のデータだけをvalidationする
                        $validationTargetData = [
                            $fieldName => $data[$fieldName]
                        ];
                        $validationErrors = $this->_table->validator()->errors($validationTargetData);
                        if (empty($validationErrors[$fieldName])) {
                            // validationエラーでない場合
                            // tempNameをフィールドに入れる
                            $data[$originalFieldName] = $tmpDestName;
                        }

                    }

                } else {
                    //MimeTypeの検証に失敗した。フィールドごとunset
                    if (isset($data[$fieldName])) {
                        unset($data[$fieldName]);
                    }
                    if (isset($data[$originalFieldName])) {
                        unset($data[$originalFieldName]);
                    }
                }

            } else {
                //ファイルがアップロードされていない場合
                //$fieldName . '_shouldDelete'が存在してtrueなら当該フィールドにnullを入れる。
                //$fieldName . '_shouldDelete'はunset

                $deleteFieldName = $originalFieldName . $this->config('shouldDeleteSuffix');
                if (isset($data[$deleteFieldName]) && $data[$deleteFieldName]) {

                    $data[$originalFieldName] = '';
                    unset($data[$deleteFieldName]);

                }

            }

        }
    }

    /**
     * afterSave
     * 画像ファイルを正規の場所に保存し、field値を書き換える
     *
     * @param Event $event
     * @param Entity $entity
     * @param ArrayObject $options
     * @return bool
     */
    public function afterSave(Event $event, Entity $entity, ArrayObject $options)
    {
        //entityの指定フィールドにアクセス
        $isUpdate = false;
        foreach ($this->config('targetFields') as $fieldName) {
            if ($entity->has($fieldName)) {

                //指定のフィールドがdirtyかどうか
                if ($entity->dirty($fieldName)) {
                    //dirtyなら以前のファイルは削除する

                    //変更前のデータを読み出す。今回の変更が初回変更の場合は今回の変更分が出てくるのであとで除外する
                    $original = $entity->getOriginal($fieldName);

                    if ($original !== null
                        && $original !== $entity->get($fieldName)
                        && is_file($path = $this->config('webrootPath') . $original)
                    ) {
                        //Originalのvalueを読む、webrootPath内に当該ファイルが存在すれば削除する
                        unlink($path);
                        // callback
                        if (!empty($this->config('afterDeleteFile'))) {
                            $func = $this->config('afterDeleteFile');
                            $func($fieldName, $path);
                        }
                    }

                }

                //新しいデータをよむ
                $fieldData = $entity->get($fieldName);

                if (strpos($fieldData, $this->config('tmpDirectory')) === 0) {
                    //files/temp/から始まるパスを含んでいるときは、tempファイルを持っていると判断する

                    $tmpPath = $this->config('webrootPath') . $fieldData;
                    //fileの存在確認
                    if (is_file($tmpPath)) {
                        //新しいファイル名を生成する
                        $fileHash = sha1_file($tmpPath);
                        $extension = pathinfo($tmpPath, PATHINFO_EXTENSION);

                        // 新しいファイル名とパス
                        $newFileName = $this->_generateCompName($fieldName, $entity->get('id'), $fileHash, $extension);
                        $newFilePath = $this->config('webrootPath') . $newFileName;

                        if (rename($tmpPath, $newFilePath)) {
                            chmod($newFilePath, 0777);
                            $entity->set($fieldName, $newFileName);
                            $isUpdate = true;
                            // callback
                            if (!empty($this->config('afterSaveFile'))) {
                                $func = $this->config('afterSaveFile');
                                $func($fieldName, $newFilePath);
                            }
                        }
                    }
                }
            }

        }

        if ($isUpdate) {
            return $this->_table->save($entity);
        }
        return true;
    }

    /**
     * afterDelete
     */
    public function afterDelete(Event $event, Entity $entity, ArrayObject $options)
    {
        //entityの指定フィールドにアクセス
        foreach ($this->config('targetFields') as $fieldName) {
            if ($entity->has($fieldName)) {
                if (is_file($path = $this->config('webrootPath') . $entity->get($fieldName))) {
                    unlink($path);
                    // callback
                    if (!empty($this->config('afterDeleteFile'))) {
                        $func = $this->config('afterDeleteFile');
                        $func($fieldName, $path);
                    }
                }
            }
        }
        return true;
    }

    /**
     * ファイルをtmpディレクトリにコピーする
     * 複製処理用
     */
    public function copyTmpFile(Entity $entity)
    {
        foreach ($this->config('targetFields') as $fieldName) {
            if ($entity->has($fieldName)) {
                // コピー元のファイルパス
                $filename = $entity->get($fieldName);
                $currentPath = $path = $this->config('webrootPath') . $filename;
                if (is_file($currentPath)) {
                    // コピー先パスを求める
                    $extension = pathinfo($filename, PATHINFO_EXTENSION);
                    $tmpDestName = $this->_generateTmpName($fieldName, $extension);
                    $tmpDestPath = $this->config('webrootPath') . $tmpDestName;
                    // コピーする
                    if (copy($currentPath, $tmpDestPath)) {
                        chmod($tmpDestPath, 0777);

                        // entityに格納
                        $entity->{$fieldName} = $tmpDestName;
                    }
                }
            }
        }
        return true;
    }

    /**
     * アップロードファイルを正しい位置に配置
     */
    private function _moveUploadedFile($src, $dest, $isUri) {
        if (!$isUri) {
            // validation時のタイミングでファイルが/tmp/からなくなり
            // 例外発生するのでmove_uploaded_fileは使わない
            // return move_uploaded_file($src, $dest);
            return copy($src, $dest) && chmod($dest, 0777);;
        } else {
            return rename($src, $dest);
        }
    }

    /**
     * tmp保存用のファイル名を生成
     */
    private function _generateTmpName($fieldName, $extension) {
        $name = $this->config('tmpDirectory') . "{$this->_table->alias()}_{$fieldName}_" . md5(microtime(true) . '_' . rand(000, 999));
        if (!empty($extension) && !in_array($extension, $this->config('ignoreExtensions'))) {
            //安全な拡張子なら付与する
            $name .= ".{$extension}";
        }
        return $name;
    }

    /**
     * 保存用のファイル名を生成
     */
    private function _generateCompName($fieldName, $id, $hash, $extension) {
        $dir = $this->config('filesDirectory');
        $tableName = $this->_table->alias();
        $idDir = floor($id / 1000);
        // files/モデル名/idを1000で割った整数値(0～999は0、1000～1999は1)/ファイル名
        if (!is_dir("{$dir}{$tableName}")) {
            mkdir("{$dir}{$tableName}", 0777);
            chmod("{$dir}{$tableName}", 0777);
        }
        if (!is_dir("{$dir}{$tableName}" . DS . $idDir)) {
            mkdir("{$dir}{$tableName}" . DS . $idDir, 0777);
            chmod("{$dir}{$tableName}" . DS . $idDir, 0777);
        }
        $dir = "{$dir}{$tableName}" . DS . $idDir . DS;

        $name = "{$dir}{$tableName}_{$id}_{$fieldName}_{$hash}";
        if (!empty($extension)) {
            $name .= ".{$extension}";
        }
        return $name;
    }

    /**
     * dataURI文字列をファイルに変換
     * 一旦$this->config('tmpDirectory')に適当な名前で置く
     * @return array [path, mime type, extension]
     */
    private function _dataUriToFile($dataURL) {
        // mime typeを抜き出す
        $mime = null;
        preg_match("/data:[^,]+;/i", $dataURL, $match);
        if(!empty($match[0])) $mime = preg_replace("(data:|;)", '', $match[0]);
        if(empty($mime)) die(); // エラー

        // ヘッダに「data:image/png;base64,」が付いているので、それは外す
        $canvas = preg_replace("/data:[^,]+,/i", '', $dataURL);
        // 残りのデータはbase64エンコードされているので、デコードする
        $canvas = base64_decode($canvas);

        // まだ文字列の状態なので、画像リソース化
        $image = imagecreatefromstring($canvas);

        // 拡張子求める
        $ext = '';
        list($other, $type) = explode("/", $mime);
        switch($type){
            case "png":
            case "x-png":
                $ext = 'png';
                break;
            case "jpeg":
            case "jpg":
            case "pjpeg":
                $ext = 'jpg';
                break;
            case "gif":
            case "pgif":
                $ext = 'gif';
                break;
            default:
                die();
        }

        // 保存ファイル名
        $tmpFileName = 'AjaxUploadImage_' . md5(microtime(true) . '_' . rand(000, 999)) . '.' . $ext;
        // 保存ファイルパス
        $tmpPath = $this->config('tmpDirectory') . $tmpFileName;

        // 画像を配置
        switch($ext){
            case "png":
                imagesavealpha($image, true);
                imagepng($image, $tmpPath);
                break;
            case "jpg":
                imagejpeg($image, $tmpPath);
                break;
            case "gif":
                imagegif($image, $tmpPath);
                break;
            default:
                die();
        }
        chmod($tmpPath, 0777);

        return [$tmpPath, $mime, $ext];
    }

}
