<?php
namespace Cbase\View\Helper;
use Cake\View\Helper;
use Cake\View\View;
use Cake\Core\Configure;
use Cake\ORM\TableRegistry;
use Cake\Utility\Inflector;
use Cake\Utility\Hash;

/**
 * PrivateHelper
 *
 * @author s_miki
 *
 * @property HtmlHelper $Html
 * @property FormHelper $Form
 * @property PaginatorHelper $Paginator
 * @property AjaxHelper $Ajax
 * @property MinifyHelper $Minify
 */
class PrivateHelper extends Helper
{

    public $helpers = ['Html', 'Url', 'Form', 'Paginator', 'Ajax', 'Cbase.Minify'];

    public $stackMaster = [];

    ////////////////////////////////////////////////////////////////////////////
    // create form hidden //////////////////////////////////////////////////////
    ////////////////////////////////////////////////////////////////////////////

    /**
     * 全フォーム値のinput hidden タグを生成
     *
     * @param array|null $excepts 除外する配列※複数あれば配列で指定
     *  例) "User", "User.0"等
     * @return string HtmlString
     */
    public function hiddenVars($excepts = null)
    {
        $ret = "\n";
        if (!is_array($excepts)) $excepts[] = $excepts;

        if (!empty($this->request->data)) {
            $hiddenArr = $this->request->data;
            // 除外分
            foreach ($excepts as $except) {
                $hiddenArr = Hash::remove($hiddenArr, $except);
            }

            foreach ($hiddenArr as $key => $val) {
                $ret .= $this->_hiddenVarsRecursive($val, $key);
            }
        }

        return $ret;
    }

    /**
     * hiddenVarsの再帰処理
     *
     * @param array $data
     * @param string $parentKey
     * @return string
     */
    public function _hiddenVarsRecursive($data, $parentKey)
    {
        $ret = '';
        if (is_array($data)) {
            foreach ($data as $key => $val) {
                if(is_array($val)){
                    $ret .= $this->_hiddenVarsRecursive($val, "{$parentKey}.{$key}");
                } else {
                    $ret .= $this->Form->hidden("{$parentKey}.{$key}") . "\n";
                }
            }
        } else {
            $ret .= $this->Form->hidden("{$parentKey}") . "\n";
        }
        return $ret;
    }

    ////////////////////////////////////////////////////////////////////////////
    // get Master Data /////////////////////////////////////////////////////////
    ////////////////////////////////////////////////////////////////////////////

    /**
     * マスターテーブルのデータリストを取得
     *
     * @param string $model      取得するモデル名 頭大文字
     * @param string|null $field      取得カラム       省略でdisplayFieldを取得
     * @param array $conditions 取得条件         省略でpublic=1,deletedがnullの条件
     * @param array $order      取得順序         省略で"sequence IS NULL ASC, sequence DESC, id ASC"
     * @return array masterDataArray
     */
    public function getMasters($model, $field = null, $conditions = [], $order = [])
    {
        $key = serialize(func_get_args());

        // 同じ条件のデータがなければ取得
        if (!isset($this->stackMaster[$model][$key])) {

            // モデル読み込み
            $Model = TableRegistry::get($model);
            $Model->recursive = -1;

            // 取得条件を初期化
            //  ['public' => 1, 'deleted' => null]
            if (empty($conditions)) {
                $public = $Model->schema()->column('public');
                $deleted = $Model->schema()->column('deleted');
                if (!empty($public)) $conditions['public'] = 1;
                if (!empty($deleted)) $conditions['deleted IS'] = null;
            }

            // 取得順序
            if (empty($order)) {
                // sequenceがない場合はidにする
                $sequence = $Model->schema()->column('sequence');
                if (empty($sequence)) {
                    $order = ['id ASC'];
                } else {
                    $order = ['sequence IS NULL ASC, sequence DESC, id ASC'];
                }
            }

            // フィールド指定検索
            if ($field) {
                $Model->displayField($field);
                $this->stackMaster[$model][$key] = $Model->find('list', compact('conditions', 'order'))->toArray();
            } else {
                $this->stackMaster[$model][$key] = $Model->find('list', compact('conditions', 'order'))->toArray();
            }

        }
        return $this->stackMaster[$model][$key];
    }

    /**
     * マスターテーブルデータリストの指定したIDの値を取得
     *
     * @param string $model      取得するモデル名 ※頭大文字
     * @param int|string|null $id         指定ID
     * @param string|null $field      取得カラム
     * @param array $conditions 取得条件
     * @param array $order      取得順序
     * @param boolean $null       値がない時に
     * @return mixed|null masterDataArray
     */
    public function getMaster($model, $id = null, $field = null, $conditions = [], $order = [], $null = true)
    {
        $ret = $this->getMasters($model, $field, $conditions, $order);
        if (!empty($ret[$id])) {
            $ret = $ret[$id];
        } else {
            if ($null) $ret = null;
        }
        return $ret;
    }

    /**
     * config/values.phpデータリストの指定した配列を取得
     *
     * @param string $name      取得する配列名
     * @return mixed valueDataArray
     */
    public function getValues($name)
    {
        return Configure::read($name);
    }

    /**
     * config/values.phpデータリストの指定した配列の指定したIDの値を取得
     *
     * @param string $name      取得する配列名
     * @param int|string $id        取得する配列内のキー
     * @return mixed valueDataArray
     */
    public function getValue($name, $id)
    {
        return Configure::read("{$name}.{$id}");
    }

    ////////////////////////////////////////////////////////////////////////////
    // file ////////////////////////////////////////////////////////////////////
    ////////////////////////////////////////////////////////////////////////////

    /**
     * getImage.phpを利用した画像の表示
     * @param string $filename ファイル名
     * @param array $size array(width, height)
     * @param array $options HtmlHelper->imageのオプション
     *      ただしsrcOnly = trueにするとimageタグを使わずパスのみを返す
     *          alpha = trueにするとpngは透過
     * @param boolean $temp files/temp内の画像か
     * @return string
     */
    public function getImage($filename = null, $size = [160, 120], $options = [], $temp = false)
    {
        $defaults = [
            'srcOnly' => false,
            'alpha' => false,
            'resizeCrop' => false,
        ];
        $options = array_merge($defaults, $options);
        // 同名の変数に入れ、optionsから消す
        $srcOnly = $alpha = $rewrite = null;
        foreach (['srcOnly', 'alpha', 'resizeCrop'] as $key) {
            $$key = $options[$key];
            unset($options[$key]);
        }
        $width = !empty($size[0]) ? $size[0] : null;
        $height = !empty($size[1]) ? $size[1] : null;
        if (empty($width) && empty($height)) {
            $width = 160;
            $height = 120;
        }

        // fileパス
        if (!$path = $this->getFilePath($filename, $temp, true)) return null;

        // リサイズ後のパスを取得
        App::import('Vendor', 'Cbase.GetImage');
        $GetImage = new GetImage();
        $newPath = $GetImage->getName(array(
            'src' => $path,
            'width' => !empty($width) ? $width : null,
            'height' => !empty($height) ? $height : null,
            'alpha' => !empty($alpha) ? $alpha : null,
            'resizeCrop' => !empty($resizeCrop) ? $resizeCrop : null,
        ));

        return $srcOnly ? $newPath : $this->Html->image("/{$newPath}", $options);
    }

    /**
     * あるモデルとhasManyで結びついたモデルの1つ目のデータの画像を取得
     * @param array $params 配列 :モデル、子モデルデータまるごと入っている想定, 配列でない: idの想定
     * @param array $target array('親モデル名', '子モデル名', 'field名')を指定
     * @param array $size
     * @param array $options
     *      srcOnly: trueでパスのみ返す
     *      noimage: 画像パスを指定するとその画像を使用、false及び空指定だとnoimage時にfalseをreturn
     *      noimageSize: array(w,h) noimageの画像のサイズ
     * @param boolean $temp
     * @return string
     */
    public function getChildTopImage($params, $target, $size = [160, 120], $options = [], $temp)
    {

        if (!isset($options['srcOnly'])) $options['srcOnly'] = false;

        $noImg = '';
        if ($options['srcOnly']) {
            $noImg = (!empty($options['noimage'])) ? $options['noimage'] : '/cbase/img/admin/noimage.jpg';
        } else {
            // noimg画像は正方形を用意しているので$sizeの縦横の小さい方をはめる
            $noimageSize = [];
            if (empty($options['noimageSize'])) {
                $smaller = ($size[0] < $size[1]) ? $size[0] : $size[1];
                $noimageSize = ['width' => $smaller, 'height' => $smaller];
            } else {
                // 指定時
                if(!empty($options['noimageSize'][0])) $noimageSize['width'] = $options['noimageSize'][0];
                if(!empty($options['noimageSize'][1])) $noimageSize['height'] = $options['noimageSize'][1];
            }
            $noImgPath = (!empty($options['noimage'])) ? $options['noimage'] : '/cbase/img/admin/noimage.jpg';
            $noImg = $this->Html->image($noImgPath, $noimageSize + $options);
        }

        $details = null;
        $alt = '';
        if (is_array($params)) {
            // データない場合noimage
            if (empty($params[$target[1]])) {
                if (isset($options['noimage']) && !$options['noimage']) return false;
                return $noImg;
            }
            $details = $params[$target[1]];

            // alt
            if (!empty($params[$target[0]]['title'])) $alt = $params[$target[0]]['title'];
            elseif (!empty($params[$target[0]]['name'])) $alt = $params[$target[0]]['name'];
        } else {
            App::import('Model', $target[0]);
            /** @var Model $Model */
            $Model = new $target[0]();

            $conditions = ["{$target[0]}.id" => $params];
            $datas = $Model->find('first', compact('conditions'));

            // データない場合noimage
            if (empty($datas[$target[1]])) {
                if (isset($options['noimage']) && !$options['noimage']) return false;
                return $noImg;
            }
            $details = $datas[$target[1]];

            // alt
            if (!empty($datas[$target[0]]['title'])) $alt = $datas[$target[0]]['title'];
            elseif (!empty($datas[$target[0]]['name'])) $alt = $datas[$target[0]]['name'];
        }

        // altは親モデルのtitleかname
        $options['alt'] = $alt;

        // 最初のimageを取得
        $image = '';
        foreach ($details as $detail) {
            if (!empty($detail[$target[2]])) {
                $image = $detail[$target[2]];
                break;
            }
        }
        // データない場合noimage
        if (empty($image)) {
            if(isset($options['noimage']) && !$options['noimage']) return false;
            return $noImg;
        }

        return $this->getImage($image, $size, $options, $temp);
    }

    /**
     * アップロードしたファイルのURLを返す
     * @param string $filename ファイル名
     * @param boolean $temp files/temp内のファイルか
     * @param boolean $getimagephp trueなら./から始まるパスを返す
     * @return string
     */
    public function getFilePath($filename = null, $temp = false, $getimagephp = false)
    {
        if (empty($filename)) return null;
        // full urlならそのまま返す
        if (preg_match("/^https?:\/\//", $filename)) return $filename;

        // 最初の/
        $topSlash = ($getimagephp) ? './' : '/';

        // fileパス
        if ($temp) {
            // temp
            return $topSlash . TMP_DIR . $filename;
        } else {
            // 既にアップされているもの
            // ディレクトリを求める
            $model = $id = $other = null;
            @list($model, $id, $other) = explode('_', $filename);
            if (empty($model) || empty($id) || empty($other)) {
                return ".{$filename}"; // kcfinder等
            }
            $idPath = floor($id / 1000);
            return $topSlash . REAL_DIR . "{$model}/{$idPath}/{$filename}";
        }
    }

    /**
     * アップロードしたファイルの容量を調べる
     *
     * @param string $filename
     * @return string
     */
    public function getFileSize($filename)
    {
        $path = WWW_ROOT . $this->getFilePath($filename);
        $size = floor(filesize($path) / 1024);
        return "{$size}kB";
    }

    ////////////////////////////////////////////////////////////////////////////
    // util ////////////////////////////////////////////////////////////////////
    ////////////////////////////////////////////////////////////////////////////

    /**
     * フォーマットされた日付データを返す
     *
     * @param string $format
     * @param string|object $value
     * @return string
     */
    public function getFormatDate($format = 'Y-m-d', $value = null)
    {
        if (is_object($value)) {
            if (get_class($value) === 'Cake\I18n\FrozenDate' || get_class($value) === 'Cake\I18n\FrozenTime') {
                return $value->format($format);
            } else {
                return false;
            }
        }
        return date($format, strtotime($value));
    }

    /**
     * セレクトボックス用の配列を生成 ※$key => int
     *
     * @param int $start     開始
     * @param int $end       終了
     * @param int $interval  間隔
     * @param string $desinence 末尾に付加する文字列
     * @param boolean $zeroPad   先頭に0を不可するかどうか
     * @param boolean $wareki    和暦データを返すかどうか
     * @return array
     */
    public function getSelectNum($start, $end, $interval = 1, $desinence = null, $zeroPad = false, $wareki_flg = false)
    {
        $ret = [];
        for ($key = $start; $key <= $end; $key = $key + $interval) {
            if ($zeroPad) $key = sprintf("%02d", $key);
            $wareki = '';
            if ($wareki_flg) {
                $wareki = ' (' . $this->getWareki($key) . ')';
            }
            $ret[$key] = $key . $wareki . $desinence;
        }

        return $ret;
    }

    /**
     * 西暦から和暦データを返す
     *
     * @param int $year
     * @return string
     */
    public function getWareki($year = null)
    {
        $ret = '';

        $meiji = $year - 1867;
        $taisyo = $year - 1911;
        $syowa = $year - 1925;
        $heisei = $year - 1988;

        if ($meiji == 1)
            $meiji = '元';
        if ($taisyo == 1)
            $taisyo = '元';
        if ($syowa == 1)
            $syowa = '元';
        if ($heisei == 1)
            $heisei = '元';

        if ($year < 1912)
            $ret = 'M' . $meiji;
        elseif ($year == 1912)
            $ret = 'M' . $meiji . '/T' . $taisyo;
        elseif ($year < 1926)
            $ret = 'T' . $taisyo;
        elseif ($year == 1926)
            $ret = 'T' . $taisyo . '/S' . $syowa;
        elseif ($year < 1989)
            $ret = 'S' . $syowa;
        elseif ($year == 1989)
            $ret = 'S' . $syowa . '/H' . $heisei;
        else
            $ret = 'H' . $heisei;

        return $ret;
    }

    /**
     * $date時の年齢を取得
     * 省略時は現在
     *
     * @param string $birth Y-m-d形式の日付 or object
     * @param string $date
     * @return int
     */
    public function getAge($birth = null, $date = false)
    {
        if (!$birth) return false;
        if (!$date) $date = date('Y-m-d');
        $birth = $this->getFormatDate('Y-m-d', $birth);

        $ty = date('Y', strtotime($date));
        $tm = date('m', strtotime($date));
        $td = date('d', strtotime($date));

        list($by, $bm, $bd) = explode('-', $birth);
        $age = $ty - $by;
        if ($tm * 100 + $td < $bm * 100 + $bd) $age--;

        return $age;
    }

    /**
     * マルチバイト文字列用文字省略
     * @param string $text      テキスト
     * @param int $length   何文字で切るか
     * @param string $ending    文字列の語尾に付加する文字列
     * @param boolean $exact
     * @return string
     */
    public function mbTruncate($text, $length, $ending = '...', $exact = true)
    {
        if (strlen($text) <= $length) return $text;
        else {
            mb_internal_encoding("UTF-8");
            if (mb_strlen($text) > $length) {
                $length -= mb_strlen($ending);
                if (!$exact) {
                    $text = preg_replace('/\s+?(\S+)?$/', '', mb_substr($text, 0, $length + 1));
                }
                return mb_substr($text, 0, $length) . $ending;
            } else {
                return $text;
            }
        }
    }

    ////////////////////////////////////////////////////////////////////////////
    // for CMS /////////////////////////////////////////////////////////////////
    ////////////////////////////////////////////////////////////////////////////

    /**
     * 更新者名を返す(created_id,modified_id,deleted_id用)
     */
    public function inputUser($id, $model = 'AdminUsers', $nameField = null, $conditions = [])
    {
        $Model = TableRegistry::get($model);
        $Model->recursive = -1;
        if (empty($conditions)) {
            if ($Model->schema()->column('public')) {
                $conditions['public'] = 1;
            }
            if ($Model->schema()->column('deleted')) {
                $conditions['deleted IS'] = null;
            }
        }
        $conditions['id'] = $id;
        $result = $Model->find('all', compact('conditions'))->first();

        $name = '-';
        if (!empty($nameField)) $name = $result->{$nameField};
        elseif (!empty($result->name)) $name = $result->name;
        elseif (!empty($result->title)) $name = $result->title;
        elseif (!empty($result->disp_name)) $result->disp_name;

        return $name;
    }

    /**
     * ページングを生成する
     * @param array $labels array(前のページのリンク文言, 次のページのリンク文言)
     * @param array $options 以下の内容キーを受け付ける
     *      div         :       大外のdivを記述するか(true or false)
     *      divClass    :       大外のdivのclass
     *      prevClass   :       前のページのclass
     *      nextClass   :       次のページのclass
     *      numberClass :       ページ番号部分のclass
     *      separator   :       セパレータ
     * @return string
     */
    public function paginator($labels = array('<', '>', '<<', '>>'), $options = [])
    {
        // backは除き本来のパラメータを埋める
        $params = [];
        // $params = $this->request->params['named'];
        // if (!empty($params['back'])) unset($params['back']);
        // $this->Paginator->options(array('url' => $params));

        // // 該当0件の場合やページングが存在しない場合は何もなし
        // if($this->Paginator->counter(array('format' => '%count%')) == '0' || (!$this->Paginator->hasPrev() && !$this->Paginator->hasNext())){
        //     return null;
        // }

        // defaultオプション
        $defaults = [
            'div'           => true,
            'divClass'      => 'pagination',
            'prevClass'     => 'prev',
            'nextClass'     => 'next',
            'numberClass'   => 'page-numbers',
            'separator'     => null,
        ];
        // マージ
        $options = array_merge($defaults, $options);

        $ret  = '';
        if ($options['div']) $ret .= '<div class="'.$options['divClass'].'">' . PHP_EOL;
        if ($this->Paginator->hasPrev()) {
            $ret .= $this->Paginator->first($labels[2], ['separator' => $options['separator'], 'class' => $options['prevClass'], 'tag' => 'li'], null, ['tag' => 'li','class' => 'disabled','disabledTag' => 'a']) . PHP_EOL;
            $ret .= $this->Paginator->prev($labels[0], ['separator' => $options['separator'], 'class' => $options['prevClass'], 'tag' => 'li'], null, ['tag' => 'li','class' => 'disabled','disabledTag' => 'a']) . PHP_EOL;
        }
        $ret .= $this->Paginator->numbers(['separator' => $options['separator'], 'class' => $options['numberClass'], 'tag' => 'li', 'currentTag' => 'a', 'currentClass' => 'active']) . PHP_EOL;
        if ($this->Paginator->hasNext()) {
            $ret .= $this->Paginator->next($labels[1], ['separator' => $options['separator'], 'class' => $options['nextClass'], 'tag' => 'li'], null, ['tag' => 'li','class' => 'disabled','disabledTag' => 'a']) . PHP_EOL;
            $ret .= $this->Paginator->last($labels[3], ['separator' => $options['separator'], 'class' => $options['nextClass'], 'tag' => 'li'], null, ['tag' => 'li','class' => 'disabled','disabledTag' => 'a']) . PHP_EOL;
        }
        if ($options['div']) $ret .= '</div>' . PHP_EOL;

        return $ret;
    }

    /**
     * 公開非公開のAjax切り替えプルダウンの表示(管理側用)
     * @param array $modeldata
     * @param array $options
     *  $prefix = 'admin', $list = array(), $callback = ''
     * @return string
     */
    public function selectPublic($modeldata, $options = [])
    {
        $defaults = [
            'url' => $this->Url->build(['controller' => 'Cbase', 'action' => 'changePublic', 'plugin' => 'Cbase']),
            'list' => Configure::read('MasterPublic'),
            'modelName' => Inflector::pluralize($this->request->controller),
            'callback' => '',
            'field' => 'public',
        ];
        $options = array_merge($defaults, $options);

        $ret  = '';
        $ret .= $this->Form->create(false, ['url' => $options['url']]);
        $ret .= $this->Form->select("public_flag", $options['list'], ['default' => $modeldata[$options['field']], 'id' => "publicFlag{$modeldata['id']}", 'empty' => false, 'class' => 'form-control', 'name' => 'public']);
        $ret .= $this->Form->hidden('id', ['value' => $modeldata['id'], 'name' => 'id']);
        $ret .= $this->Form->hidden('model', ['value' => $options['modelName'], 'name' => 'model']);
        $ret .= $this->Form->hidden('field', ['value' => $options['field'], 'name' => 'field']);
        $ret .= $this->Form->end();
        $ret .= "<div id='publicMsg{$modeldata['id']}'></div>";
        $ret .= $this->Html->scriptBlock($this->Minify->scriptBlock("
            jQuery(function($){
                // ロード時に色変える
                var public = '{$modeldata[$options['field']]}';
                var publicColor = '';
                var notPublicColor = '#E3E3E3';
                $('#publicFlag{$modeldata['id']}').closest('tr').children('td').css('background-color', (public == '1' ? publicColor : notPublicColor));

                // イベント
                $('#publicFlag{$modeldata['id']}').change(function(e){
                    $('#publicMsg{$modeldata['id']}').text('変更中…');
                    var keyValues = {};
                    $('#publicFlag{$modeldata['id']}').closest('form').find(':input').each(function(){
                        keyValues[$(this).attr('name')] = $(this).val();
                    });
                    $.ajax({
                        type: 'POST',
                        url: '{$options['url']}',
                        data: keyValues,
                        success: function(html){
                            html = html.replace(/(^\s+)|(\s+$)/g, '');
                            if(html.length > 0){
                                $('#publicMsg{$modeldata['id']}').text('変更失敗！');
                            }else{
                                // 色の変更
                                $(e.target).closest('tr').children('td').css('background-color', ($(e.target).val() == '1' ? publicColor : notPublicColor));

                                $('#publicMsg{$modeldata['id']}').text('');

                                // callback
                                {$options['callback']}
                            }
                        },
                        error: function(){
                            $('#publicMsg{$modeldata['id']}').text('変更失敗！');
                        }
                    });
                });
            });
        "), ['block' => true]);

        return $ret;
    }

    /**
     * 日時期間をテキストで返す
     *
     * @param string $start
     * @param string $end
     * @param array $options
     * $optionsに指定可能なもの
     *      'type'      : 'br'を指定すると<br/>改行した文字列になる
     *      'format'    : 日時のフォーマット, 指定なければY/n/jまたはY/n/j H:i:s
     *      'separator' : 開始と終了の間に入れる文字列, デフォルト'〜'
     *      'empty'     : 開始も終了も存在しない場合の文字列, デフォルト'常時'
     * @return string
     */
    public function getTermText($start, $end, $options = [])
    {
        // フォーマットのチェック
        $defaultFormat = '';
        if (empty($options['format'])) {
            $date = $start;
            if (empty($date)) $date = $end;
            if (!empty($date)) {
                if (preg_match('/^([1-9][0-9]{3})-(0[1-9]{1}|1[0-2]{1})-(0[1-9]{1}|[1-2]{1}[0-9]{1}|3[0-1]{1})$/', $date)) {
                    // $start, $endどちらかが日付
                    $defaultFormat = 'Y/n/j';
                } else {
                    // $start, $endどちらかが日付でない=日時
                    $defaultFormat = 'Y/n/j H:i:s';
                }
            }
        }
        // デフォルトのオプション
        $defaults = [
            'type' => '',
            'format' => $defaultFormat,
            'separator' => '〜',
            'empty' => '常時',
        ];
        // merge
        $options = array_merge($defaults, $options);

        // どちらもデータ存在しない
        if (empty($start) && empty($end)) return $options['empty'];
        // 文字列作る
        $startStr = $this->getFormatDate($options['format'], $start);
        $endStr = $this->getFormatDate($options['format'], $end);
        if ($startStr == $endStr) {
            // start,end同じ場合
            return $startStr;
        }
        // typeによって異なる形式
        $typeFormat = '%s %s %s';
        if ($options['type'] == 'br') $typeFormat = '%s<br />%s<br />%s';
        // return
        return sprintf($typeFormat, $startStr, $options['separator'], $endStr);
    }

    /**
     * 管理画面一覧の"編集""コピー""プレビュー""削除"のパネルを返す
     * @param int|string $id 操作するID
     * @param mixed $displayButtons 表示するボタン
     *      'show', 'preview', 'edit', 'copy', 'delete'が指定可能
     *      配列順に表示される
     * @param array $urlOptions urlのpluginやcontrollerを書き換えるため
     * @return string
     */
    public function getOperationPanel($id, $displayButtons = ['show', 'preview', 'edit', 'copy', 'delete'], array $urlOptions = [])
    {
        // controllerで指定したprivateSettings['useConfirm']
        $privateSettings = $this->_View->get('privateSettings');
        $useConfirm = !empty($privateSettings['useConfirm']);
        // 削除時のconfirm js
        $delJs = '';
        if (!$useConfirm) {
            $delButtonId = "delBtn_" . uniqid();
            $delJs = $this->Html->scriptBlock($this->Minify->scriptBlock("
                jQuery(function($){
                    // 削除ボタンのconfirm
                    $('#{$delButtonId}').bind('click', function(){
                        return window.confirm('削除してもいいですか?');
                    });
                });
            "), ['inline' => false]);
        }

        $ret = '';
        foreach ($displayButtons as $type) {
            if ($type == 'show') $ret .= $this->Html->link('詳細', array_merge($urlOptions, ['action' => 'show', $id]), ['class' => 'btn btn-default']);
            elseif ($type == 'preview') $ret .= $this->Html->link('プレビュー', array_merge($urlOptions, ['action' => 'preview', $id]), ['class' => 'btn btn-default', 'target' => '_blank']);
            elseif ($type == 'edit') $ret .= $this->Html->link('編集', array_merge($urlOptions, ['action' => 'edit', $id]), ['class' => 'btn btn-primary']);
            elseif ($type == 'copy') $ret .= $this->Html->link('コピー', array_merge($urlOptions, ['action' => 'copy', $id]), ['class' => 'btn btn-primary']);
            elseif ($type == 'delete') {
                if (!$useConfirm) {
                    $ret .= $this->Html->link('削除', array_merge($urlOptions, ['action' => 'deleteComplete', $id]), ['class' => "btn btn-danger", 'id' => $delButtonId]);
                } else {
                    $ret .= $this->Html->link('削除', array_merge($urlOptions, ['action' => 'deleteConfirm', $id]), ['class' => "btn btn-danger"]);
                }
            }
        }
        return "<div class='btn-group'>{$ret}</div>{$delJs}";
    }

    /**
     * 並び順変更(input type text)を表示
     */
    public function getSequenceInput($id, $value, array $options = [])
    {
        $defaults = [
            'url' => $this->Url->build(['controller' => 'Cbase', 'action' => 'changeSequence', 'plugin' => 'Cbase']),
            'modelName' => Inflector::pluralize($this->request->controller),
            'field' => 'sequence',
            'redirectUrl' => null
        ];
        $options = array_merge($defaults, $options);
        $formId = 'changeSequence_' . uniqid();

        $ret = '';
        $ret .= $this->Form->create(false, ['url' => $options['url'], 'id' => $formId]);
        $ret .= '<div class="input-group">';
        $ret .= $this->Form->text('value', ['value' => $value, 'class' => 'form-control', 'size' => 10]);
        $ret .= $this->Form->hidden('model', ['value' => $options['modelName']]);
        $ret .= $this->Form->hidden('id', ['value' => $id]);
        $ret .= $this->Form->hidden('field', ['value' => $options['field']]);
        $ret .= $this->Form->hidden('redirect_url', ['value' => $options['redirectUrl']]);
        $ret .= '<span class="input-group-btn">';
        $ret .= $this->Html->link('変更', 'javascript:void(0);', ['class' => 'btn btn-primary', 'onclick' => "jQuery('#{$formId}').submit();"]);
        $ret .= '</span>';
        $ret .= '</div>';
        $ret .= $this->Form->end();

        return $ret;
    }


}
