<?php
namespace Cbase\Model\Table;

use ArrayObject;
use Cake\Datasource\EntityInterface;
use Cake\Event\Event;
use Cake\Network\Request;
use Cake\ORM\Table;
use Cake\Validation\Validator;
use Cbase\Model\Validation\CbaseValidator;

/**
 * Cbase Table
 * base table class
 */
class CbaseTable extends Table
{

    /**
     * Initialize method
     * @param array $config
     */
    public function initialize(array $config)
    {
        if ($this->schema()->column('title')) {
            $this->displayField('title');
        } elseif ($this->schema()->column('name')) {
            $this->displayField('name');
        }
        $this->primaryKey('id');
        $this->addBehavior('Timestamp');
    }

    /**
     * validator
     * @param Validator $validator
     * @return CbaseValidator
     */
    public function validationDefault(Validator $validator)
    {
        // Custom validatorを利用する
        return new CbaseValidator();
    }

    /**
     * beforeSave
     * @param Event $event
     * @param EntityInterface $entity
     * @param ArrayObject $options
     * @return bool
     */
    public function beforeSave(Event $event, EntityInterface $entity, ArrayObject $options)
    {
        // created_id, modified_idカラムが存在する場合ログイン者IDを入れる
        if($this->schema()->column('created_id') || $this->schema()->column('modified_id')){
            // 登録者・更新者IDを付加
            $Request = Request::createFromGlobals();
            if($Request->session()->check('Auth.AdminUser.id')){
                if ($this->schema()->column('created_id') && empty($entity->id)) $entity->created_id = $Request->session()->read('Auth.AdminUser.id');
                // $entity->modifiedを更新する場合はこの段階で飛んでくる
                if ($this->schema()->column('modified_id') && !empty($entity->modified)) $entity->modified_id = $Request->session()->read('Auth.AdminUser.id');
            }
        }

        return true;
    }

    /**
     * paginate用に検索条件を取得
     * @param array $params
     * @param bool $admin
     * @return array
     */
    public function getOptions(array $params = [], $admin = false)
    {
        $conditions = [];

        // deleted
        if ($this->schema()->column('deleted')) {
            $conditions += ["{$this->alias()}.deleted IS" => null];
        }

        // public
        if ($this->schema()->column('public')) {
            if ($admin) {
                if (!empty($params['public'])) {
                    $conditions += ["{$this->alias()}.public" => h($params['public'])];
                }
            } else {
                $conditions += ["{$this->alias()}.public" => 1];
            }
        }

        // keywords
        if (!empty($params['keywords'])) {
            $keywordsConditions = $this->createKeywordsConditions($params['keywords'], $admin);
            if (!empty($keywordsConditions)) {
                if (empty($conditions['AND'])) {
                    $conditions['AND'] = [];
                }
                $conditions['AND'][] = $keywordsConditions;
            }
        }

        // order
        $order = (!empty($params['order'])) ? $params['order'] : null;

        $ret = ['conditions' => $conditions];
        if (!empty($params['limit'])) {
            $ret['limit'] = $params['limit'];
        }
        if (!empty($order)) {
            $ret['order'] = $order;
        }
        return $ret;
    }

    /**
     * キーワードの検索conditions配列を作る
     */
    public function createKeywordsConditions($paramKeywords, $admin = false)
    {
        // スペース区切りのキーワード検索対応
        $keyword = mb_convert_kana($paramKeywords, 's');
        $keywords = preg_split('/[\s]+/', $keyword, -1, PREG_SPLIT_NO_EMPTY);
        $item = [];
        $targetFields = $this->_getKeywordsTargets();
        if (empty($targetFields[0]) && empty($targetFields[1])) {
            return [];
        }
        foreach ($keywords as $kwd) {
            if (empty($kwd)) continue;
            $or = [];
            if (!empty($targetFields[0])) {
                foreach ($targetFields[0] as $field) {
                    $f = (strpos($field,'.') === false) ? "{$this->alias()}.{$field}" : $field;
                    $or[] = ["({$f}) COLLATE utf8_unicode_ci LIKE" => '%' . $kwd . '%'];
                }
            }
            if ($admin && !empty($targetFields[1])) {
                foreach ($targetFields[1] as $field) {
                    $f = (strpos($field,'.') === false) ? "{$this->alias()}.{$field}" : $field;
                    $or[] = ["({$f}) COLLATE utf8_unicode_ci LIKE" => '%' . $kwd . '%'];
                }
            }
            $item[] = ['OR' => $or];
        }
        return ['OR' => $item];
    }

    /**
     * キーワード検索の対象となるフィールドを設定
     * overrideして使う
     * @return array [['通常時kwd1', '通常時kwd2'...], ['admin時kwd1', 'admin時kwd2'...]]
     *      例) [['title', 'text'], ['note']]
     */
    protected function _getKeywordsTargets()
    {
        $regularTargets = [];
        if ($this->schema()->column('title')) {
            $regularTargets[] = 'title';
        }
        if ($this->schema()->column('text')) {
            $regularTargets[] = 'text';
        }

        $adminTargets = [];
        if ($this->schema()->column('note')) {
            $regularTargets[] = 'note';
        }
        return [$regularTargets, $adminTargets];
    }

    /**
     * 論理削除をする
     * @param $id
     * @return bool|EntityInterface|mixed
     */
    public function softDelete($id)
    {
        if (!$entity = $this->get($id)) return false;
        $entity->deleted = date('Y-m-d H:i:s');

        // deleted_idカラムが存在する場合ログイン者IDを入れる
        if($this->schema()->column('deleted_id')){
            $Request = Request::createFromGlobals();
            if($Request->session()->check('Auth.AdminUser.id')){
                $entity->deleted_id = $Request->session()->read('Auth.AdminUser.id');
            }
        }

        return $this->save($entity);
    }

    /**
     * 条件を指定した論理削除
     * @param $conditions
     * @return bool
     */
    public function softDeleteAll($conditions)
    {
        // 対象のidを取得
        $ids = $this->find('list', ['conditions' => $conditions, 'valueField' => 'id']);
        foreach($ids as $id){
            if(!$this->softDelete($id)) return false;
        }
        return true;
    }

    /**
     * 一覧を取得
     * @param bool $list
     * @param bool $admin
     * @return array
     */
    public function getPublic($list = false, $admin = false)
    {
        $options = $this->getOptions([], $admin);
        if ($list) {
            $this->displayField('title');
            return $this->find('list', $options)->toArray();
        } else {
            $datas = $this->find('all', $options)->toArray();
            if (empty($datas)) {
                return $datas;
            }
            // idをkeyにしておく
            $ret = [];
            foreach ($datas as $data) {
                $ret[$data->id] = $data;
            }
            return $ret;
        }
    }

    /**
     * optionsに含まれているfinderを分離してreturnする
     * @param $options
     * @return array
     */
    protected function _extractFinder($options)
    {
        $type = !empty($options['finder']) ? $options['finder'] : 'all';
        unset($options['finder'], $options['maxLimit']);

        if (is_array($type)) {
            $options = (array)current($type) + $options;
            $type = key($type);
        }

        return [$type, $options];
    }

}
