<?php
/**
 * PrefixBasic認証コンポネント
 * @author ueno
 *
 * 使い方
 * AppController.php
 *      componentの読み込み
 *          public $components = ['Auth', 'Cbase.PrefixBasicAuth'];
 *      before_filterの最初に以下を記述
 *          // Basic認証
 *          $this->PrefixBasicAuth->setup(
 *              ['allow' => ['api', 'apisp', '']], // Basic認証をかけるprefix (※)
 *              [['id', 'pass']] // ID,pass
 *          );
 * (※) Basic認証をかけるprefixの書き方
 *      例1) 全てかける               : ['*'] ...... *要素を配列にいれると全てかける
 *      例2) 全て認証外す             : [] or false ...... emptyだと認証がなにもかからない
 *      例3) adminのみかける          : ['deny' => ['admin']] ...... denyにadminを含める
 *      例4) 表側とapiだけ認証外す    : ['allow' => ['api', '']] ...... allowに表側(prefixなし'')とapiを含める
 *      例5) 表側とapiだけ認証外す    : ['allow' => ['api', '']] ...... allowに表側(prefixなし'')とapiを含める
 */
namespace Cbase\Controller\Component;

use Cake\Controller\Component;
use Cake\Event\Event;
use Cake\Core\Configure;

/**
 * PrefixBasicAuth Component
 */
class PrefixBasicAuthComponent extends Component
{

    /**
     * @var array 対象とするprefix
     */
    protected $prefixes = [];

    /**
     * @var array ID, PASSのペアを配列指定
     * [['id1', 'pass1'], ['id2', 'pass2'], ...]
     */
    protected $authPair = [];

    /**
     * @var array 認証をかけないaction名を配列で指定
     */
    protected $allow = [];

    /**
     * Basic認証かける条件を設定
     *
     * @param array $prefixes
     * @param array $authPair
     * @param array $allow
     */
    public function setup($prefixes = [], $authPair = [], $allow = [])
    {
        // prefixes
        $this->prefixes = $prefixes;
        // authPair
        $this->authPair = $authPair;
        // allow
        $this->allow = $allow;
    }

    /**
     * startup
     *
     * @param Controller $controller
     */
    public function startup(Event $event)
    {
        // Controller
        $controller = $this->_registry->getController();

        // prefixesない場合なにもしない
        if (empty($this->prefixes)) return;

        // 該当のprefixでない場合スルー
        if (in_array('*', $this->prefixes)) {
            // 全てかける
        } elseif (!empty($this->prefixes['deny']) && in_array($controller->request->prefix, $this->prefixes['deny'])) {
            // 現在のprefixがdenyに含まれる場合、basicかける
        } elseif (!empty($this->prefixes['allow']) && !in_array($controller->request->prefix, $this->prefixes['allow'])) {
            // 現在のprefixがallowに含まれない場合、basicかける
        } else {
            // それ以外かけない
            return;
        }

        // 認証をかけないaction名の場合スルー
        if (in_array($controller->action, $this->allow)) {
            return;
        }

        // ID, PASSのペアがない、もしくは不完全な場合Basicかけない
        if (empty($this->authPair)) return;
        else {
            foreach ($this->authPair as $val) {
                if (count($val) != 2) return;
            }
        }

        $user = env('PHP_AUTH_USER');
        $pass = env('PHP_AUTH_PW');

        if (empty($user) || empty($pass)) {
            $this->_unauthorized($controller);
        } else {
            if (!$this->_auth($user, $pass)) {
                // 一旦forbiddenになるとブラウザ再起動するまで認証しなおせないためずっと出すように変更
                $this->_unauthorized($controller);
                //$this->_forbidden($controller);
            }
        }
    }

    /**
    * unauthorized
    *
    * @param Controller $controller
    */
    protected function _unauthorized($controller)
    {
        header("WWW-Authenticate: Basic realm=\"Please Enter Your Password\"");
        $controller->redirect(null, 401, false);
        echo "Authorization Required";
        exit;
    }

    /**
    * forbidden
    *
    * @param Controller $controller
    */
    protected function _forbidden($controller)
    {
        $controller->redirect(null, 403, false);
        echo "Authorization Required";
        exit;
    }

    /**
    * auth
    *
    * @param string $user
    * @param string $pass
    * @return boolean
    */
    protected function _auth($user, $pass)
    {
        $paramHash = sha1($user . $pass . Configure::read('Security.salt'));

        // 認証
        foreach ($this->authPair as $val) {
            $authPairHash = sha1($val[0] . $val[1] . Configure::read('Security.salt'));
            if ($paramHash === $authPairHash) return true;
        }
        return false;
    }
}
