<?php

namespace GMO\Services\CreditCard;

use GMO\Commons\ParamParser;
use GMO\Configs\GmoConfig;
use GMO\Inputs\SaveCardInput;
use GMO\Inputs\SearchCardDetailInput;
use GMO\Inputs\TradedCardInput;
use GMO\Outputs\SaveCardOutput;
use GMO\Outputs\TradedCardOutput;
use GMO\Trans\SaveCard;
use GMO\Trans\SearchCardDetail;
use GMO\Trans\TradedCard;

/**
 * Class SimpleAuthorizationService
 * @package GMO\Services\CreditCard
 */
class CardService
{
    /**
     * 決済後カード登録
     *
     * @param $orderId
     * @param $memberId
     * @param null $holderName
     * @param int $defaultFlag 0:継続課金対象としない(デフォルト) 1:継続課金対象
     * @param bool $disableDebit デビットカードの使用禁止
     * @return TradedCardOutput
     * @throws \GMO\Commons\GPayException
     * @throws \ReflectionException
     */
    public function traded($orderId, $memberId, $holderName = null, $defaultFlag = 0, $disableDebit = false)
    {
        // 決済後カード登録
        // 入力パラメータクラスをインスタンス化します
        $input = new TradedCardInput();

        // ショップID・サイトID・パスワードはコンフィグファイルから設定
        $input->setShopId( GmoConfig::getShopId() );
        $input->setShopPass( GmoConfig::getShopPassword() );
        $input->setSiteId( GmoConfig::getSiteId() );
        $input->setSitePass( GmoConfig::getSitePassword() );

        // 登録したい取引と、会員IDを指定します。
        $input->setMemberId( $memberId );
        $input->setOrderId( $orderId );
        if (isset($holderName)) $input->setHolderName($holderName);
        $input->setDefaultFlag($defaultFlag);

        // API通信クラスをインスタンス化します
        $exe = new TradedCard();

        // パラメータオブジェクトを引数に、実行メソッドを呼びます。
        // 正常に終了した場合、結果オブジェクトが返るはずです。
        $result = $exe->exec( $input );

        // エラーがない場合、デビットカードチェック
        if (!$result->isErrorOccurred() && $disableDebit) {
            $chkDebit = $this->checkDebit(null, $orderId);
            if (!is_null($chkDebit)) {
                $parser = new ParamParser();
                $errList = $parser->errParse($chkDebit[0], $chkDebit[1]);
                $result->setErrList($errList);
            }
        }

        return $result;
    }

    /**
     * 登録
     *
     * @param $memberId
     * @param $token
     * @param int $defaultFlag 0:継続課金対象としない(デフォルト) 1:継続課金対象
     * @param bool $disableDebit デビットカードの使用禁止
     * @return SaveCardOutput
     * @throws \GMO\Commons\GPayException
     * @throws \ReflectionException
     */
    public function save($memberId, $token, $defaultFlag = 0, $disableDebit = false)
    {
        // 会員登録の処理
        // 入力パラメータクラスをインスタンス化します
        $input = new SaveCardInput();

        // サイトID・パスワードはコンフィグファイルから設定
        $input->setSiteId( GmoConfig::getSiteId() );
        $input->setSitePass( GmoConfig::getSitePassword() );

        //会員IDは必須です
        $input->setMemberId( $memberId );

        // トークンでカード番号を設定します。
        $input->setToken( $token);

        $input->setDefaultFlag( $defaultFlag );

        //API通信クラスをインスタンス化します
        $exe = new SaveCard();

        //パラメータオブジェクトを引数に、実行メソッドを呼びます。
        //正常に終了した場合、結果オブジェクトが返るはずです。
        $result = $exe->exec( $input );

        // エラーがない場合、デビットカードチェック
        if (!$result->isErrorOccurred() && $disableDebit) {
            $chkDebit = $this->checkDebit($token, null);
            if (!is_null($chkDebit)) {
                $parser = new ParamParser();
                $errList = $parser->errParse($chkDebit[0], $chkDebit[1]);
                $result->setErrList($errList);
            }
        }

        return $result;
    }

    private function checkDebit($token = null, $orderId = null)
    {
        // 入力パラメータクラスをインスタンス化します
        $input = new SearchCardDetailInput();

        // ショップID・サイトID・パスワードはコンフィグファイルから設定
        $input->setShopId( GmoConfig::getShopId() );
        $input->setShopPass( GmoConfig::getShopPassword() );

        // 各種パラメータを設定しています。
        if (!is_null($token)) {
            $input->setToken($token);
        } else if (!is_null($orderId)) {
            $input->setOrderID($orderId);
        }

        // API通信クラスをインスタンス化します
        $exe = new SearchCardDetail();

        // パラメータオブジェクトを引数に、実行メソッドを呼び、結果を受け取ります。
        $output = $exe->exec( $input );

        //実行後、その結果を確認します。
        if($output->isErrorOccurred() ){
            // エラーが発生した場合、失敗
            $errorList = $output->getErrList();
            return [$errorList[0]->getErrCode(), $errorList[0]->getErrInfo()];
        }

        $cardList = $output->getCardList();
        if (count($cardList) != 1) {
            // 取得カードが一個ではない場合、失敗
            return ['E11', 'E11010099'];
        }

        $cardInfo = $cardList[0];
        if (isset($cardInfo['DebitPrepaidFlag']) || $cardInfo['DebitPrepaidFlag'] != 1) {
            return null;
        } else {
            return ['S00', 'S00000001'];
        }
    }
}
