<?php

namespace FileMaker\Services;

use FileMaker\Configs\FmConfig;
use FileMaker\Dto\FmResponseDto;

/**
 * FMへデーターの転送サービス
 *
 * Class Post2FmService
 */
abstract class Post2FmService
{
    /**
     * ログインユーザー項目名
     */
    const FIELD_NM_USER_ID = 'username';
    /**
     * ログインパスワード項目名
     */
    const FIELD_NM_USER_PASSWORD = 'password';
    /**
     * 商材区分項目名
     */
    const FIELD_NM_PRODUCT_KBN = 'product';

    /**
     * FMのAPI環境設定
     * @var FmConfig
     */
    protected $config;

    /**
     * Post2FmService constructor.
     * @param FmConfig $config
     */
    public function __construct(FmConfig $config)
    {
        $this->config = $config;
    }

    /**
     * APIへデーターを転送
     *
     * @param array $requestData
     * @return FmResponseDto
     */
    public function sendApi(array $requestData) {
        // 結果
        $res = new FmResponseDto();
        // 入力データーから必要な情報のみ格納
        $inputData = $this->convertForFm($requestData);
        $requireFields = $this->getRequireFields();
        // 必須チェック
        foreach ($requireFields as $requireField) {
            if (!array_key_exists($requireField, $inputData) || empty($inputData[$requireField])) {
                $res->errorList[] = "{$requireField}が存在しません。";
            }
        }
        // 変換作業
        foreach ($inputData as $key => $value) {
            // 日付変換
            if (in_array($key, $this->getYmdFields())) {
                $inputData[$key] = $this->convert2Ymd($value);
            }
        }

        $res->requestData = $inputData;
        // 入力チェックでエラーがある場合
        if ($res->isError()) {
            return $res;
        }

        // ログイン情報の格納
        $inputData[self::FIELD_NM_USER_ID] = $this->config->authUserId;
        $inputData[self::FIELD_NM_USER_PASSWORD] = $this->config->authPassword;
        $inputData[self::FIELD_NM_PRODUCT_KBN] = $this->config->productKbn;

        // FMへデーターの転送
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $this->config->apiUrl);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($inputData));
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_USERPWD, $this->config->basicUserId . ':' . $this->config->basicPassword);

        $result = curl_exec($ch); //通信に成功したら受注番号の返却「success,PTT126455」<-みたいな
        $info = curl_getinfo($ch);
        $errno = curl_errno($ch);
        $error = curl_error($ch);
        curl_close($ch);

        if (CURLE_OK !== $errno || strpos($result, 'error') !== false) {
            if (preg_match('/\{error:([\s\S.]*)\}/', $result, $errorMatches)) {
                $errorList = explode("\n", [$errorMatches[1]]);
                array_map('trim', $errorList);
                $errorList = array_filter($errorList, 'strlen');
                $errorList = array_values($errorList);
                $res->errorList = $errorList;
            } else {
                $res->errorList = [$result];
            }
        }

        return $res;
    }

    /**
     * 仕様上年月日フォーマットではダメなためデータを整形する
     *
     * @param string $dateStr
     * @return mixed|string
     */
    private function convert2Ymd($dateStr)
    {
        $date = $this->checkDatetimeFormat($dateStr);
        if ($date !== false) {
            return vsprintf('%s/%s/%s', [$date['year'],
                str_pad($date['month'], 2, 0, STR_PAD_LEFT),
                str_pad($date['day'], 2, 0, STR_PAD_LEFT)]);
        }

        return null;
    }

    /**
     * 日付の入力フォーマットをチェックし、値を渡す
     *
     * @param $datetime
     * @return bool | array
     */
    private function checkDatetimeFormat($datetime){
        if (preg_match('/^(?P<year>[0-9]{4})年(?P<month>[0-9]{1,2})月(?P<day>[0-9]{1,2})日$/', $datetime, $m) === 1
            && checkdate($m['month'] , $m['day'] , $m['year'])) {
            return $m;
        } else {
            return false;
        }
    }
}
